/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.streaming.OutgoingStream;
import org.apache.cassandra.streaming.PreviewKind;
import org.apache.cassandra.streaming.ProgressInfo;
import org.apache.cassandra.streaming.SessionInfo;
import org.apache.cassandra.streaming.StreamConnectionFactory;
import org.apache.cassandra.streaming.StreamEvent;
import org.apache.cassandra.streaming.StreamEventHandler;
import org.apache.cassandra.streaming.StreamOperation;
import org.apache.cassandra.streaming.StreamResultFuture;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.StreamState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamCoordinator {
    private static final Logger logger = LoggerFactory.getLogger(StreamCoordinator.class);
    private final boolean connectSequentially;
    private final Map<InetAddressAndPort, HostStreamingData> peerSessions = new ConcurrentHashMap<InetAddressAndPort, HostStreamingData>();
    private final StreamOperation streamOperation;
    private final int connectionsPerHost;
    private final boolean follower;
    private StreamConnectionFactory factory;
    private Iterator<StreamSession> sessionsToConnect = null;
    private final UUID pendingRepair;
    private final PreviewKind previewKind;

    public StreamCoordinator(StreamOperation streamOperation, int connectionsPerHost, StreamConnectionFactory factory, boolean follower, boolean connectSequentially, UUID pendingRepair, PreviewKind previewKind) {
        this.streamOperation = streamOperation;
        this.connectionsPerHost = connectionsPerHost;
        this.factory = factory;
        this.follower = follower;
        this.connectSequentially = connectSequentially;
        this.pendingRepair = pendingRepair;
        this.previewKind = previewKind;
    }

    public void setConnectionFactory(StreamConnectionFactory factory) {
        this.factory = factory;
    }

    public synchronized boolean hasActiveSessions() {
        for (HostStreamingData data : this.peerSessions.values()) {
            if (!data.hasActiveSessions()) continue;
            return true;
        }
        return false;
    }

    public synchronized Collection<StreamSession> getAllStreamSessions() {
        ArrayList<StreamSession> results = new ArrayList<StreamSession>();
        for (HostStreamingData data : this.peerSessions.values()) {
            results.addAll(data.getAllStreamSessions());
        }
        return results;
    }

    public boolean isFollower() {
        return this.follower;
    }

    public void connect(StreamResultFuture future) {
        if (this.connectSequentially) {
            this.connectSequentially(future);
        } else {
            this.connectAllStreamSessions();
        }
    }

    private void connectAllStreamSessions() {
        for (HostStreamingData data : this.peerSessions.values()) {
            data.connectAllStreamSessions();
        }
    }

    private void connectSequentially(StreamResultFuture future) {
        this.sessionsToConnect = this.getAllStreamSessions().iterator();
        future.addEventListener(new StreamEventHandler(){

            @Override
            public void handleStreamEvent(StreamEvent event) {
                if (event.eventType == StreamEvent.Type.STREAM_PREPARED || event.eventType == StreamEvent.Type.STREAM_COMPLETE) {
                    StreamCoordinator.this.connectNext();
                }
            }

            @Override
            public void onSuccess(StreamState result) {
            }

            @Override
            public void onFailure(Throwable t) {
            }
        });
        this.connectNext();
    }

    private void connectNext() {
        if (this.sessionsToConnect == null) {
            return;
        }
        if (this.sessionsToConnect.hasNext()) {
            StreamSession next = this.sessionsToConnect.next();
            if (logger.isDebugEnabled()) {
                logger.debug("Connecting next session {} with {}.", (Object)next.planId(), (Object)next.peer.toString());
            }
            this.startSession(next);
        } else {
            logger.debug("Finished connecting all sessions");
        }
    }

    public synchronized Set<InetAddressAndPort> getPeers() {
        return new HashSet<InetAddressAndPort>(this.peerSessions.keySet());
    }

    public synchronized StreamSession getOrCreateNextSession(InetAddressAndPort peer) {
        return this.getOrCreateHostData(peer).getOrCreateNextSession(peer);
    }

    public synchronized StreamSession getOrCreateSessionById(InetAddressAndPort peer, int id) {
        return this.getOrCreateHostData(peer).getOrCreateSessionById(peer, id);
    }

    public StreamSession getSessionById(InetAddressAndPort peer, int id) {
        return this.getHostData(peer).getSessionById(id);
    }

    public synchronized void updateProgress(ProgressInfo info) {
        this.getHostData(info.peer).updateProgress(info);
    }

    public synchronized void addSessionInfo(SessionInfo session) {
        HostStreamingData data = this.getOrCreateHostData(session.peer);
        data.addSessionInfo(session);
    }

    public synchronized Set<SessionInfo> getAllSessionInfo() {
        HashSet<SessionInfo> result = new HashSet<SessionInfo>();
        for (HostStreamingData data : this.peerSessions.values()) {
            result.addAll(data.getAllSessionInfo());
        }
        return result;
    }

    public synchronized void transferStreams(InetAddressAndPort to, Collection<OutgoingStream> streams) {
        HostStreamingData sessionList = this.getOrCreateHostData(to);
        if (this.connectionsPerHost > 1) {
            List<Collection<OutgoingStream>> buckets = this.bucketStreams(streams);
            for (Collection<OutgoingStream> bucket : buckets) {
                StreamSession session = sessionList.getOrCreateNextSession(to);
                session.addTransferStreams(bucket);
            }
        } else {
            StreamSession session = sessionList.getOrCreateNextSession(to);
            session.addTransferStreams(streams);
        }
    }

    private List<Collection<OutgoingStream>> bucketStreams(Collection<OutgoingStream> streams) {
        int targetSlices = Math.min(streams.size(), this.connectionsPerHost);
        int step = Math.round((float)streams.size() / (float)targetSlices);
        int index = 0;
        ArrayList<Collection<OutgoingStream>> result = new ArrayList<Collection<OutgoingStream>>();
        ArrayList<OutgoingStream> slice = null;
        for (OutgoingStream stream : streams) {
            if (index % step == 0) {
                slice = new ArrayList<OutgoingStream>();
                result.add(slice);
            }
            slice.add(stream);
            ++index;
        }
        return result;
    }

    private HostStreamingData getHostData(InetAddressAndPort peer) {
        HostStreamingData data = this.peerSessions.get(peer);
        if (data == null) {
            throw new IllegalArgumentException("Unknown peer requested: " + peer);
        }
        return data;
    }

    private HostStreamingData getOrCreateHostData(InetAddressAndPort peer) {
        HostStreamingData data = this.peerSessions.get(peer);
        if (data == null) {
            data = new HostStreamingData();
            this.peerSessions.put(peer, data);
        }
        return data;
    }

    public UUID getPendingRepair() {
        return this.pendingRepair;
    }

    private void startSession(StreamSession session) {
        session.start();
        logger.info("[Stream #{}, ID#{}] Beginning stream session with {}", new Object[]{session.planId(), session.sessionIndex(), session.peer});
    }

    private class HostStreamingData {
        private final Map<Integer, StreamSession> streamSessions = new HashMap<Integer, StreamSession>();
        private final Map<Integer, SessionInfo> sessionInfos = new HashMap<Integer, SessionInfo>();
        private int lastReturned = -1;

        private HostStreamingData() {
        }

        public boolean hasActiveSessions() {
            for (StreamSession session : this.streamSessions.values()) {
                if (session.state().isFinalState()) continue;
                return true;
            }
            return false;
        }

        public StreamSession getOrCreateNextSession(InetAddressAndPort peer) {
            if (this.streamSessions.size() < StreamCoordinator.this.connectionsPerHost) {
                StreamSession session = new StreamSession(StreamCoordinator.this.streamOperation, peer, StreamCoordinator.this.factory, StreamCoordinator.this.isFollower(), this.streamSessions.size(), StreamCoordinator.this.pendingRepair, StreamCoordinator.this.previewKind);
                this.streamSessions.put(++this.lastReturned, session);
                this.sessionInfos.put(this.lastReturned, session.getSessionInfo());
                return session;
            }
            if (this.lastReturned >= this.streamSessions.size() - 1) {
                this.lastReturned = 0;
            }
            return this.streamSessions.get(this.lastReturned++);
        }

        public void connectAllStreamSessions() {
            for (StreamSession session : this.streamSessions.values()) {
                StreamCoordinator.this.startSession(session);
            }
        }

        public Collection<StreamSession> getAllStreamSessions() {
            return Collections.unmodifiableCollection(this.streamSessions.values());
        }

        public StreamSession getOrCreateSessionById(InetAddressAndPort peer, int id) {
            StreamSession session = this.streamSessions.get(id);
            if (session == null) {
                session = new StreamSession(StreamCoordinator.this.streamOperation, peer, StreamCoordinator.this.factory, StreamCoordinator.this.isFollower(), id, StreamCoordinator.this.pendingRepair, StreamCoordinator.this.previewKind);
                this.streamSessions.put(id, session);
                this.sessionInfos.put(id, session.getSessionInfo());
            }
            return session;
        }

        public StreamSession getSessionById(int id) {
            return this.streamSessions.get(id);
        }

        public void updateProgress(ProgressInfo info) {
            this.sessionInfos.get(info.sessionIndex).updateProgress(info);
        }

        public void addSessionInfo(SessionInfo info) {
            this.sessionInfos.put(info.sessionIndex, info);
        }

        public Collection<SessionInfo> getAllSessionInfo() {
            return this.sessionInfos.values();
        }

        @VisibleForTesting
        public void shutdown() {
            this.streamSessions.values().forEach(ss -> ss.sessionFailed());
        }
    }
}

