/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.bulkwriter;

import com.google.common.base.Preconditions;
import com.google.common.collect.Range;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.cassandra.bridge.SSTableDescriptor;
import org.apache.cassandra.spark.bulkwriter.BulkWriteValidator;
import org.apache.cassandra.spark.bulkwriter.BulkWriterContext;
import org.apache.cassandra.spark.bulkwriter.CommitCoordinator;
import org.apache.cassandra.spark.bulkwriter.CommitResult;
import org.apache.cassandra.spark.bulkwriter.DirectDataTransferApi;
import org.apache.cassandra.spark.bulkwriter.DirectStreamResult;
import org.apache.cassandra.spark.bulkwriter.RingInstance;
import org.apache.cassandra.spark.bulkwriter.SortedSSTableWriter;
import org.apache.cassandra.spark.bulkwriter.StreamError;
import org.apache.cassandra.spark.bulkwriter.StreamResult;
import org.apache.cassandra.spark.bulkwriter.StreamSession;
import org.apache.cassandra.spark.bulkwriter.TransportContext;
import org.apache.cassandra.spark.bulkwriter.token.ReplicaAwareFailureHandler;
import org.apache.cassandra.spark.common.Digest;
import org.apache.cassandra.spark.common.SSTables;
import org.apache.cassandra.spark.data.FileType;
import org.apache.cassandra.util.IntWrapper;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectStreamSession
extends StreamSession<TransportContext.DirectDataBulkWriterContext> {
    private static final Logger LOGGER = LoggerFactory.getLogger(DirectStreamSession.class);
    private static final String WRITE_PHASE = "UploadAndCommit";
    private final AtomicInteger nextSSTableIdx = new AtomicInteger(1);
    private final DirectDataTransferApi directDataTransferApi;

    public DirectStreamSession(BulkWriterContext writerContext, SortedSSTableWriter sstableWriter, TransportContext.DirectDataBulkWriterContext transportContext, String sessionID, Range<BigInteger> tokenRange, ReplicaAwareFailureHandler<RingInstance> failureHandler, ExecutorService executorService) {
        super(writerContext, sstableWriter, transportContext, sessionID, tokenRange, failureHandler, executorService);
        this.directDataTransferApi = transportContext.dataTransferApi();
    }

    @Override
    protected void onSSTablesProduced(Set<SSTableDescriptor> sstables) {
        if (sstables.isEmpty() || this.isStreamFinalized()) {
            return;
        }
        this.executorService.submit(() -> {
            try {
                Map<Path, Digest> fileDigests = this.sstableWriter.prepareSStablesToSend(this.writerContext, sstables);
                IntWrapper sstableCounter = new IntWrapper();
                fileDigests.keySet().stream().filter(p -> p.getFileName().toString().endsWith(FileType.DATA.getFileSuffix())).forEach(sstable -> {
                    ++sstableCounter.value;
                    this.sendSStableToReplicas((Path)sstable);
                });
                LOGGER.info("[{}]: Sent newly produced SSTables. sstables={}", (Object)this.sessionID, (Object)sstableCounter.value);
                LOGGER.info("[{}]: Removing temporary files after streaming. files={}", (Object)this.sessionID, fileDigests);
                fileDigests.keySet().forEach(path -> {
                    try {
                        Files.deleteIfExists(path);
                    }
                    catch (IOException e) {
                        LOGGER.warn("[{}]: Failed to delete temporary file. file={}", (Object)this.sessionID, path);
                    }
                });
            }
            catch (IOException e) {
                LOGGER.error("[{}]: Unexpected exception while streaming SSTables {}", (Object)this.sessionID, (Object)this.sstableWriter.getOutDir());
                this.setLastStreamFailure(e);
                this.cleanAllReplicas();
            }
        });
    }

    @Override
    protected StreamResult doFinalizeStream() {
        List<CommitResult> cr;
        this.sendRemainingSSTables();
        DirectStreamResult streamResult = new DirectStreamResult(this.sessionID, (Range<BigInteger>)this.tokenRange, this.errors, new ArrayList<RingInstance>(this.replicas), this.sstableWriter.rowCount(), this.sstableWriter.bytesWritten());
        try {
            cr = this.commit(streamResult);
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new RuntimeException(e);
        }
        streamResult.setCommitResults(cr);
        LOGGER.debug("StreamResult: {}", (Object)streamResult);
        BulkWriteValidator.validateClOrFail(this.tokenRangeMapping, this.failureHandler, LOGGER, WRITE_PHASE, this.writerContext.job(), this.writerContext.cluster());
        return streamResult;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    protected void sendRemainingSSTables() {
        File tempDir;
        try {
            try (DirectoryStream<Path> dataFileStream = Files.newDirectoryStream(this.sstableWriter.getOutDir(), "*Data.db");){
                for (Path dataFile : dataFileStream) {
                    if (this.isFileStreamed(dataFile)) continue;
                    this.sendSStableToReplicas(dataFile);
                }
                LOGGER.info("[{}]: Sent SSTables. sstables={}", (Object)this.sessionID, (Object)this.sstableWriter.sstableCount());
            }
            tempDir = this.sstableWriter.getOutDir().toFile();
        }
        catch (IOException exception) {
            try {
                LOGGER.error("[{}]: Unexpected exception while streaming SSTables {}", (Object)this.sessionID, (Object)this.sstableWriter.getOutDir());
                this.cleanAllReplicas();
                throw new RuntimeException(exception);
            }
            catch (Throwable throwable) {
                File tempDir2 = this.sstableWriter.getOutDir().toFile();
                LOGGER.info("[{}]: Removing temporary files after stream session from {}", (Object)this.sessionID, (Object)tempDir2);
                try {
                    FileUtils.deleteDirectory((File)tempDir2);
                    throw throwable;
                }
                catch (IOException exception2) {
                    LOGGER.warn("[{}]: Failed to delete temporary directory {}", new Object[]{this.sessionID, tempDir2, exception2});
                }
                throw throwable;
            }
        }
        LOGGER.info("[{}]: Removing temporary files after stream session from {}", (Object)this.sessionID, (Object)tempDir);
        try {
            FileUtils.deleteDirectory((File)tempDir);
            return;
        }
        catch (IOException exception) {
            LOGGER.warn("[{}]: Failed to delete temporary directory {}", new Object[]{this.sessionID, tempDir, exception});
            return;
        }
    }

    private void sendSStableToReplicas(Path dataFile) {
        int ssTableIdx = this.nextSSTableIdx.getAndIncrement();
        LOGGER.info("[{}]: Pushing SSTable {} to replicas {}", new Object[]{this.sessionID, dataFile, this.replicas.stream().map(RingInstance::nodeName).collect(Collectors.joining(","))});
        this.replicas.removeIf(replica -> !this.trySendSSTableToOneReplica(dataFile, ssTableIdx, (RingInstance)replica, this.sstableWriter.fileDigestMap()));
    }

    private boolean trySendSSTableToOneReplica(Path dataFile, int ssTableIdx, RingInstance replica, Map<Path, Digest> fileDigests) {
        try {
            this.sendSSTableToOneReplica(dataFile, ssTableIdx, replica, fileDigests);
            return true;
        }
        catch (Exception exception) {
            LOGGER.error("[{}]: Failed to stream range {} to instance {}", new Object[]{this.sessionID, this.tokenRange, replica.nodeName(), exception});
            this.writerContext.cluster().refreshClusterInfo();
            this.failureHandler.addFailure((Range<BigInteger>)this.tokenRange, replica, exception.getMessage());
            this.errors.add(new StreamError((Range<BigInteger>)this.tokenRange, replica, exception.getMessage()));
            this.clean(replica, this.sessionID);
            return false;
        }
    }

    private void sendSSTableToOneReplica(Path dataFile, int ssTableIdx, RingInstance instance, Map<Path, Digest> fileHashes) throws IOException {
        try (DirectoryStream<Path> componentFileStream = Files.newDirectoryStream(dataFile.getParent(), SSTables.getSSTableBaseName((Path)dataFile) + "*");){
            for (Path componentFile : componentFileStream) {
                if (componentFile.getFileName().toString().endsWith("Data.db")) continue;
                this.sendSSTableComponent(componentFile, ssTableIdx, instance, fileHashes.get(componentFile));
            }
            this.sendSSTableComponent(dataFile, ssTableIdx, instance, fileHashes.get(dataFile));
        }
    }

    private void sendSSTableComponent(Path componentFile, int ssTableIdx, RingInstance instance, Digest digest) throws IOException {
        Preconditions.checkNotNull((Object)digest, (Object)"All files must have a digest. SSTableWriter should have calculated these.");
        LOGGER.info("[{}]: Uploading {} to {}: size={} digest={}", new Object[]{this.sessionID, componentFile, instance.nodeName(), Files.size(componentFile), digest});
        this.directDataTransferApi.uploadSSTableComponent(componentFile, ssTableIdx, instance, this.sessionID, digest);
        this.recordStreamedFile(componentFile);
    }

    private List<CommitResult> commit(DirectStreamResult streamResult) throws ExecutionException, InterruptedException {
        try (CommitCoordinator cc = CommitCoordinator.commit(this.writerContext, (TransportContext.DirectDataBulkWriterContext)this.transportContext, streamResult);){
            List commitResults = (List)cc.get();
            LOGGER.debug("All CommitResults: {}", (Object)commitResults);
            commitResults.forEach(cr -> BulkWriteValidator.updateFailureHandler(cr, WRITE_PHASE, this.failureHandler));
            List list = commitResults;
            return list;
        }
    }

    private void cleanAllReplicas() {
        HashSet instances = new HashSet(this.replicas);
        this.errors.forEach(streamError -> instances.add(streamError.instance));
        instances.forEach(instance -> this.clean((RingInstance)instance, this.sessionID));
    }

    private void clean(RingInstance instance, String sessionID) {
        if (this.writerContext.job().getSkipClean()) {
            LOGGER.info("Skip clean requested - not cleaning SSTable session {} on instance {}", (Object)sessionID, (Object)instance.nodeName());
            return;
        }
        String jobID = this.writerContext.job().getId();
        LOGGER.info("Cleaning SSTable session {} on instance {}", (Object)sessionID, (Object)instance.nodeName());
        try {
            this.directDataTransferApi.cleanUploadSession(instance, sessionID, jobID);
        }
        catch (Exception exception) {
            LOGGER.warn("Failed to clean SSTables on {} for session {} and ignoring errMsg", new Object[]{instance.nodeName(), sessionID, exception});
        }
    }
}

