/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.pipeline;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.hdds.client.RatisReplicationConfig;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.client.StandaloneReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.SCMCommonPlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.BackgroundSCMService;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMHAManager;
import org.apache.hadoop.hdds.scm.ha.SCMServiceManager;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.pipeline.BackgroundPipelineCreator;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineFactory;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineProvider;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineStateManagerImpl;
import org.apache.hadoop.hdds.scm.pipeline.RatisPipelineUtils;
import org.apache.hadoop.hdds.scm.pipeline.SCMPipelineMetrics;
import org.apache.hadoop.hdds.scm.server.upgrade.FinalizationManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.ozone.ClientVersion;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.util.Time;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipelineManagerImpl
implements PipelineManager {
    private static final Logger LOG = LoggerFactory.getLogger(PipelineManagerImpl.class);
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private PipelineFactory pipelineFactory;
    private PipelineStateManager stateManager;
    private BackgroundPipelineCreator backgroundPipelineCreator;
    private BackgroundSCMService backgroundPipelineScrubber;
    private final ConfigurationSource conf;
    private final EventPublisher eventPublisher;
    private ObjectName pmInfoBean;
    private final SCMPipelineMetrics metrics;
    private final long pipelineWaitDefaultTimeout;
    private final SCMHAManager scmhaManager;
    private SCMContext scmContext;
    private final NodeManager nodeManager;
    private AtomicBoolean freezePipelineCreation;
    private final Clock clock;

    protected PipelineManagerImpl(ConfigurationSource conf, SCMHAManager scmhaManager, NodeManager nodeManager, PipelineStateManager pipelineStateManager, PipelineFactory pipelineFactory, EventPublisher eventPublisher, SCMContext scmContext, Clock clock) {
        this.pipelineFactory = pipelineFactory;
        this.stateManager = pipelineStateManager;
        this.conf = conf;
        this.scmhaManager = scmhaManager;
        this.nodeManager = nodeManager;
        this.eventPublisher = eventPublisher;
        this.scmContext = scmContext;
        this.clock = clock;
        this.pmInfoBean = MBeans.register((String)"SCMPipelineManager", (String)"SCMPipelineManagerInfo", (Object)this);
        this.metrics = SCMPipelineMetrics.create();
        this.pipelineWaitDefaultTimeout = conf.getTimeDuration("hdds.pipeline.report.interval", "60s", TimeUnit.MILLISECONDS);
        this.freezePipelineCreation = new AtomicBoolean();
    }

    public static PipelineManagerImpl newPipelineManager(ConfigurationSource conf, SCMHAManager scmhaManager, NodeManager nodeManager, Table<PipelineID, Pipeline> pipelineStore, EventPublisher eventPublisher, SCMContext scmContext, SCMServiceManager serviceManager, Clock clock) throws IOException {
        PipelineStateManager stateManager = PipelineStateManagerImpl.newBuilder().setPipelineStore(pipelineStore).setRatisServer(scmhaManager.getRatisServer()).setNodeManager(nodeManager).setSCMDBTransactionBuffer(scmhaManager.getDBTransactionBuffer()).build();
        PipelineFactory pipelineFactory = new PipelineFactory(nodeManager, stateManager, conf, eventPublisher, scmContext);
        PipelineManagerImpl pipelineManager = new PipelineManagerImpl(conf, scmhaManager, nodeManager, stateManager, pipelineFactory, eventPublisher, scmContext, clock);
        BackgroundPipelineCreator backgroundPipelineCreator = new BackgroundPipelineCreator(pipelineManager, conf, scmContext, clock);
        pipelineManager.setBackgroundPipelineCreator(backgroundPipelineCreator);
        serviceManager.register(backgroundPipelineCreator);
        if (FinalizationManager.shouldCreateNewPipelines(scmContext.getFinalizationCheckpoint())) {
            pipelineManager.resumePipelineCreation();
        } else {
            pipelineManager.freezePipelineCreation();
        }
        long scrubberIntervalInMillis = conf.getTimeDuration("ozone.scm.pipeline.scrub.interval", "150s", TimeUnit.MILLISECONDS);
        long safeModeWaitMs = conf.getTimeDuration("hdds.scm.wait.time.after.safemode.exit", "5m", TimeUnit.MILLISECONDS);
        BackgroundSCMService backgroundPipelineScrubber = new BackgroundSCMService.Builder().setClock(clock).setScmContext(scmContext).setServiceName("BackgroundPipelineScrubber").setIntervalInMillis(scrubberIntervalInMillis).setWaitTimeInMillis(safeModeWaitMs).setPeriodicalTask(() -> {
            try {
                pipelineManager.scrubPipelines();
            }
            catch (IOException e) {
                LOG.error("Unexpected error during pipeline scrubbing", (Throwable)e);
            }
        }).build();
        pipelineManager.setBackgroundPipelineScrubber(backgroundPipelineScrubber);
        serviceManager.register(backgroundPipelineScrubber);
        return pipelineManager;
    }

    @Override
    public Pipeline buildECPipeline(ReplicationConfig replicationConfig, List<DatanodeDetails> excludedNodes, List<DatanodeDetails> favoredNodes) throws IOException {
        if (replicationConfig.getReplicationType() != HddsProtos.ReplicationType.EC) {
            throw new IllegalArgumentException("Replication type must be EC");
        }
        this.checkIfPipelineCreationIsAllowed(replicationConfig);
        return this.pipelineFactory.create(replicationConfig, excludedNodes, favoredNodes);
    }

    @Override
    public void addEcPipeline(Pipeline pipeline) throws IOException {
        if (pipeline.getReplicationConfig().getReplicationType() != HddsProtos.ReplicationType.EC) {
            throw new IllegalArgumentException("Pipeline replication type must be EC");
        }
        this.checkIfPipelineCreationIsAllowed(pipeline.getReplicationConfig());
        this.addPipelineToManager(pipeline);
    }

    @Override
    public Pipeline createPipeline(ReplicationConfig replicationConfig) throws IOException {
        return this.createPipeline(replicationConfig, Collections.emptyList(), Collections.emptyList());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Pipeline createPipeline(ReplicationConfig replicationConfig, List<DatanodeDetails> excludedNodes, List<DatanodeDetails> favoredNodes) throws IOException {
        this.checkIfPipelineCreationIsAllowed(replicationConfig);
        this.acquireWriteLock();
        try {
            Pipeline pipeline;
            try {
                pipeline = this.pipelineFactory.create(replicationConfig, excludedNodes, favoredNodes);
            }
            catch (IOException e) {
                this.metrics.incNumPipelineCreationFailed();
                throw e;
            }
            this.addPipelineToManager(pipeline);
            Pipeline pipeline2 = pipeline;
            return pipeline2;
        }
        finally {
            this.releaseWriteLock();
        }
    }

    private void checkIfPipelineCreationIsAllowed(ReplicationConfig replicationConfig) throws IOException {
        if (!this.isPipelineCreationAllowed() && !this.factorOne(replicationConfig)) {
            LOG.debug("Pipeline creation is not allowed until safe mode prechecks complete");
            throw new IOException("Pipeline creation is not allowed as safe mode prechecks have not yet passed");
        }
        if (this.freezePipelineCreation.get()) {
            String message = "Cannot create new pipelines while pipeline creation is frozen.";
            LOG.info(message);
            throw new IOException(message);
        }
    }

    private void addPipelineToManager(Pipeline pipeline) throws IOException {
        HddsProtos.Pipeline pipelineProto = pipeline.getProtobufMessage(ClientVersion.CURRENT_VERSION);
        this.acquireWriteLock();
        try {
            this.stateManager.addPipeline(pipelineProto);
        }
        catch (IOException ex) {
            LOG.debug("Failed to add pipeline {}.", (Object)pipeline, (Object)ex);
            this.metrics.incNumPipelineCreationFailed();
            throw ex;
        }
        finally {
            this.releaseWriteLock();
        }
        this.recordMetricsForPipeline(pipeline);
    }

    private boolean factorOne(ReplicationConfig replicationConfig) {
        if (replicationConfig.getReplicationType() == HddsProtos.ReplicationType.RATIS) {
            return ((RatisReplicationConfig)replicationConfig).getReplicationFactor() == HddsProtos.ReplicationFactor.ONE;
        }
        if (replicationConfig.getReplicationType() == HddsProtos.ReplicationType.STAND_ALONE) {
            return ((StandaloneReplicationConfig)replicationConfig).getReplicationFactor() == HddsProtos.ReplicationFactor.ONE;
        }
        return false;
    }

    @Override
    public Pipeline createPipeline(ReplicationConfig replicationConfig, List<DatanodeDetails> nodes) {
        return this.pipelineFactory.create(replicationConfig, nodes);
    }

    @Override
    public Pipeline createPipelineForRead(ReplicationConfig replicationConfig, Set<ContainerReplica> replicas) {
        return this.pipelineFactory.createForRead(replicationConfig, replicas);
    }

    @Override
    public Pipeline getPipeline(PipelineID pipelineID) throws PipelineNotFoundException {
        return this.stateManager.getPipeline(pipelineID);
    }

    @Override
    public boolean containsPipeline(PipelineID pipelineID) {
        try {
            this.getPipeline(pipelineID);
            return true;
        }
        catch (PipelineNotFoundException e) {
            return false;
        }
    }

    @Override
    public List<Pipeline> getPipelines() {
        return this.stateManager.getPipelines();
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig) {
        return this.stateManager.getPipelines(replicationConfig);
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig config, Pipeline.PipelineState state) {
        return this.stateManager.getPipelines(config, state);
    }

    @Override
    public List<Pipeline> getPipelines(ReplicationConfig replicationConfig, Pipeline.PipelineState state, Collection<DatanodeDetails> excludeDns, Collection<PipelineID> excludePipelines) {
        return this.stateManager.getPipelines(replicationConfig, state, excludeDns, excludePipelines);
    }

    @Override
    public int getPipelineCount(ReplicationConfig config, Pipeline.PipelineState state) {
        return this.stateManager.getPipelineCount(config, state);
    }

    @Override
    public void addContainerToPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.addContainerToPipeline(pipelineID, containerID);
    }

    @Override
    public void addContainerToPipelineSCMStart(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.addContainerToPipelineForce(pipelineID, containerID);
    }

    @Override
    public void removeContainerFromPipeline(PipelineID pipelineID, ContainerID containerID) throws IOException {
        this.stateManager.removeContainerFromPipeline(pipelineID, containerID);
    }

    @Override
    public NavigableSet<ContainerID> getContainersInPipeline(PipelineID pipelineID) throws IOException {
        return this.stateManager.getContainers(pipelineID);
    }

    @Override
    public int getNumberOfContainers(PipelineID pipelineID) throws IOException {
        return this.stateManager.getNumberOfContainers(pipelineID);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void openPipeline(PipelineID pipelineId) throws IOException {
        Pipeline pipeline;
        long startNanos = Time.monotonicNowNanos();
        HddsProtos.PipelineID pipelineIdProtobuf = pipelineId.getProtobuf();
        this.acquireWriteLock();
        try {
            pipeline = this.stateManager.getPipeline(pipelineId);
            if (pipeline.isClosed()) {
                throw new IOException("Closed pipeline can not be opened");
            }
            if (pipeline.getPipelineState() == Pipeline.PipelineState.ALLOCATED) {
                this.stateManager.updatePipelineState(pipelineIdProtobuf, HddsProtos.PipelineState.PIPELINE_OPEN);
            }
        }
        finally {
            this.releaseWriteLock();
        }
        this.metrics.updatePipelineCreationLatencyNs(startNanos);
        this.metrics.incNumPipelineCreated();
        this.metrics.createPerPipelineMetrics(pipeline);
    }

    protected void removePipeline(Pipeline pipeline) throws IOException {
        HddsProtos.PipelineID pipelineID = pipeline.getId().getProtobuf();
        this.acquireWriteLock();
        try {
            this.stateManager.removePipeline(pipelineID);
        }
        catch (IOException ex) {
            this.metrics.incNumPipelineDestroyFailed();
            throw ex;
        }
        finally {
            this.releaseWriteLock();
        }
        this.pipelineFactory.close(pipeline.getType(), pipeline);
        LOG.info("Pipeline {} removed.", (Object)pipeline);
        this.metrics.incNumPipelineDestroyed();
    }

    private void closeContainersForPipeline(PipelineID pipelineId) throws IOException {
        NavigableSet<ContainerID> containerIDs = this.stateManager.getContainers(pipelineId);
        ContainerManager containerManager = this.scmContext.getScm().getContainerManager();
        for (ContainerID containerID : containerIDs) {
            if (containerManager.getContainer(containerID).getState() == HddsProtos.LifeCycleState.OPEN) {
                try {
                    containerManager.updateContainerState(containerID, HddsProtos.LifeCycleEvent.FINALIZE);
                }
                catch (InvalidStateTransitionException ex) {
                    throw new IOException(ex);
                }
            }
            this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)containerID);
            LOG.info("Container {} closed for pipeline={}", (Object)containerID, (Object)pipelineId);
        }
    }

    @Override
    public void closePipeline(PipelineID pipelineID) throws IOException {
        HddsProtos.PipelineID pipelineIDProtobuf = pipelineID.getProtobuf();
        this.closeContainersForPipeline(pipelineID);
        if (!this.getPipeline(pipelineID).isClosed()) {
            this.acquireWriteLock();
            try {
                this.stateManager.updatePipelineState(pipelineIDProtobuf, HddsProtos.PipelineState.PIPELINE_CLOSED);
            }
            finally {
                this.releaseWriteLock();
            }
            LOG.info("Pipeline {} moved to CLOSED state", (Object)pipelineID);
        }
        this.metrics.removePipelineMetrics(pipelineID);
    }

    @Override
    public void deletePipeline(PipelineID pipelineID) throws IOException {
        this.removePipeline(this.getPipeline(pipelineID));
    }

    @Override
    public void closeStalePipelines(DatanodeDetails datanodeDetails) {
        List<Pipeline> pipelinesWithStaleIpOrHostname = this.getStalePipelines(datanodeDetails);
        if (pipelinesWithStaleIpOrHostname.isEmpty()) {
            LOG.debug("No stale pipelines for datanode {}", (Object)datanodeDetails);
            return;
        }
        LOG.info("Found {} stale pipelines", (Object)pipelinesWithStaleIpOrHostname.size());
        pipelinesWithStaleIpOrHostname.forEach(p -> {
            try {
                PipelineID id = p.getId();
                LOG.info("Closing the stale pipeline: {}", (Object)id);
                this.closePipeline(id);
                this.deletePipeline(id);
            }
            catch (IOException e) {
                LOG.error("Closing the stale pipeline failed: {}", p, (Object)e);
            }
        });
    }

    @VisibleForTesting
    List<Pipeline> getStalePipelines(DatanodeDetails datanodeDetails) {
        return this.getPipelines().stream().filter(p -> p.getNodes().stream().anyMatch(n -> PipelineManagerImpl.sameIdDifferentHostOrAddress(n, datanodeDetails))).collect(Collectors.toList());
    }

    static boolean sameIdDifferentHostOrAddress(DatanodeDetails left, DatanodeDetails right) {
        return left.getID().equals((Object)right.getID()) && (!left.getIpAddress().equals(right.getIpAddress()) || !left.getHostName().equals(right.getHostName()));
    }

    @Override
    public void scrubPipelines() throws IOException {
        Instant currentTime = this.clock.instant();
        long pipelineScrubTimeoutInMills = this.conf.getTimeDuration("ozone.scm.pipeline.allocated.timeout", "5m", TimeUnit.MILLISECONDS);
        long pipelineDeleteTimoutInMills = this.conf.getTimeDuration("ozone.scm.pipeline.destroy.timeout", "300s", TimeUnit.MILLISECONDS);
        List<Pipeline> candidates = this.stateManager.getPipelines();
        for (Pipeline p : candidates) {
            PipelineID id = p.getId();
            if (p.getPipelineState() == Pipeline.PipelineState.ALLOCATED && currentTime.toEpochMilli() - p.getCreationTimestamp().toEpochMilli() >= pipelineScrubTimeoutInMills) {
                LOG.info("Scrubbing pipeline: id: {} since it stays at ALLOCATED stage for {} mins.", (Object)id, (Object)Duration.between(currentTime, p.getCreationTimestamp()).toMinutes());
                this.closePipeline(id);
                this.deletePipeline(id);
            }
            if (p.getPipelineState() == Pipeline.PipelineState.CLOSED && currentTime.toEpochMilli() - p.getStateEnterTime().toEpochMilli() >= pipelineDeleteTimoutInMills) {
                LOG.info("Scrubbing pipeline: id: {} since it stays at CLOSED stage.", (Object)p.getId());
                this.deletePipeline(id);
            }
            if (!this.isOpenWithUnregisteredNodes(p)) continue;
            LOG.info("Scrubbing pipeline: id: {} as it has unregistered nodes", (Object)p.getId());
            this.closePipeline(id);
        }
    }

    private boolean isOpenWithUnregisteredNodes(Pipeline pipeline) {
        if (!pipeline.isOpen()) {
            return false;
        }
        for (DatanodeDetails dn : pipeline.getNodes()) {
            if (this.nodeManager.getNode(dn.getID()) != null) continue;
            return true;
        }
        return false;
    }

    @Override
    public boolean hasEnoughSpace(Pipeline pipeline, long containerSize) {
        for (DatanodeDetails node : pipeline.getNodes()) {
            if (!(node instanceof DatanodeInfo)) {
                node = this.nodeManager.getDatanodeInfo(node);
            }
            if (SCMCommonPlacementPolicy.hasEnoughSpace(node, 0L, containerSize, null)) continue;
            return false;
        }
        return true;
    }

    @Override
    public void startPipelineCreator() {
        throw new RuntimeException("Not supported in HA code.");
    }

    @Override
    public void triggerPipelineCreation() {
        throw new RuntimeException("Not supported in HA code.");
    }

    @Override
    public void incNumBlocksAllocatedMetric(PipelineID id) {
        this.metrics.incNumBlocksAllocated(id);
    }

    @Override
    public int minHealthyVolumeNum(Pipeline pipeline) {
        return this.nodeManager.minHealthyVolumeNum(pipeline.getNodes());
    }

    @Override
    public int minPipelineLimit(Pipeline pipeline) {
        return this.nodeManager.minPipelineLimit(pipeline.getNodes());
    }

    @Override
    public void activatePipeline(PipelineID pipelineID) throws IOException {
        HddsProtos.PipelineID pipelineIDProtobuf = pipelineID.getProtobuf();
        this.acquireWriteLock();
        try {
            this.stateManager.updatePipelineState(pipelineIDProtobuf, HddsProtos.PipelineState.PIPELINE_OPEN);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    @Override
    public void deactivatePipeline(PipelineID pipelineID) throws IOException {
        HddsProtos.PipelineID pipelineIDProtobuf = pipelineID.getProtobuf();
        this.acquireWriteLock();
        try {
            this.stateManager.updatePipelineState(pipelineIDProtobuf, HddsProtos.PipelineState.PIPELINE_DORMANT);
        }
        finally {
            this.releaseWriteLock();
        }
    }

    @Override
    public void waitPipelineReady(PipelineID pipelineID, long timeout) throws IOException {
        this.waitOnePipelineReady(Lists.newArrayList((Object[])new PipelineID[]{pipelineID}), timeout);
    }

    @Override
    public Pipeline waitOnePipelineReady(Collection<PipelineID> pipelineIDs, long timeout) throws IOException {
        long st = this.clock.millis();
        if (timeout == 0L) {
            timeout = this.pipelineWaitDefaultTimeout;
        }
        List pipelineIDStrs = pipelineIDs.stream().map(id -> id.getId().toString()).collect(Collectors.toList());
        String piplineIdsStr = String.join((CharSequence)",", pipelineIDStrs);
        Pipeline pipeline = null;
        do {
            boolean found = false;
            for (PipelineID pipelineID : pipelineIDs) {
                try {
                    Pipeline tempPipeline = this.stateManager.getPipeline(pipelineID);
                    found = true;
                    if (!tempPipeline.isOpen()) continue;
                    pipeline = tempPipeline;
                    break;
                }
                catch (PipelineNotFoundException e) {
                    LOG.warn("Pipeline {} cannot be found", (Object)pipelineID);
                }
            }
            if (!found) {
                throw new PipelineNotFoundException("The input pipeline IDs " + piplineIdsStr + " cannot be found");
            }
            if (pipeline != null) continue;
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        } while (pipeline == null && this.clock.millis() - st < timeout);
        if (pipeline == null) {
            throw new IOException(String.format("Pipeline %s is not ready in %d ms", piplineIdsStr, timeout));
        }
        return pipeline;
    }

    @Override
    public Map<String, Integer> getPipelineInfo() throws NotLeaderException {
        HashMap<String, Integer> pipelineInfo = new HashMap<String, Integer>();
        for (Pipeline.PipelineState state : Pipeline.PipelineState.values()) {
            pipelineInfo.put(state.toString(), 0);
        }
        this.stateManager.getPipelines().forEach(pipeline -> pipelineInfo.computeIfPresent(pipeline.getPipelineState().toString(), (k, v) -> v + 1));
        return pipelineInfo;
    }

    @Override
    public boolean getSafeModeStatus() {
        return this.scmContext.isInSafeMode();
    }

    @Override
    public void reinitialize(Table<PipelineID, Pipeline> pipelineStore) throws IOException {
        this.stateManager.reinitialize(pipelineStore);
    }

    @Override
    public void freezePipelineCreation() {
        this.freezePipelineCreation.set(true);
        this.backgroundPipelineCreator.stop();
    }

    @Override
    public void resumePipelineCreation() {
        this.freezePipelineCreation.set(false);
        this.backgroundPipelineCreator.start();
    }

    @Override
    public boolean isPipelineCreationFrozen() {
        return this.freezePipelineCreation.get() && !this.backgroundPipelineCreator.isRunning();
    }

    @Override
    public void close() throws IOException {
        if (this.backgroundPipelineCreator != null) {
            this.backgroundPipelineCreator.stop();
        }
        if (this.backgroundPipelineScrubber != null) {
            this.backgroundPipelineScrubber.stop();
        }
        if (this.pmInfoBean != null) {
            MBeans.unregister((ObjectName)this.pmInfoBean);
            this.pmInfoBean = null;
        }
        SCMPipelineMetrics.unRegister();
        this.pipelineFactory.shutdown();
        try {
            this.stateManager.close();
        }
        catch (Exception ex) {
            LOG.error("PipelineStateManagerImpl close failed", (Throwable)ex);
        }
    }

    @VisibleForTesting
    public boolean isPipelineCreationAllowed() {
        return this.scmContext.isLeader() && this.scmContext.isPreCheckComplete();
    }

    @VisibleForTesting
    public void setPipelineProvider(HddsProtos.ReplicationType replicationType, PipelineProvider provider) {
        this.pipelineFactory.setProvider(replicationType, provider);
    }

    @VisibleForTesting
    public PipelineStateManager getStateManager() {
        return this.stateManager;
    }

    @VisibleForTesting
    public SCMHAManager getScmhaManager() {
        return this.scmhaManager;
    }

    private void setBackgroundPipelineCreator(BackgroundPipelineCreator backgroundPipelineCreator) {
        this.backgroundPipelineCreator = backgroundPipelineCreator;
    }

    @VisibleForTesting
    public BackgroundPipelineCreator getBackgroundPipelineCreator() {
        return this.backgroundPipelineCreator;
    }

    private void setBackgroundPipelineScrubber(BackgroundSCMService backgroundPipelineScrubber) {
        this.backgroundPipelineScrubber = backgroundPipelineScrubber;
    }

    @VisibleForTesting
    public BackgroundSCMService getBackgroundPipelineScrubber() {
        return this.backgroundPipelineScrubber;
    }

    @VisibleForTesting
    public PipelineFactory getPipelineFactory() {
        return this.pipelineFactory;
    }

    @VisibleForTesting
    public void setScmContext(SCMContext context) {
        this.scmContext = context;
    }

    private void recordMetricsForPipeline(Pipeline pipeline) {
        this.metrics.incNumPipelineAllocated();
        if (pipeline.isOpen()) {
            this.metrics.incNumPipelineCreated();
            this.metrics.createPerPipelineMetrics(pipeline);
        }
        switch (pipeline.getType()) {
            case STAND_ALONE: {
                return;
            }
            case RATIS: {
                List<Pipeline> overlapPipelines = RatisPipelineUtils.checkPipelineContainSameDatanodes(this.stateManager, pipeline);
                if (!overlapPipelines.isEmpty()) {
                    this.metrics.incNumPipelineContainSameDatanodes();
                    for (Pipeline overlapPipeline : overlapPipelines) {
                        LOG.info("{} and {} have exactly the same set of datanodes: {}", new Object[]{pipeline.getId(), overlapPipeline.getId(), pipeline.getNodeSet()});
                    }
                }
                return;
            }
        }
    }

    @Override
    public void acquireReadLock() {
        this.lock.readLock().lock();
    }

    @Override
    public void releaseReadLock() {
        this.lock.readLock().unlock();
    }

    @Override
    public void acquireWriteLock() {
        this.lock.writeLock().lock();
    }

    @Override
    public void releaseWriteLock() {
        this.lock.writeLock().unlock();
    }
}

