﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribeLocalGatewayVirtualInterfacesRequest : public EC2Request {
 public:
  AWS_EC2_API DescribeLocalGatewayVirtualInterfacesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeLocalGatewayVirtualInterfaces"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The IDs of the virtual interfaces.</p>
   */
  inline const Aws::Vector<Aws::String>& GetLocalGatewayVirtualInterfaceIds() const { return m_localGatewayVirtualInterfaceIds; }
  inline bool LocalGatewayVirtualInterfaceIdsHasBeenSet() const { return m_localGatewayVirtualInterfaceIdsHasBeenSet; }
  template <typename LocalGatewayVirtualInterfaceIdsT = Aws::Vector<Aws::String>>
  void SetLocalGatewayVirtualInterfaceIds(LocalGatewayVirtualInterfaceIdsT&& value) {
    m_localGatewayVirtualInterfaceIdsHasBeenSet = true;
    m_localGatewayVirtualInterfaceIds = std::forward<LocalGatewayVirtualInterfaceIdsT>(value);
  }
  template <typename LocalGatewayVirtualInterfaceIdsT = Aws::Vector<Aws::String>>
  DescribeLocalGatewayVirtualInterfacesRequest& WithLocalGatewayVirtualInterfaceIds(LocalGatewayVirtualInterfaceIdsT&& value) {
    SetLocalGatewayVirtualInterfaceIds(std::forward<LocalGatewayVirtualInterfaceIdsT>(value));
    return *this;
  }
  template <typename LocalGatewayVirtualInterfaceIdsT = Aws::String>
  DescribeLocalGatewayVirtualInterfacesRequest& AddLocalGatewayVirtualInterfaceIds(LocalGatewayVirtualInterfaceIdsT&& value) {
    m_localGatewayVirtualInterfaceIdsHasBeenSet = true;
    m_localGatewayVirtualInterfaceIds.emplace_back(std::forward<LocalGatewayVirtualInterfaceIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters.</p> <ul> <li> <p> <code>local-address</code> - The local
   * address.</p> </li> <li> <p> <code>local-bgp-asn</code> - The Border Gateway
   * Protocol (BGP) Autonomous System Number (ASN) of the local gateway.</p> </li>
   * <li> <p> <code>local-gateway-id</code> - The ID of the local gateway.</p> </li>
   * <li> <p> <code>local-gateway-virtual-interface-id</code> - The ID of the virtual
   * interface.</p> </li> <li> <p> <code>owner-id</code> - The ID of the Amazon Web
   * Services account that owns the local gateway virtual interface.</p> </li> <li>
   * <p> <code>peer-address</code> - The peer address.</p> </li> <li> <p>
   * <code>peer-bgp-asn</code> - The peer BGP ASN.</p> </li> <li> <p>
   * <code>vlan</code> - The ID of the VLAN.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeLocalGatewayVirtualInterfacesRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeLocalGatewayVirtualInterfacesRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with a single call. To retrieve the
   * remaining results, make another call with the returned <code>nextToken</code>
   * value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeLocalGatewayVirtualInterfacesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeLocalGatewayVirtualInterfacesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DescribeLocalGatewayVirtualInterfacesRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_localGatewayVirtualInterfaceIds;

  Aws::Vector<Filter> m_filters;

  int m_maxResults{0};

  Aws::String m_nextToken;

  bool m_dryRun{false};
  bool m_localGatewayVirtualInterfaceIdsHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
