/*
 * Functions for communication with sigma 0
 */

#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/syscalls.h>
#include <l4/kdebug.h>
#include "../include/l4_memory.h"
#include "../include/task.h"
#include "../include/config.h"
/* #define DEBUG */
#include "../include/debug.h"

extern int _end;
unsigned long root_pager_low_memory_start, root_pager_memory_start, 
  root_pager_memory_end;

extern l4_threadid_t sigma0_id;
extern int sigma0_initialized;
l4_kernel_info_t *l4_kernel_info = 0;

#ifdef CONFIG_M4_PAGES
int mb4_pages_dont_work = 0;
#endif

int setup_kernel_info_page(void)
{
  l4_msgdope_t result;
  l4_snd_fpage_t fpage;
  int error;

  /* get sigma0_id if required */
  if (!sigma0_initialized)
    {
      l4_threadid_t pager = L4_INVALID_ID, preempter = L4_INVALID_ID;
      dword_t eip = -1, esp = -1, eflags;
      l4_thread_ex_regs(l4_myself(), eip, esp, 
		 &preempter, &pager, &eflags, &eip, &esp);
      if (!l4_is_invalid_id(pager))
	{
	  sigma0_id = pager;
	  sigma0_initialized = 1;
	}
      else
	enter_kdebug("request_page_from_sigma0: invalid pager");
    }
  l4_kernel_info = (l4_kernel_info_t *)0;
  /* request kernel info page page from sigma 0 */
  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
			   1, 1,
			   L4_IPC_MAPMSG(KERNEL_INFO_PAGE_ADDRESS, 
					 L4_LOG2_PAGESIZE), 
			   &fpage.snd_base, &fpage.fpage.fpage,  
			   L4_IPC_NEVER, &result);
  if (!error)
    {
      if (l4_ipc_fpage_received(result))
	{
	  l4_kernel_info = (l4_kernel_info_t *)KERNEL_INFO_PAGE_ADDRESS;
	  return 0;
	}
      else
	printk("setup_kernel_info_page:" 
	       " sigma 0 denied mapping of kernel info page\n");
    }
  else
    printk("setup_kernel_info_page:"
	   " ipc error %x while requesting kernel info page\n", error);
  return 1; 
}

int check_error(int error, l4_msgdope_t result, l4_snd_fpage_t fpage, 
		 int address, int size)
{
  if (!error)
    {
      if (l4_ipc_fpage_received(result))
	{
	  if (l4_ipc_is_fpage_writable(fpage.fpage))
	    {
	      if (size == RP_4M)
		{
		  if (fpage.fpage.fp.size == RP_4M)
		    {
/* 		      enter_kdebug("4m page received"); */
		      kd_display("\n\r4m page received\\n\\r");
		      return (RP_4M | RP_WRITABLE);
		    }
		  else
		    return RP_WRITABLE;
		}
	      else
		return RP_WRITABLE;
	    }
	  else
	    {
	      enter_kdebug("sigma0 mapped page read only");
	      return RP_NOT_WRITABLE;
	    }
	}
      else
	{

	  kd_display("sigma0 denies mapping of address "); 
 	  outhex32(address); 
	  if ((address < ((unsigned long)&_stext & PAGE_MASK)) 
	      || (address > (unsigned long)&_end))
	    {
 	      kd_display("\\r\\nNo kernel code/data\\r\\n");
	      return RP_NOT_MAPPED;
	    }
	  else
	    {
 	      kd_display("\\r\\nkernel code/data\\r\\n"); 
	      return RP_WRITABLE;
	    }
	}
    }
  else
    {
      kd_display("request_page_from_sigma0: ipc error: ");
      outhex32(error);
      enter_kdebug("error while calling sigma 0");
    }
  return RP_NOT_MAPPED;

} 
#define MB4 (4*1024*1024)
int request_page_from_sigma0(unsigned long address)
{
  l4_msgdope_t result;
  l4_snd_fpage_t fpage;
  int error;
#ifdef CONFIG_M4_PAGES
  static int mb4_page_already_mapped=0;
#endif
  /* get sigma0_id if required */
  if (!sigma0_initialized)
    {
      l4_threadid_t pager = L4_INVALID_ID, preempter = L4_INVALID_ID;
      dword_t eip = -1, esp = -1, eflags;
      l4_thread_ex_regs(l4_myself(), eip, esp, 
		 &preempter, &pager, &eflags, &eip, &esp);
      if (!l4_is_invalid_id(pager))
	{
	  sigma0_id = pager;
	  sigma0_initialized = 1;
	}
      else
	enter_kdebug("request_page_from_sigma0: invalid pager");
    }

#ifdef CONFIG_M4_PAGES
  if ((address & (MB4-1)) || (address < MB4) || mb4_pages_dont_work)
    {
      /* normal page */
      if (!mb4_page_already_mapped)
	{
#endif
/* 	  herc_printf("requesting page: %lx\n", address); */
	  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
				   address, 0,
				   L4_IPC_MAPMSG(0, L4_HALF_ADDRESS_SPACE), 
				   &fpage.snd_base, &fpage.fpage.fpage,  
				   L4_IPC_NEVER, &result);
	  return check_error(error, result, fpage, address, 0);
#ifdef CONFIG_M4_PAGES
	}
      else
      	return RP_WRITABLE;
    }
  else
    {
      /* 4mb page */
      if (address != MB4)
	{
/* 	  enter_kdebug("requesting 4m page"); */
	  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
				   address + 1, 22*4, 
				   L4_IPC_MAPMSG(0, L4_HALF_ADDRESS_SPACE), 
				   &fpage.snd_base, &fpage.fpage.fpage,  
				   L4_IPC_NEVER, &result);
/* 	  enter_kdebug("4m page requested"); */
	  error = check_error(error, result, fpage, address, RP_4M);
	  if (error == (RP_4M | RP_WRITABLE))
	    {
	      mb4_page_already_mapped = 1;
	      return RP_WRITABLE;
	    }
	  else
	    {
	      mb4_page_already_mapped = 0;
	      return error;
	    }
	}
      else
	{
	  kd_display("\\r\\nmb4 already requested\\r\\n");
	  mb4_page_already_mapped = 1;
	  return RP_WRITABLE;
	}
    }
#endif
}

int 
request_mem_area_from_sigma0(dword_t fault_address, dword_t pte_val)
{
  l4_msgdope_t result;
  dword_t send_base, fpage;
  int error;

  if (!sigma0_initialized)
    panic("request_mem_area_from_sigma0 not inititialized");
  
/*   printk("request for physical area %x at va %x\n",  */
/* 	 fault_address, pte_val); */
/*   kd_display("request for physical area\\r\\n"); */

  fault_address &= L4_4MB_MASK;

#if 0
  if (((fault_address & PAGE_MASK)  & ~L4_4MB_MASK) != 0)
    {
      printk("request_mem_area_from_sigma0: area not 4MB aligned (%x)\n",
	     fault_address);
      enter_kdebug("request_mem_area_from_sigma0: invalid alignment");
    }
#endif

  /* request area from sigma 0 */
  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
			   (pte_val & PAGE_MASK) - L4_ONE_GIG, 0,
			   L4_IPC_MAPMSG(fault_address, L4_LOG2_4MB), 
			   &send_base, &fpage,  
			   L4_IPC_NEVER, &result);
  if (error)
    {
/*       printk("error %x while requesting region from sigma0\n",  */
/* 	     result.msgdope); */
      enter_kdebug("error in request_mem_area_from_sigma0");
      return -1;
    }
  else
    return 0;
}
  



void 
request_physical_page_from_sigma0(unsigned long physical_page, 
				  unsigned long access, 
				  unsigned long vm_page)
{
  dword_t fpage, snd_base, error;
  l4_msgdope_t result;
#ifdef DEBUG
  herc_printf("req_phy_page: page: %lx (%s), vm_page: %lx\n",
	      physical_page, ((access == L4_WRITE_ACCESS) ? "write" : "read"),
	      vm_page);
  enter_kdebug("rpp: request_page");
#endif

  if (physical_page >= root_pager_memory_end)
    enter_kdebug("pysical page >= memory end");

  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
			   (physical_page & ~L4_ACCESS_MASK) | access, 
			   0,
			   L4_IPC_MAPMSG(vm_page, L4_LOG2_PAGESIZE),
			   &fpage, &snd_base, L4_IPC_NEVER,
			   &result);
  if (error)
    enter_kdebug("rpp: ipc error :(");
  if (!l4_ipc_fpage_received(result))
    enter_kdebug("rpp: no mapping");
#ifdef DEBUG
  enter_kdebug("rpp: request_page finished");
#endif
}

void
request_zero_page_from_sigma0(void)
{
  dword_t fpage, snd_base, error;
  l4_msgdope_t result;
  error = l4_i386_ipc_call(sigma0_id, L4_IPC_SHORT_MSG,
			   L4_WRITE_ACCESS, 
			   0,
			   L4_IPC_MAPMSG(0, PAGE_SHIFT),
			   &fpage, &snd_base, L4_IPC_NEVER,
			   &result);
   if (error)
    enter_kdebug("rap: ipc error :(");
  if (!l4_ipc_fpage_received(result))
    enter_kdebug("rap: no mapping");
}
