/* drivers/atm/atmtcp.c - ATM over TCP "device" driver */
 
/* Written 1995,1996 by Werner Almesberger, EPFL LRC */
/* Various enhancements 1996 by Ronald A. McCormick */


#include <linux/module.h>
#include <linux/kernel.h>		/* printk */
#include <linux/mm.h>			/* verify_area */
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/errno.h>
#include <linux/skbuff.h>
#include <linux/in.h>
#include <linux/sched.h>		/* xtime */
#include <asm/segment.h>		/* set_fs, get_fs, ... */
#include <asm/byteorder.h>		/* ntohs, htons */


#define DEV_LABEL	"atmtcp"


#define MAX_VPI_BITS	8
#define MAX_VCI_BITS	16


#define PRIV(c) ((struct private *) ((c)->dev_data))


extern int (*atmtcp_attach_hook)(struct socket *sock);


/* BIG BUG: we can never get rid of an ATMTCP interface @@@ */


/*
 *  correct compilation of this is implementation-dependent !
 */

struct atmtcp_hdr {
	unsigned short	vpi;		
	unsigned short	vci;
	unsigned short	length;		/* ... of data part */
};

struct private {
	struct atm_dev	*dev;		/* device back pointer */
	struct socket	*sock;		/* file descriptor */
};


static int atmtcp_open(struct atm_vcc *vcc,short vpi,int vci)
{
	int error;

	error = atm_find_ci(vcc,&vpi,&vci);
	if (error) return error;
	vcc->vpi = vpi;
	vcc->vci = vci;
	if (vpi == ATM_VPI_UNSPEC  || vci == ATM_VCI_UNSPEC) return 0;
#ifdef DEBUG
	printk(KERN_DEBUG DEV_LABEL "(itf %d): open %d.%d\n",vcc->dev->number,
	    vcc->vpi,vcc->vci);
#endif
	vcc->flags |= ATM_VF_ADDR | ATM_VF_READY;
	return 0;
}


int atmtcp_ioctl(struct atm_dev *dev,unsigned int cmd,unsigned long arg)
{
	if (cmd == ATM_SETCIRANGE) {
		struct atm_cirange ci;
		struct atm_vcc *walk;

		memcpy_fromfs(&ci,(void *) arg,sizeof(struct atm_cirange));
		if (ci.vpi_bits == ATM_CI_MAX) ci.vpi_bits = MAX_VPI_BITS;
		if (ci.vci_bits == ATM_CI_MAX) ci.vci_bits = MAX_VCI_BITS;
		if (ci.vpi_bits > MAX_VPI_BITS || ci.vpi_bits < 0 ||
		    ci.vci_bits > MAX_VCI_BITS || ci.vci_bits < 0)
			return -EINVAL;
/*
 * should all this checking be done in net/atm ?
 */
		for (walk = dev->vccs; walk; walk = walk->next)
			if ((walk->vpi >> ci.vpi_bits) ||
			    (walk->vci >> ci.vci_bits)) return -EBUSY;
		dev->ci_range = ci;
		return 0;
	}
	return -EINVAL;
}



static int do_recv(struct socket *sock,unsigned char *ubuf,int size,
    int nonblock)
{
	struct iovec iov;
	struct msghdr msg;

	iov.iov_base = ubuf;
	iov.iov_len = size;

	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_control = NULL;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;

	return sock->ops->recvmsg(sock,&msg,size,nonblock,0,NULL);
}


static int do_send(struct socket *sock,const void *buff,int len,int nonblock)
{
	struct iovec iov;
	struct msghdr msg;

	iov.iov_base = (void *)buff;
	iov.iov_len = len;

	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_control = NULL;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;

	return sock->ops->sendmsg(sock,&msg,len,nonblock,0);
}


static int atmtcp_setsockopt(struct atm_vcc *vcc,int level,int optname,
    char *optval,int optlen)
{
	return -EINVAL;
}


static int atmtcp_send(struct atm_vcc *vcc,struct sk_buff *skb)
{
	struct atmtcp_hdr hdr;
	unsigned long old_fs;
	int len,size;
	void *pos;

	if (!PRIV(vcc->dev)->sock) {
		dev_kfree_skb(skb,FREE_WRITE);
		return -EINVAL;
	}
	hdr.vpi = htons(vcc->vpi);
	hdr.vci = htons(vcc->vci);
	old_fs = get_fs();
	set_fs(get_ds());
	hdr.length = htons(skb->len);
	size = do_send(PRIV(vcc->dev)->sock,(char *) &hdr,sizeof(hdr),0);
	if (size <= 0) {
		printk(KERN_WARNING "atmtcp_send: PDU lost; errno = %d\n",
		    -size);
		dev_kfree_skb(skb,FREE_WRITE);
		set_fs(old_fs);
		vcc->stats->tx_err++;
		return size;
	}
	pos = skb->data;
	len = skb->len;
	while (len) {
		size = do_send(PRIV(vcc->dev)->sock,pos,len,0);
		if (size < 0 && size != -EAGAIN) {
			printk(KERN_ERR "atmtcp_send: link failure; errno = "
			    "%d\n",-size);
			dev_kfree_skb(skb,FREE_WRITE);
			set_fs(old_fs);
			vcc->stats->tx_err++;
			return size;
		}
		if (size > 0) {
			len -= size;
			pos += size;
		}
	}
	dev_kfree_skb(skb,FREE_WRITE);
	set_fs(old_fs);
	vcc->stats->tx++;
	return 0;
}


static void atmtcp_poll(struct atm_vcc *vcc,int nonblock)
{
	struct atm_dev *dev;
	struct atm_vcc *walk;
	struct sk_buff *skb;
	struct atmtcp_hdr hdr;
	unsigned long old_fs;
	int size,left;
	void *pos;

	if (!(dev = vcc->dev) || !PRIV(dev)->sock) return;
	if (skb_peek(&vcc->recvq)) nonblock = 1;
	old_fs = get_fs();
	set_fs(get_ds());
	while ((size = do_recv(PRIV(dev)->sock,(char*) &hdr,sizeof(hdr),
	    nonblock) == sizeof(hdr))) {
		if (!(skb = vcc->peek(vcc,ntohs(hdr.length),NULL))) {
			printk(KERN_WARNING "atmtcp_poll: peek reject (%d)\n",
			    ntohs(hdr.length));
			break;
		}
		skb->atm.timestamp = xtime;
		skb->len = ntohs(hdr.length);
		skb->free = 1;
		pos = skb->data;
		for (left = ntohs(hdr.length); left; left -= size) {
			size = do_recv(PRIV(dev)->sock,pos,left,0);
			if (size == -EAGAIN) size = 0;
			if (size < 0) {
				printk(KERN_ERR "atmtcp_poll: bad read: %d\n",
				    size);
				vcc->stats->rx_err++;
				break;
			}
			pos += size;
		}
		for (walk = dev->vccs; walk; walk = walk->next)
			if (walk->vpi == ntohs(hdr.vpi) && walk->vci ==
			    ntohs(hdr.vci)) break;
		if (walk) {
			vcc->push(vcc,skb);
			nonblock = 1;
			vcc->stats->rx++;
		}
		else {
			printk(KERN_ERR "atmtcp_poll: bad label %d.%d at itf "
			    "%d\n",ntohs(hdr.vpi),ntohs(hdr.vci),dev->number);
			kfree_skb(skb,FREE_READ);
			vcc->stats->rx_err++;
		}
	}
	set_fs(old_fs);
	if (size > 0 && size != sizeof(hdr)) {
		printk(KERN_ERR "atmtcp_poll: bad header (%d)\n",size);
		vcc->stats->rx_err++;
	}
}


static struct atmdev_ops ops = {
	atmtcp_open,
	NULL,		/* no close */
	atmtcp_ioctl,
	NULL,		/* no getsockopt */
	atmtcp_setsockopt,
	atmtcp_send,
	NULL,		/* no direct writes */
	atmtcp_poll,
	NULL,		/* no send_oam */
	NULL,		/* no phy_put */
	NULL,		/* no phy_get */
	NULL,		/* no feedback */
	NULL,		/* no change_qos */
	NULL		/* no free_rx_skb */
};


int atmtcp_attach(struct socket *sock)
{
	struct private *dsc;

	if (!suser()) return -EPERM;
	dsc = kmalloc(sizeof(struct private),GFP_KERNEL);
	if (!dsc) return -ENOMEM;
	dsc->dev = atm_dev_register(DEV_LABEL,&ops,0);
	if (!dsc->dev) {
		kfree(dsc);
		return -EBUSY;
	}
	dsc->dev->dev_data = dsc;
	dsc->dev->ci_range.vpi_bits = MAX_VPI_BITS;
	dsc->dev->ci_range.vci_bits = MAX_VCI_BITS;
	sock->inode->i_count++;
	dsc->sock = sock;
	printk(KERN_NOTICE DEV_LABEL "(itf %d): ready\n",dsc->dev->number);
	MOD_INC_USE_COUNT;
	return dsc->dev->number;
}


int atmtcp_init(void)
{
	printk(KERN_NOTICE DEV_LABEL ": ready (dynamic device creation)\n");
	atmtcp_attach_hook = atmtcp_attach;
	return 1;
}


#ifdef MODULE

int init_module(void)
{
	(void) atmtcp_init();
	return 0;
}


void cleanup_module(void)
{
	/*
	 * We currently have no means to detach ATM devices. That'll follow
	 * a bit later. (Needs careful usage tracking and open/close
	 * interlocking.)
	 */
	atmtcp_attach_hook = NULL;
}

#endif
