/* drivers/atm/eni.c - Efficient Networks ENI155P device driver */
 
/* Written 1995-1997 by Werner Almesberger, EPFL LRC */
 

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <linux/errno.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/sonet.h>
#include <linux/skbuff.h>
#include <linux/time.h>
#include <linux/sched.h> /* for xtime */
#include <linux/delay.h>
#include <linux/uio.h>
#include <linux/atm_eni.h>
#include <asm/system.h>
#include <asm/string.h>
#include <asm/byteorder.h>

#include "tonga.h"
#include "midway.h"
#include "suni.h"
#include "eni.h"


/*
 * TODO:
 *
 * Show stoppers
 *  none
 *
 * Major
 *  - test ASIC Tonga support
 *
 * Minor
 *  - OAM support
 *  - fix bugs listed below
 */

/*
 * KNOWN BUGS:
 *
 * - may run into JK-JK bug and deadlock
 * - should allocate UBR channel first
 * - ATM DD interface doesn't support double connect(2) yet
 * - buffer space allocation algorithm is stupid
 *   (RX: should be maxSDU+maxdelay*rate
 *    TX: should be maxSDU+min(maxSDU,maxdelay*rate) )
 * - adapter memory may fragment badly
 * - ATM DD interface doesn't know about CLP
 * - doesn't support OAM cells
 * - buddy implementation is hyper-inefficient
 * - eni_put_free may hang if not putting memory fragments that _complete_
 *   2^n block
 * - keeps IRQ even if initialization fails
 */


#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif

#ifndef CONFIG_ATM_ENI_DEBUG


#define NULLCHECK(x)

#define EVENT(s,a,b)


static void event_dump(void)
{
}


#else


/* 
 * NULL pointer checking
 */

#define NULLCHECK(x) \
  if ((unsigned long) (x) < 0x30) printk(KERN_CRIT #x "==0x%x\n", (int) (x))

/*
 * Very extensive activity logging. Greatly improves bug detection speed but
 * costs a few Mbps if enabled.
 */

#define EV 64

static const char *ev[EV];
static unsigned long ev_a[EV],ev_b[EV];
static int ec = 0;


static void EVENT(const char *s,unsigned long a,unsigned long b)
{
	ev[ec] = s; 
	ev_a[ec] = a;
	ev_b[ec] = b;
	ec = (ec+1) % EV;
}


static void event_dump(void)
{
	int n,i;

	for (n = 0; n < EV; n++) {
		i = (ec+n) % EV;
		printk(KERN_NOTICE);
		printk(ev[i] ? ev[i] : "(null)",ev_a[i],ev_b[i]);
	}
}


#endif /* CONFIG_ATM_ENI_DEBUG */


/*
 * NExx   must not be equal at end
 * EExx   may be equal at end
 * xxPJOK verify validity of pointer jumps
 * xxPMOK operating on a circular buffer of "c" words
 */

#define NEPJOK(a0,a1,b) \
    ((a0) < (a1) ? (b) <= (a0) || (b) > (a1) : (b) <= (a0) && (b) > (a1))
#define EEPJOK(a0,a1,b) \
    ((a0) < (a1) ? (b) < (a0) || (b) >= (a1) : (b) < (a0) && (b) >= (a1))
#define NEPMOK(a0,d,b,c) NEPJOK(a0,(a0+d) & (c-1),b)
#define EEPMOK(a0,d,b,c) EEPJOK(a0,(a0+d) & (c-1),b)


static int tx_complete = 0,dma_complete = 0,queued = 0,requeued = 0,
  backlogged = 0,rx_enqueued = 0,rx_dequeued = 0,pushed = 0,submitted = 0,
  putting = 0;

static struct atm_dev *eni_boards = NULL;


/*-------------------------------- utilities --------------------------------*/


static void dump_mem(struct eni_dev *eni_dev)
{
	int i;

	for (i = 0; i < eni_dev->free_len; i++)
		printk(KERN_DEBUG "  %d: 0x%lx %d\n",i,
		    eni_dev->free_list[i].start,
		    1 << eni_dev->free_list[i].order);
}


static void dump(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;

	int i;

	eni_dev = ENI_DEV(dev);
	printk(KERN_NOTICE "Free memory\n");
	dump_mem(eni_dev);
	printk(KERN_NOTICE "TX buffers\n");
	for (i = 0; i < NR_CHAN; i++)
		if (eni_dev->tx[i].send)
			printk(KERN_NOTICE "  TX %d @ 0x%p: %ld\n",i,
			    eni_dev->tx[i].send,eni_dev->tx[i].words*4);
	printk(KERN_NOTICE "RX buffers\n");
	for (i = 0; i < 1024; i++)
		if (eni_dev->rx_map[i] && ENI_VCC(eni_dev->rx_map[i])->rx)
			printk(KERN_NOTICE "  RX %d @ 0x%p: %ld\n",i,
			    ENI_VCC(eni_dev->rx_map[i])->recv,
			    ENI_VCC(eni_dev->rx_map[i])->words*4);
	printk(KERN_NOTICE "----\n");
}


static void eni_put_free(struct eni_dev *eni_dev,unsigned long start,
    unsigned long size)
{
	struct eni_free *list;
	int len,order;

	DPRINTK("init 0x%lx+%ld(0x%lx)\n",start,size,size);
	start += eni_dev->base_diff;
	list = eni_dev->free_list;
	len = eni_dev->free_len;
	while (size) {
		if (len >= eni_dev->free_list_size) {
			printk(KERN_CRIT "eni_put_free overflow (0x%lx,%ld)\n",
			    start,size);
			break;
		}
		for (order = 0; !((start | size) & (1 << order)); order++);
		if (MID_MIN_BUF_SIZE > (1 << order)) {
			printk(KERN_CRIT "eni_put_free: order %d too small\n",
			    order);
			break;
		}
		list[len].start = start;
		list[len].order = order;
		len++;
		start += 1 << order;
		size -= 1 << order;
	}
	eni_dev->free_len = len;
	/*dump_mem(eni_dev);*/
}


static unsigned long eni_alloc_mem(struct eni_dev *eni_dev,unsigned long *size)
{
	struct eni_free *list;
	unsigned long start;
	int len,i,order,best_order,index;

	list = eni_dev->free_list;
	len = eni_dev->free_len;
	if (*size < MID_MIN_BUF_SIZE) *size = MID_MIN_BUF_SIZE;
	if (*size > MID_MAX_BUF_SIZE) return 0;
	for (order = 0; (1 << order) < *size; order++);
	DPRINTK("trying: %ld->%d\n",*size,order);
	best_order = 65; /* we don't have more than 2^64 of anything ... */
	index = 0; /* silence GCC */
	for (i = 0; i < len; i++)
		if (list[i].order == order) {
			best_order = order;
			index = i;
			break;
		}
		else if (best_order > list[i].order && list[i].order > order) {
				best_order = list[i].order;
				index = i;
			}
	if (best_order == 65) return 0;
	start = list[index].start-eni_dev->base_diff;
	list[index] = list[--len];
	eni_dev->free_len = len;
	*size = 1 << order;
	eni_put_free(eni_dev,start+*size,(1 << best_order)-*size);
	DPRINTK("%ld bytes (order %d) at 0x%lx\n",*size,order,start);
	for (i = (*size >> 2)-1; i >= 0; i--) /* never leak data */
		((unsigned long *) start)[i] = 0;
	/*dump_mem(eni_dev);*/
	return start;
}


static void eni_free_mem(struct eni_dev *eni_dev,unsigned long start,
    unsigned long size)
{
	struct eni_free *list;
	int len,i,order;

	start += eni_dev->base_diff;
	list = eni_dev->free_list;
	len = eni_dev->free_len;
	for (order = -1; size; order++) size >>= 1;
	DPRINTK("eni_free_mem: 0x%lx+0x%lx (order %d)\n",start,size,order);
	for (i = 0; i < len; i++)
		if (list[i].start == (start^(1 << order)) &&
		    list[i].order == order) {
			DPRINTK("match[%d]: 0x%lx/0x%lx(0x%x), %d/%d\n",i,
			    list[i].start,start,1 << order,list[i].order,order);
			list[i] = list[--len];
			start &= ~(1 << order);
			order++;
			i = -1;
			continue;
		}
	if (len >= eni_dev->free_list_size) {
		printk(KERN_ALERT "eni_free_mem overflow (0x%lx,%d)\n",start,
		    order);
		return;
	}
	list[len].start = start;
	list[len].order = order;
	eni_dev->free_len = len+1;
	/*dump_mem(eni_dev);*/
}


/*----------------------------------- RX ------------------------------------*/


#define ENI_VCC_NOS ((struct atm_vcc *) 1)


static __u32 fetch(struct atm_vcc *vcc,int i)
{
	return ntohl(ENI_VCC(vcc)->recv[(ENI_VCC(vcc)->descr+i+1) &
	    (ENI_VCC(vcc)->words-1)]);		           /* ^ skip descr */
}


static void rx_ident_err(struct atm_vcc *vcc)
{
	struct atm_dev *dev;
	struct eni_vcc *eni_vcc;

	dev = vcc->dev;
	/* immediately halt adapter */
	ENI_DEV(dev)->reg[MID_MC_S] &= ~(MID_DMA_ENABLE | MID_TX_ENABLE |
	  MID_RX_ENABLE);
	/* dump useful information */
	eni_vcc = ENI_VCC(vcc);
	printk(KERN_ALERT DEV_LABEL "(itf %d): driver error - RX ident "
	    "mismatch\n",dev->number);
	printk(KERN_ALERT "  VCI %d, rxing %d, words %ld\n",vcc->vci,
	    eni_vcc->rxing,eni_vcc->words);
	printk(KERN_ALERT "  host descr 0x%08lx, rx pos 0x%08lx, descr value "
	    "0x%08lx\n",eni_vcc->descr,eni_vcc->rx_pos,
	    eni_vcc->recv[eni_vcc->descr]);
	printk(KERN_ALERT "  last 0x%p, servicing %d\n",eni_vcc->last,
	    eni_vcc->servicing);
	EVENT("---dump ends here---\n",0,0);
	printk(KERN_NOTICE "---recent events---\n");
	event_dump();
	ENI_DEV(dev)->fast = NULL; /* really stop it */
	ENI_DEV(dev)->slow = NULL;
	skb_queue_head_init(&ENI_DEV(dev)->rx_queue);
}


static int do_rx_dma(struct atm_vcc *vcc,struct sk_buff *skb,
    unsigned long skip,unsigned long size,unsigned long eff)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long dma_rd,dma_wr;
	unsigned long dma[RX_DMA_BUF*2];
	unsigned long paddr,here;
	int i,j;

	eni_dev = ENI_DEV(vcc->dev);
	eni_vcc = ENI_VCC(vcc);
	paddr = 0; /* GCC, shut up */
	if (skb) {
		paddr = (unsigned long) skb->data;
		if (paddr & 3)
			printk(KERN_CRIT DEV_LABEL "(itf %d): VCI %d has "
			    "mis-aligned RX data (0x%lx)\n",vcc->dev->number,
			    vcc->vci,paddr);
		skb->atm.size = size+skip;
		    /* PDU plus descriptor */
		skb->atm.vcc = vcc;
	}
	j = 0;
	if ((eff && skip) || 1) { /* @@@ actually, skip is always == 1 ... */
		here = (eni_vcc->descr+skip) & (eni_vcc->words-1);
		dma[j++] = (here << MID_DMA_COUNT_SHIFT) | (vcc->vci
		    << MID_DMA_VCI_SHIFT) | MID_DT_JK;
		j++;
	}
	here = (eni_vcc->descr+size+skip) & (eni_vcc->words-1);
	if (!eff) size += skip;
	else {
		unsigned long words;

		if (!size) {
			DPRINTK("strange things happen ...\n");
			EVENT("strange things happen ... (skip=%ld,eff=%ld)\n",
			    size,eff);
		}
		words = eff;
		if (paddr & 15) {
			unsigned long init;

			init = 4-((paddr & 15) >> 2);
			if (init > words) init = words;
			dma[j++] = MID_DT_WORD | (init << MID_DMA_COUNT_SHIFT) |
			    (vcc->vci << MID_DMA_VCI_SHIFT);
			dma[j++] = paddr;
			paddr += init << 2;
			words -= init;
		}
		if (words & ~7) {
#if 0 /* works with *some* PCI chipsets ... */
			dma[j++] = MID_DT_8W | ((words >> 3) <<
			    MID_DMA_COUNT_SHIFT) | (vcc->vci <<
			    MID_DMA_VCI_SHIFT);
			dma[j++] = paddr;
			paddr += (words & ~7) << 2;
			words &= 7;
#else
			dma[j++] = MID_DT_4W | ((words >> 2) <<
			    MID_DMA_COUNT_SHIFT) | (vcc->vci <<
			    MID_DMA_VCI_SHIFT);
			dma[j++] = paddr;
			paddr += (words & ~3) << 2;
			words &= 3;
#endif
		}
		if (words) {
			dma[j++] = MID_DT_WORD | (words << MID_DMA_COUNT_SHIFT)
			    | (vcc->vci << MID_DMA_VCI_SHIFT);
			dma[j++] = paddr;
		}
	}
	if (size != eff) {
		dma[j++] = (here << MID_DMA_COUNT_SHIFT) |
		    (vcc->vci << MID_DMA_VCI_SHIFT) | MID_DT_JK;
		j++;
	}
	if (!j || j > 2*RX_DMA_BUF) {
		printk(KERN_CRIT DEV_LABEL "!j or j too big!!!\n");
		if (skb) kfree_skb(skb,FREE_READ);
		return -1;
	}
	dma[j-2] |= MID_DMA_END;
	j = j >> 1;
	dma_wr = eni_dev->reg[MID_DMA_WR_RX];
	dma_rd = eni_dev->reg[MID_DMA_RD_RX];
	/*
	 * Can I move the dma_wr pointer by 2j+1 positions without overwriting
	 * data that hasn't been read (position of dma_rd) yet ?
	 */
	if (!NEPMOK(dma_wr,j+j+1,dma_rd,NR_DMA_RX)) { /* @@@ +1 is ugly */
		printk(KERN_WARNING DEV_LABEL "(itf %d): RX DMA full\n",
		    vcc->dev->number);
		if (skb) kfree_skb(skb,FREE_READ);
		return -1;
	}
        for (i = 0; i < j; i++) {
		eni_dev->rx_dma[dma_wr*2] = dma[i*2];
		eni_dev->rx_dma[dma_wr*2+1] = dma[i*2+1];
		dma_wr = (dma_wr+1) & (NR_DMA_RX-1);
        }
	if (skb) {
		skb->atm.pos = eni_vcc->descr+size+1;
		skb_queue_tail(&eni_dev->rx_queue,skb);
eni_vcc->last = skb;
rx_enqueued++;
	}
	eni_vcc->descr = here;
	eni_dev->reg[MID_DMA_WR_RX] = dma_wr;
	return 0;
}


static void discard(struct atm_vcc *vcc,unsigned long size)
{
	struct eni_vcc *eni_vcc;

	eni_vcc = ENI_VCC(vcc);
	EVENT("discard (size=%ld)\n",size,0);
	while (do_rx_dma(vcc,NULL,1,size,0)) EVENT("BUSY LOOP",0,0);
	    /* could do a full fallback, but that might be more expensive */
	if (eni_vcc->rxing) eni_vcc->last->atm.pos += size+1;
	else eni_vcc->rx_pos = (eni_vcc->rx_pos+size+1) & (eni_vcc->words-1);
}


/*
 * TODO: should check whether direct copies (without DMA setup, dequeuing on
 * interrupt, etc.) aren't much faster for AAL0
 */

static int rx_aal0(struct atm_vcc *vcc)
{
	struct eni_vcc *eni_vcc;
	unsigned long descr;
	unsigned long length;
	struct sk_buff *skb;

	DPRINTK(">rx_aal0\n");
	eni_vcc = ENI_VCC(vcc);
	descr = eni_vcc->recv[eni_vcc->descr];
	if ((descr & MID_RED_IDEN) != (MID_RED_RX_ID << MID_RED_SHIFT)) {
		rx_ident_err(vcc);
		return 1;
	}
	if (descr & MID_RED_T) {
		DPRINTK(DEV_LABEL "(itf %d): trashing empty cell\n",
		    vcc->dev->number);
		length = 0;
		vcc->stats->rx_err++;
	}
	else {
		length = ATM_CELL_SIZE-1; /* no HEC */
	}
	if (!length) skb = NULL;
	else skb = vcc->peek(vcc,length,fetch);
	if (!skb) {
		discard(vcc,length >> 2);
		return 0;
	}
	skb->free = 1; /* not very useful ... */
	skb->len = length;
	skb->atm.timestamp = eni_vcc->timestamp;
	DPRINTK("got len %ld\n",length);
	if (do_rx_dma(vcc,skb,1,length >> 2,length >> 2)) return 1;
	eni_vcc->rxing++;
	return 0;
}


static int rx_aal5(struct atm_vcc *vcc)
{
	struct eni_vcc *eni_vcc;
	unsigned long descr;
	unsigned long size,eff,length;
	struct sk_buff *skb;

	EVENT("rx_aal5\n",0,0);
	DPRINTK(">rx_aal5\n");
	eni_vcc = ENI_VCC(vcc);
	descr = eni_vcc->recv[eni_vcc->descr];
	if ((descr & MID_RED_IDEN) != (MID_RED_RX_ID << MID_RED_SHIFT)) {
		rx_ident_err(vcc);
		return 1;
	}
	if (descr & (MID_RED_T | MID_RED_CRC_ERR)) {
		if (descr & MID_RED_T) {
			EVENT("empty cell (descr=0x%08lx)\n",descr,0);
			DPRINTK(DEV_LABEL "(itf %d): trashing empty cell\n",
			    vcc->dev->number);
			size = 0;
		}
		else {
			static unsigned long silence = 0;

			if (jiffies > silence) {
				printk(KERN_WARNING DEV_LABEL "(itf %d): "
				    "discarding PDU(s) with CRC error\n",
				    vcc->dev->number);
				silence = jiffies+2*HZ;
			}
			size = (descr & MID_RED_COUNT)*(ATM_CELL_PAYLOAD >> 2);
			EVENT("CRC error (descr=0x%08lx,size=%ld)\n",descr,
			    size);
		}
		eff = length = 0;
		vcc->stats->rx_err++;
	}
	else {
		size = (descr & MID_RED_COUNT)*(ATM_CELL_PAYLOAD >> 2);
		DPRINTK("size=%ld\n",size);
		length = eni_vcc->recv[(eni_vcc->descr+size-1) &
		    (eni_vcc->words-1)] & 0xffff; /* -trailer(2)+header(1) */
		if (length && length <= (size << 2)-8 && length <=
		  ATM_MAX_AAL5_PDU) eff = (length+3) >> 2;
		else {				 /* ^ trailer length (8) */
			EVENT("bad PDU (descr=0x08%lx,length=%ld)\n",descr,
			    length);
			printk(KERN_ERR DEV_LABEL "(itf %d): bad AAL5 PDU "
			    "(VCI=%d,length=%ld,size=%ld (descr 0x%lx))\n",
			    vcc->dev->number,vcc->vci,length,size << 2,descr);
			length = eff = 0;
			vcc->stats->rx_err++;
		}
	}
	if (!eff) skb = NULL;
	else {
		skb = vcc->peek(vcc,eff << 2,fetch);
		if (!skb) {
			EVENT("peek reject (eff << 2=%ld)\n",eff << 2,0);
			DPRINTK(DEV_LABEL "(itf %d): peek reject for %ld "
			    "bytes\n",vcc->dev->number,eff << 2);
		}
	}
	if (!skb) {
		discard(vcc,size);
		return 0;
	}
	skb->free = 1; /* not very useful ... */
	skb->len = length;
	DPRINTK("got len %ld\n",length);
	if (do_rx_dma(vcc,skb,1,size,eff)) return 1;
	eni_vcc->rxing++;
	return 0;
}


static inline int rx_vcc(struct atm_vcc *vcc)
{
	volatile unsigned long *vci_dsc;
	struct eni_vcc *eni_vcc;
	unsigned long tmp;

	eni_vcc = ENI_VCC(vcc);
	vci_dsc = ENI_DEV(vcc->dev)->vci+vcc->vci*4;
	EVENT("rx_vcc(1)\n",0,0);
	while (eni_vcc->descr != (tmp = (vci_dsc[1] & MID_VCI_DESCR) >>
	    MID_VCI_DESCR_SHIFT)) {
		EVENT("rx_vcc(2: host dsc=0x%lx, nic dsc=0x%lx)\n",
		    eni_vcc->descr,tmp);
		DPRINTK("CB_DESCR %ld REG_DESCR %ld\n",ENI_VCC(vcc)->descr,
		    ((vci_dsc[1] & MID_VCI_DESCR) >> MID_VCI_DESCR_SHIFT));
		if (ENI_VCC(vcc)->rx(vcc)) return 1;
	}
	/* clear IN_SERVICE flag */
	vci_dsc[0] &= ~MID_VCI_IN_SERVICE;
	/*
	 * If new data has arrived between evaluating the while condition and
	 * clearing IN_SERVICE, we wouldn't be notified until additional data
	 * follows. So we have to loop again to be sure.
	 */
	EVENT("rx_vcc(3)\n",0,0);
	while (ENI_VCC(vcc)->descr != (tmp = (vci_dsc[1] & MID_VCI_DESCR) >>
	    MID_VCI_DESCR_SHIFT)) {
		EVENT("rx_vcc(4: host dsc=0x%lx, nic dsc=0x%lx)\n",
		    eni_vcc->descr,tmp);
		DPRINTK("CB_DESCR %ld REG_DESCR %ld\n",ENI_VCC(vcc)->descr,
		    ((vci_dsc[1] & MID_VCI_DESCR) >> MID_VCI_DESCR_SHIFT));
		if (ENI_VCC(vcc)->rx(vcc)) return 1;
	}
	return 0;
}


static void poll_rx(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	struct atm_vcc *curr;

	eni_dev = ENI_DEV(dev);
	while ((curr = eni_dev->fast)) {
		EVENT("poll_rx.fast\n",0,0);
		if (rx_vcc(curr)) return;
		eni_dev->fast = ENI_VCC(curr)->next;
		ENI_VCC(curr)->next = ENI_VCC_NOS;
		ENI_VCC(curr)->servicing--;
	}
	while ((curr = eni_dev->slow)) {
		EVENT("poll_rx.slow\n",0,0);
		if (rx_vcc(curr)) return;
		eni_dev->slow = ENI_VCC(curr)->next;
		ENI_VCC(curr)->next = ENI_VCC_NOS;
		ENI_VCC(curr)->servicing--;
	}
}


static void get_service(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	struct atm_vcc *vcc;
	unsigned long vci;

	DPRINTK(">get_service\n");
	eni_dev = ENI_DEV(dev);
	while (eni_dev->reg[MID_SERV_WRITE] != eni_dev->serv_read) {
		vci = eni_dev->service[eni_dev->serv_read];
		eni_dev->serv_read = (eni_dev->serv_read+1) & (NR_SERVICE-1);
		vcc = eni_dev->rx_map[vci & 1023];
		if (!vcc) {
			printk(KERN_CRIT DEV_LABEL "(itf %d): VCI %ld not "
			    "found\n",dev->number,vci);
			continue; /* nasty but we try to go on anyway */
			/* @@@ nope, doesn't work */
		}
		EVENT("getting from service\n",0,0);
		if (ENI_VCC(vcc)->next != ENI_VCC_NOS) {
			EVENT("double service\n",0,0);
			DPRINTK("Grr, servicing VCC %ld twice\n",vci);
			continue;
		}
		ENI_VCC(vcc)->timestamp = xtime;
		ENI_VCC(vcc)->next = NULL;
		if (vcc->qos.rxtp.traffic_class == ATM_CBR) {
			if (eni_dev->fast)
				ENI_VCC(eni_dev->last_fast)->next = vcc;
			else eni_dev->fast = vcc;
			eni_dev->last_fast = vcc;
		}
		else {
			if (eni_dev->slow)
				ENI_VCC(eni_dev->last_slow)->next = vcc;
			else eni_dev->slow = vcc;
			eni_dev->last_slow = vcc;
		}
putting++;
		ENI_VCC(vcc)->servicing++;
	}
}


static void dequeue_rx(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	struct atm_vcc *vcc;
	struct sk_buff *skb;
	volatile unsigned long *vci_dsc;
	int first;

	eni_dev = ENI_DEV(dev);
	first = 1;
	while (1) {
		skb = skb_dequeue(&eni_dev->rx_queue);
		if (!skb) {
			if (first) {
				DPRINTK(DEV_LABEL "(itf %d): RX but not "
				    "rxing\n",dev->number);
				EVENT("nothing to dequeue\n",0,0);
			}
			break;
		}
		EVENT("dequeued (size=%d,pos=0x%lx)\n",skb->atm.size,
		    skb->atm.pos);
/*printk("DMA@0x%X\n",eni_dev->reg[MID_DMA_ADDR]);*/
rx_dequeued++;
		vcc = skb->atm.vcc;
		eni_vcc = ENI_VCC(vcc);
		first = 0;
		vci_dsc = eni_dev->vci+(vcc->vci << 2);
		if (!EEPMOK(eni_vcc->rx_pos,skb->atm.size,(vci_dsc[1] &
		    MID_VCI_READ) >> MID_VCI_READ_SHIFT,eni_vcc->words)) {
			EVENT("requeuing\n",0,0);
			skb_queue_head(&eni_dev->rx_queue,skb);
			break;
		}
		eni_vcc->rxing--;
		eni_vcc->rx_pos = skb->atm.pos & (eni_vcc->words-1);
		if (!skb->len) kfree_skb(skb,FREE_READ);
		else {
			EVENT("pushing (len=%d)\n",skb->len,0);
			if (vcc->aal == ATM_AAL0)
				*(unsigned long *) skb->data =
				    ntohl(*(unsigned long *) skb->data);
			vcc->push(vcc,skb);
			pushed++;
		}
		vcc->stats->rx++;
	}
	wake_up(&eni_dev->rx_wait);
}


static int open_rx_first(struct atm_vcc *vcc)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long size;

	DPRINTK("open_rx_first\n");
	eni_dev = ENI_DEV(vcc->dev);
	eni_vcc = ENI_VCC(vcc);
	eni_vcc->rx = NULL;
	if (vcc->qos.rxtp.traffic_class == ATM_NONE) return 0;
	size = vcc->qos.rxtp.max_sdu*3; /* @@@ improve this */
	eni_vcc->recv = (volatile unsigned long *) eni_alloc_mem(eni_dev,&size);
	DPRINTK("rx at 0x%p\n",eni_vcc->recv);
	eni_vcc->words = size >> 2;
	if (!eni_vcc->recv) return -ENOBUFS;
	eni_vcc->rx = vcc->aal == ATM_AAL5 ? rx_aal5 : rx_aal0;
	eni_vcc->descr = 0;
	eni_vcc->rx_pos = 0;
	eni_vcc->rxing = 0;
	eni_vcc->servicing = 0;
	eni_vcc->next = ENI_VCC_NOS;
	return 0;
}


static int open_rx_second(struct atm_vcc *vcc)
{
	volatile unsigned long *here;
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long size;
	int order;

	DPRINTK("open_rx_second\n");
	eni_dev = ENI_DEV(vcc->dev);
	eni_vcc = ENI_VCC(vcc);
	if (!eni_vcc->rx) return 0;
	/* set up VCI descriptor */
	here = eni_dev->vci+(vcc->vci << 2);
	DPRINTK("loc 0x%x\n",eni_vcc->recv-eni_dev->ram);
	size = eni_vcc->words >> 8;
	for (order = -1; size; order++) size >>= 1;
	here[1] = 0; /* descr, read = 0 */
	here[2] = 0; /* write, state, count = 0 */
	if (eni_dev->rx_map[vcc->vci])
		printk(KERN_CRIT DEV_LABEL "(itf %d): BUG - VCI %d already "
		    "in use\n",vcc->dev->number,vcc->vci);
	eni_dev->rx_map[vcc->vci] = vcc; /* now it counts */
	here[0] = ((vcc->aal != ATM_AAL5 ? MID_MODE_RAW : MID_MODE_AAL5) <<
	    MID_VCI_MODE_SHIFT) | MID_VCI_PTI_MODE |
	    (((eni_vcc->recv-eni_dev->ram) >> MID_LOC_SKIP) <<
	    MID_VCI_LOCATION_SHIFT) | (order << MID_VCI_SIZE_SHIFT);
	return 0;
}


static void close_rx(struct atm_vcc *vcc)
{
	volatile unsigned long *here;
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long flags,tmp;

	eni_vcc = ENI_VCC(vcc);
	if (!eni_vcc->rx) return;
	eni_dev = ENI_DEV(vcc->dev);
	if (vcc->vpi != ATM_VPI_UNSPEC && vcc->vci != ATM_VCI_UNSPEC) {
		here = eni_dev->vci+(vcc->vci << 2);
		/* block receiver */
		*here = (*here & ~MID_VCI_MODE) | (MID_MODE_TRASH <<
		    MID_VCI_MODE_SHIFT);
		/* wait for receiver to become idle */
		udelay(27);
		/* discard pending cell */
		*here &= ~MID_VCI_IN_SERVICE;
		/* don't accept any new ones */
		eni_dev->rx_map[vcc->vci] = NULL;
		/* wait for RX queue to drain */
		DPRINTK("eni_close: waiting for RX ...\n");
		EVENT("RX closing\n",0,0);
		save_flags(flags);
		cli();
		while (eni_vcc->rxing || eni_vcc->servicing) {
			EVENT("drain PDUs (rx %ld, serv %ld)\n",eni_vcc->rxing,
			    eni_vcc->servicing);
			printk(KERN_INFO "%d+%d RX left\n",eni_vcc->servicing,
			    eni_vcc->rxing);
			sleep_on(&eni_dev->rx_wait);
		}
		while (eni_vcc->rx_pos != (tmp = eni_dev->vci[vcc->vci*4+1] &
		    MID_VCI_READ) >> MID_VCI_READ_SHIFT) {
			EVENT("drain discard (host 0x%lx, nic 0x%lx)\n",
			    eni_vcc->rx_pos,tmp);
			printk(KERN_INFO "draining RX: host 0x%lx, nic 0x%lx\n",
			    eni_vcc->rx_pos,tmp);
			sleep_on(&eni_dev->rx_wait);
		}
		restore_flags(flags);
	}
	eni_free_mem(eni_dev,(unsigned long) eni_vcc->recv,
	    eni_vcc->words << 2);
	eni_vcc->rx = NULL;
}


static int start_rx(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;

	eni_dev = ENI_DEV(dev);
	eni_dev->rx_map = (struct atm_vcc **) get_free_page(GFP_KERNEL);
	if (!eni_dev->rx_map) {
		printk(KERN_ERR DEV_LABEL "(itf %d): couldn't get free page\n",
		    dev->number);
		free_page((unsigned long) eni_dev->free_list);
		return -ENOMEM;
	}
	memset(eni_dev->rx_map,0,PAGE_SIZE);
	eni_dev->fast = eni_dev->last_fast = NULL;
	eni_dev->slow = eni_dev->last_slow = NULL;
	eni_dev->rx_wait = NULL;
	skb_queue_head_init(&eni_dev->rx_queue);
	eni_dev->serv_read = eni_dev->reg[MID_SERV_WRITE];
	eni_dev->reg[MID_DMA_WR_RX] = 0;
	return 0;
}


/*----------------------------------- TX ------------------------------------*/


enum enq_res { enq_ok,enq_next,enq_jam };


static inline void put_dma(int chan,unsigned long *dma,int *j,
    unsigned long paddr,unsigned long size)
{
	unsigned long init;

	DPRINTK("put_dma: 0x%lx+0x%lx\n",paddr,size);
	EVENT("put_dma: 0x%lx+0x%lx\n",paddr,size);
	if (paddr & 3)
		printk(KERN_ERR "put_dma: unaligned addr (0x%lx)\n",paddr);
#if 0 /* don't complain - all cases that pass are handled somewhere else */
	if (size & 3)
		printk(KERN_ERR "put_dma: unaligned size (0x%lx)\n",size);
#endif
	size = (size+3) >> 2;
/* fixme: extra DMA descr when misaligned short PDU @@@ */
	if (paddr & 31) {
		init = 8-((paddr & 31) >> 2);
		if (init > size) init = size;
		dma[(*j)++] = MID_DT_WORD | (init << MID_DMA_COUNT_SHIFT) |
		    (chan << MID_DMA_CHAN_SHIFT);
		dma[(*j)++] = paddr;
		paddr += init << 2;
		size -= init;
	}
	if (size & ~7) {
		dma[(*j)++] = MID_DT_8W | ((size >> 3) << MID_DMA_COUNT_SHIFT)
		    | (chan << MID_DMA_CHAN_SHIFT);
		dma[(*j)++] = paddr;
		paddr += (size & ~7) << 2;
		size &= 7;
	}
	if (size) {
		dma[(*j)++] = MID_DT_WORD | (size << MID_DMA_COUNT_SHIFT) |
		    (chan << MID_DMA_CHAN_SHIFT);
		dma[(*j)++] = paddr;
	}
}


static enum enq_res do_tx(struct sk_buff *skb)
{
	struct atm_vcc *vcc;
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	struct eni_tx *tx;
	unsigned long dma_rd,dma_wr;
	unsigned long size; /* in words */
	int aal5,dma_size,i,j;

	DPRINTK(">do_tx\n");
	NULLCHECK(skb);
	EVENT("do_tx: skb=0x%p, %d bytes\n",(unsigned long) skb,skb->len);
	vcc = skb->atm.vcc;
	NULLCHECK(vcc);
	eni_dev = ENI_DEV(vcc->dev);
	NULLCHECK(eni_dev);
	eni_vcc = ENI_VCC(vcc);
	tx = eni_vcc->tx;
	NULLCHECK(tx);
	if ((unsigned long) skb->data & 3)
		printk(KERN_ERR DEV_LABEL "(itf %d): VCI %d has mis-aligned "
		    "TX data\n",vcc->dev->number,vcc->vci);
	/*
	 * Potential future IP speedup: make hard_header big enough to put
	 * segmentation descriptor directly into PDU. Saves: 4 slave writes,
	 * 1 DMA xfer & 2 DMA'ed bytes (protocol layering is for wimps :-)
	 */

	/* check space in buffer */
	if (!(aal5 = vcc->aal == ATM_AAL5))
		size = (ATM_CELL_PAYLOAD >> 2)+TX_DESCR_SIZE;
			/* cell without HEC plus segmentation header (includes
			   four-byte cell header) */
	else {
		size = skb->len+4*AAL5_TRAILER+ATM_CELL_PAYLOAD-1;
			/* add AAL5 trailer */
		size = ((size-(size % ATM_CELL_PAYLOAD)) >> 2)+TX_DESCR_SIZE;
						/* add segmentation header */
	}
	/*
	 * Can I move tx_pos by size bytes without getting closer than TX_GAP
	 * to the read pointer ? TX_GAP means to leave some space for what
	 * the manual calls "too close".
	 */
	if (!NEPMOK(tx->tx_pos,size+TX_GAP,
	    eni_dev->reg[MID_TX_RDPTR(tx->index)],tx->words)) {
		DPRINTK(DEV_LABEL "(itf %d): TX full (size %ld)\n",
		    vcc->dev->number,size);
		return enq_next;
	}
	/* check DMA */
	dma_wr = eni_dev->reg[MID_DMA_WR_TX];
	dma_rd = eni_dev->reg[MID_DMA_RD_TX];
	dma_size = 2; /* JK for descriptor and final fill */
DPRINTK("iovcnt = %d\n",skb->atm.iovcnt);
	if (!skb->atm.iovcnt) dma_size += 3;
	else dma_size += 3*skb->atm.iovcnt;
	if (dma_size > TX_DMA_BUF) {
		printk(KERN_WARNING DEV_LABEL "(itf %d): needs %d DMA entries "
		    "(got only %d)\n",vcc->dev->number,dma_size,TX_DMA_BUF);
	}
	DPRINTK("dma_wr is %ld, tx_pos is %ld\n",dma_wr,tx->tx_pos);
	if (dma_wr != dma_rd && ((dma_rd+NR_DMA_TX-dma_wr) & (NR_DMA_TX-1)) <
	     dma_size) {
		printk(KERN_WARNING DEV_LABEL "(itf %d): TX DMA full\n",
		    vcc->dev->number);
		return enq_jam;
	}
	/* prepare DMA queue entries */
	j = 0;
	eni_dev->dma[j++] = (((tx->tx_pos+TX_DESCR_SIZE) & (tx->words-1)) <<
	     MID_DMA_COUNT_SHIFT) | (tx->index << MID_DMA_CHAN_SHIFT) |
	     MID_DT_JK;
	j++;
	if (!skb->atm.iovcnt)
		if (aal5)
			put_dma(tx->index,eni_dev->dma,&j,
			    (unsigned long) skb->data,skb->len);
		else put_dma(tx->index,eni_dev->dma,&j,
			    (unsigned long) skb->data+4,skb->len-4);
	else {
DPRINTK("doing direct send\n");
		for (i = 0; i < skb->atm.iovcnt; i++)
			put_dma(tx->index,eni_dev->dma,&j,(unsigned long)
			    ((struct iovec *) skb->data)[i].iov_base,
			    ((struct iovec *) skb->data)[i].iov_len);
	}
	/* JK for AAL5 trailer - AAL0 doesn't need it, but who cares ... */
	eni_dev->dma[j++] = (((tx->tx_pos+size) & (tx->words-1)) <<
	     MID_DMA_COUNT_SHIFT) | (tx->index << MID_DMA_CHAN_SHIFT) |
	     MID_DMA_END | MID_DT_JK;
	j++;
	DPRINTK("DMA at end: %d\n",j);
	/* store frame */
	tx->send[tx->tx_pos] = (MID_SEG_TX_ID << MID_SEG_ID_SHIFT) |
	    (aal5 ? MID_SEG_AAL5 : 0) | (tx->prescaler << MID_SEG_PR_SHIFT) |
	    (tx->resolution << MID_SEG_RATE_SHIFT) |
	    (size/(ATM_CELL_PAYLOAD/4));
/*printk("dsc = 0x%08lx\n",tx->send[tx->tx_pos]);*/
	tx->send[(tx->tx_pos+1) & (tx->words-1)] = (vcc->vci <<
	    MID_SEG_VCI_SHIFT) | (aal5 ? 0 : (skb->data[3] & 0xf));
	DPRINTK("size: %ld, len:%ld\n",size,skb->len);
	if (aal5)
		tx->send[(tx->tx_pos+size-AAL5_TRAILER) & (tx->words-1)] =
		    skb->len;
	j = j >> 1;
	for (i = 0; i < j; i++) {
		eni_dev->tx_dma[dma_wr*2] = eni_dev->dma[i*2];
		eni_dev->tx_dma[dma_wr*2+1] = eni_dev->dma[i*2+1];
		dma_wr = (dma_wr+1) & (NR_DMA_TX-1);
	}
	skb->atm.pos = tx->tx_pos;
	skb->atm.size = size;
	ENI_VCC(vcc)->txing += size;
	tx->tx_pos = (tx->tx_pos+size) & (tx->words-1);
	DPRINTK("dma_wr set to %ld, tx_pos is now %ld\n",dma_wr,tx->tx_pos);
	eni_dev->reg[MID_DMA_WR_TX] = dma_wr;
	skb_queue_tail(&eni_dev->tx_queue,skb);
queued++;
	return enq_ok;
}


static void poll_tx(struct atm_dev *dev)
{
	struct eni_tx *tx;
	struct sk_buff *skb;
	enum enq_res res;
	int i;

	DPRINTK(">poll_tx\n");
	for (i = NR_CHAN-1; i >= 0; i--) {
		tx = &ENI_DEV(dev)->tx[i];
		if (tx->send)
			while ((skb = skb_dequeue(&tx->backlog))) {
				res = do_tx(skb);
				if (res != enq_ok) {
					DPRINTK("re-queuing TX PDU\n");
					skb_queue_head(&tx->backlog,skb);
requeued++;
					if (res == enq_jam) return;
					else break;
				}
			}
	}
}


static void dequeue_tx(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	struct atm_vcc *vcc;
	struct sk_buff *skb;
	struct eni_tx *tx;

	NULLCHECK(dev);
	eni_dev = ENI_DEV(dev);
	NULLCHECK(eni_dev);
	while ((skb = skb_dequeue(&eni_dev->tx_queue))) {
		vcc = skb->atm.vcc;
		NULLCHECK(vcc);
		tx = ENI_VCC(vcc)->tx;
		NULLCHECK(ENI_VCC(vcc)->tx);
		DPRINTK("dequeue_tx: next 0x%lx curr 0x%lx\n",skb->atm.pos,
		    eni_dev->reg[MID_TX_DESCRSTART(tx->index)]);
		if (ENI_VCC(vcc)->txing < tx->words && skb->atm.pos ==
		    eni_dev->reg[MID_TX_DESCRSTART(tx->index)]) {
			skb_queue_head(&eni_dev->tx_queue,skb);
			break;
		}
		ENI_VCC(vcc)->txing -= skb->atm.size;
		if (vcc->pop) vcc->pop(vcc,skb);
		else dev_kfree_skb(skb,FREE_WRITE);
		vcc->stats->tx++;
		wake_up(&eni_dev->tx_wait);
dma_complete++;
	}
}


static int alloc_tx(struct eni_dev *eni_dev,int *pcr,int min,int max,int ubr)
{
	static const int pre_div[] = { 4,16,128,2048 };
	    /* 2^(((x+2)^2-(x+2))/2+1) */
	int i,pre,res;

	DPRINTK("in pcr: %d/%d%s\n",min,max,ubr ? " (ubr)" : "");
	for (i = !ubr; i < NR_CHAN; i++)
		if (!eni_dev->tx[i].send) break;
	if (i == NR_CHAN) return -EAGAIN;
	if (min)
		if (min == ATM_MAX_PCR) pre = res = 0;
		else {
			int div;

			for (pre = 0; pre < 3; pre++)
				if (TS_CLOCK/pre_div[pre]/64 <= min) break;
			div = pre_div[pre]*min;
			DPRINTK("min div %d\n",div);
			res = TS_CLOCK/div-1;
		}
	else {
		int div;

		if (max > eni_dev->tx_bw && !ubr) max = eni_dev->tx_bw;
		for (pre = 3; pre >= 0; pre--)
			if (TS_CLOCK/pre_div[pre]/64 > max) break;
		if (pre < 3) pre++; /* else fail later */
		div = pre_div[pre]*max;
		DPRINTK("max div %d\n",div);
		res = (TS_CLOCK+div-1)/div-1;
	}
	if (res < 0) res = 0;
	if (res > MID_SEG_MAX_RATE) res = MID_SEG_MAX_RATE;
	eni_dev->tx[i].prescaler = pre;
	eni_dev->tx[i].resolution = res;
	eni_dev->tx[i].pcr = *pcr = TS_CLOCK/pre_div[pre]/(res+1);
	DPRINTK("out pcr: %d (%d:%d) <%d,%d>\n",*pcr,pre,res,min,max);
	DPRINTK("got chan %d\n",i);
	if ((min && *pcr < min) || (max && *pcr > max)) return -EINVAL;
	if (*pcr > eni_dev->tx_bw && !ubr) return -EAGAIN;
	return i;
}


static int open_tx_first(struct atm_vcc *vcc)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long size,mem;
	int tx_ind,pcr,order;
	int unlimited;

	eni_dev = ENI_DEV(vcc->dev);
	eni_vcc = ENI_VCC(vcc);
	eni_vcc->tx = NULL;
	if (vcc->qos.txtp.traffic_class == ATM_NONE) return 0;
	eni_vcc->txing = 0;
	unlimited = vcc->qos.txtp.traffic_class == ATM_UBR &&
	    (!vcc->qos.txtp.max_pcr || vcc->qos.txtp.max_pcr == ATM_MAX_PCR ||
	    vcc->qos.txtp.max_pcr >= ATM_OC3_PCR);
	if (!unlimited) size = vcc->qos.txtp.max_sdu*3; /* @@@ improve */
	else {
		if (eni_dev->ubr) {
			eni_vcc->tx = eni_dev->ubr;
			return 0;
		}
		size = UBR_BUFFER;
		vcc->qos.txtp.min_pcr = ATM_MAX_PCR;
		vcc->qos.txtp.max_pcr = 0;
	}
	DPRINTK("get_tx\n");
	mem = eni_alloc_mem(eni_dev,&size);
	if (!mem) return -ENOBUFS;
	if ((tx_ind = alloc_tx(eni_dev,&pcr,vcc->qos.txtp.min_pcr,
	    vcc->qos.txtp.max_pcr,unlimited)) < 0) {
		eni_free_mem(eni_dev,mem,size);
		return tx_ind;
	}
	if (unlimited) eni_dev->ubr = &eni_dev->tx[tx_ind];
	else eni_dev->tx_bw -= pcr;
	if (pcr > ATM_OC3_PCR) pcr = ATM_OC3_PCR;
	vcc->qos.txtp.min_pcr = vcc->qos.txtp.max_pcr = pcr;
	eni_dev->tx[tx_ind].send = (volatile unsigned long *) mem;
	eni_dev->tx[tx_ind].words = size >> 2;
	skb_queue_head_init(&eni_dev->tx[tx_ind].backlog);
	size >>= 10;
	for (order = -1; size; order++) size >>= 1;
	eni_dev->reg[MID_TX_PLACE(tx_ind)] = (order << MID_SIZE_SHIFT) |
	    ((eni_dev->tx[tx_ind].send-eni_dev->ram) >> MID_LOC_SKIP);
	eni_dev->tx[tx_ind].tx_pos = eni_dev->reg[MID_TX_DESCRSTART(tx_ind)] &
	    MID_DESCR_START;
	eni_vcc->tx = &eni_dev->tx[tx_ind];
	return 0;
}


static int open_tx_second(struct atm_vcc *vcc)
{
	return 0; /* nothing to do */
}


static void close_tx(struct atm_vcc *vcc)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	unsigned long flags;

	eni_vcc = ENI_VCC(vcc);
	if (!eni_vcc->tx) return;
	eni_dev = ENI_DEV(vcc->dev);
	/* wait for TX queue to drain */
	DPRINTK("eni_close: waiting for TX ...\n");
	save_flags(flags);
	cli();
	while (skb_peek(&eni_vcc->tx->backlog) || eni_vcc->txing) {
		DPRINTK("%d TX left\n",eni_vcc->txing);
		sleep_on(&eni_dev->tx_wait);
	}
	/*
	 * Looping a few times in here is probably far cheaper than keeping
	 * track of TX completions all the time, so let's poll a bit ...
	 */
	while (eni_dev->reg[MID_TX_RDPTR(eni_vcc->tx->index)] !=
	    eni_dev->reg[MID_TX_DESCRSTART(eni_vcc->tx->index)])
		schedule();
	restore_flags(flags);
#if 0
	if (skb_peek(&eni_vcc->tx->backlog))
		printk(KERN_CRIT DEV_LABEL "SKBs in BACKLOG !!!\n");
#endif
	if (eni_vcc->tx != eni_dev->ubr) {
		eni_free_mem(eni_dev,(unsigned long) eni_vcc->tx->send,
		    eni_vcc->tx->words << 2);
		eni_vcc->tx->send = NULL;
		eni_dev->tx_bw += eni_vcc->tx->pcr;
	}
	eni_vcc->tx = NULL;
}


static int start_tx(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	int i;

	eni_dev = ENI_DEV(dev);
	eni_dev->lost = 0;
	eni_dev->tx_bw = ATM_OC3_PCR;
	eni_dev->tx_wait = NULL;
	eni_dev->ubr = NULL;
	skb_queue_head_init(&eni_dev->tx_queue);
	eni_dev->reg[MID_DMA_WR_TX] = 0;
	for (i = 0; i < NR_CHAN; i++) {
		eni_dev->tx[i].send = NULL;
		eni_dev->tx[i].index = i;
	}
	return 0;
}


/*--------------------------------- common ----------------------------------*/


static void foo(void)
{
printk(KERN_INFO
  "tx_complete=%d,dma_complete=%d,queued=%d,requeued=%d,sub=%d,\n"
  "backlogged=%d,rx_enqueued=%d,rx_dequeued=%d,putting=%d,pushed=%d\n",
  tx_complete,dma_complete,queued,requeued,submitted,backlogged,
  rx_enqueued,rx_dequeued,putting,pushed);
if (eni_boards) printk(KERN_INFO "loss: %ld\n",ENI_DEV(eni_boards)->lost);
}

static void misc_int(struct atm_dev *dev,unsigned long reason)
{
	struct eni_dev *eni_dev;

	DPRINTK(">misc_int\n");
	eni_dev = ENI_DEV(dev);
	if (reason & MID_STAT_OVFL) {
		EVENT("stat overflow\n",0,0);
		eni_dev->lost += eni_dev->reg[MID_STAT] & MID_OVFL_TRASH;
	}
	if (reason & MID_SUNI_INT) {
		EVENT("SUNI int\n",0,0);
		dev->phy->interrupt(dev);
		foo();
	}
	if (reason & MID_TX_IDENT_MISM) {
		printk(KERN_CRIT DEV_LABEL "(itf %d): driver error - ident "
		    "mismatch\n",dev->number);
		EVENT("---dump ends here---\n",0,0);
		printk(KERN_NOTICE "---recent events---\n");
		event_dump();
	}
	if (reason & MID_TX_DMA_OVFL) {
		printk(KERN_CRIT DEV_LABEL "(itf %d): driver error - DMA "
		    "overflow\n",dev->number);
		EVENT("---dump ends here---\n",0,0);
		printk(KERN_NOTICE "---recent events---\n");
		event_dump();
	}
}


static void eni_int(int irq,void *dev_id,struct pt_regs *regs)
{
	struct atm_dev *dev;
	struct eni_dev *eni_dev;
	unsigned long reason;

	DPRINTK(">eni_int\n");
	dev = dev_id;
	eni_dev = ENI_DEV(dev);
	while ((reason = eni_dev->reg[MID_ISA])) {
		DPRINTK(DEV_LABEL ": int 0x%lx\n",reason);
		if (reason & MID_RX_DMA_COMPLETE) {
			EVENT("INT: RX DMA complete, starting dequeue_rx\n",
			    0,0);
			dequeue_rx(dev);
			EVENT("dequeue_rx done, starting poll_rx\n",0,0);
			poll_rx(dev);
			EVENT("poll_rx done\n",0,0);
			/* poll_tx ? */
		}
		if (reason & MID_SERVICE) {
			EVENT("INT: service, starting get_service\n",0,0);
			get_service(dev);
			EVENT("get_service done, starting poll_rx\n",0,0);
			poll_rx(dev);
			EVENT("poll_rx done\n",0,0);
		}
 		if (reason & MID_TX_DMA_COMPLETE) {
			EVENT("INT: TX DMA COMPLETE\n",0,0);
			dequeue_tx(dev);
		}
		if (reason & MID_TX_COMPLETE) {
			EVENT("INT: TX COMPLETE\n",0,0);
tx_complete++;
			wake_up(&eni_dev->tx_wait);
			poll_tx(dev);
			/* poll_rx ? */
		}
		if (reason & (MID_STAT_OVFL | MID_SUNI_INT | MID_TX_IDENT_MISM
		    | MID_TX_DMA_OVFL)) {
			EVENT("misc interrupt\n",0,0);
			misc_int(dev,reason);
		}
	}
}


/*--------------------------------- entries ---------------------------------*/


static const char *media_name[] = {
    "MMF", "SMF", "MMF", "03?", /*  0- 3 */
    "UTP", "05?", "06?", "07?", /*  4- 7 */
    "TAXI","09?", "10?", "11?", /*  8-11 */
    "12?", "13?", "14?", "15?", /* 12-15 */
    "MMF", "SMF", "18?", "19?", /* 16-19 */
    "UTP", "21?", "22?", "23?", /* 20-23 */
    "24?", "25?", "26?", "27?", /* 24-27 */
    "28?", "29?", "30?", "31?"  /* 28-31 */
};


#define SET_SEPROM \
  ({ if (!error && !pci_error) { \
    pci_error = pcibios_write_config_byte(eni_dev->bus,eni_dev->dev_fn, \
      PCI_TONGA_CTRL,tonga); \
    udelay(10); /* 10 usecs */ \
  } })
#define GET_SEPROM \
  ({ if (!error && !pci_error) { \
    pci_error = pcibios_read_config_byte(eni_dev->bus,eni_dev->dev_fn, \
      PCI_TONGA_CTRL,&tonga); \
    udelay(10); /* 10 usecs */ \
  } })


static int get_esi_asic(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	unsigned char tonga;
	int error,failed,pci_error;
	int address,i,j;

	eni_dev = ENI_DEV(dev);
	error = pci_error = 0;
	tonga = SEPROM_MAGIC | SEPROM_DATA | SEPROM_CLK;
	SET_SEPROM;
	for (i = 0; i < ESI_LEN && !error && !pci_error; i++) {
		/* start operation */
		tonga |= SEPROM_DATA;
		SET_SEPROM;
		tonga |= SEPROM_CLK;
		SET_SEPROM;
		tonga &= ~SEPROM_DATA;
		SET_SEPROM;
		tonga &= ~SEPROM_CLK;
		SET_SEPROM;
		/* send address */
		address = ((i+SEPROM_ESI_BASE) << 1)+1;
		for (j = 7; j >= 0; j--) {
			tonga = (address >> j) & 1 ? tonga | SEPROM_DATA :
			    tonga & ~SEPROM_DATA;
			SET_SEPROM;
			tonga |= SEPROM_CLK;
			SET_SEPROM;
			tonga &= ~SEPROM_CLK;
			SET_SEPROM;
		}
		/* get ack */
		tonga |= SEPROM_DATA;
		SET_SEPROM;
		tonga |= SEPROM_CLK;
		SET_SEPROM;
		GET_SEPROM;
		failed = tonga & SEPROM_DATA;
		tonga &= ~SEPROM_CLK;
		SET_SEPROM;
		tonga |= SEPROM_DATA;
		SET_SEPROM;
		if (failed) error = -EIO;
		else {
			dev->esi[i] = 0;
			for (j = 7; j >= 0; j--) {
				dev->esi[i] <<= 1;
				tonga |= SEPROM_DATA;
				SET_SEPROM;
				tonga |= SEPROM_CLK;
				SET_SEPROM;
				GET_SEPROM;
				if (tonga & SEPROM_DATA) dev->esi[i] |= 1;
				tonga &= ~SEPROM_CLK;
				SET_SEPROM;
				tonga |= SEPROM_DATA;
				SET_SEPROM;
			}
			/* get ack */
			tonga |= SEPROM_DATA;
			SET_SEPROM;
			tonga |= SEPROM_CLK;
			SET_SEPROM;
			GET_SEPROM;
			if (!(tonga & SEPROM_DATA)) error = -EIO;
			tonga &= ~SEPROM_CLK;
			SET_SEPROM;
			tonga |= SEPROM_DATA;
			SET_SEPROM;
		}
		/* stop operation */
		tonga &= ~SEPROM_DATA;
		SET_SEPROM;
		tonga |= SEPROM_CLK;
		SET_SEPROM;
		tonga |= SEPROM_DATA;
		SET_SEPROM;
	}
	if (pci_error) {
		printk(KERN_ERR DEV_LABEL "(itf %d): error reading ESI (%s)\n",
		    dev->number,pcibios_strerror(pci_error));
		error = -EIO;
	}
	return error;
}


#undef SET_SEPROM
#undef GET_SEPROM


static int get_esi_fpga(struct atm_dev *dev,unsigned long base)
{
	struct eni_dev *eni_dev;
	struct midway_eprom *eprom;
	int i;

	eprom = (struct midway_eprom *) (base+EPROM_SIZE-sizeof(struct
	    midway_eprom));
	eni_dev = ENI_DEV(dev);
	for (i = 0; i < ESI_LEN; i++)
		dev->esi[i] = eprom->mac[(i & ~3) | (3-(i & 3))];
	return 0;
}


static int eni_init(struct atm_dev *dev)
{
	struct midway_eprom *eprom;
	struct eni_dev *eni_dev;
	unsigned int real_base,base;
	unsigned short command;
	unsigned char revision;
	int error,i,last;

	DPRINTK(">eni_init\n");
	dev->ci_range.vpi_bits = 0;
	dev->ci_range.vci_bits = NR_VCI_LD;
	eni_dev = ENI_DEV(dev);
	if ((error = pcibios_read_config_word(eni_dev->bus,eni_dev->dev_fn,
	    PCI_COMMAND,&command)) || (error = pcibios_read_config_dword(
	    eni_dev->bus,eni_dev->dev_fn,PCI_BASE_ADDRESS_0,&real_base)) ||
	    (error = pcibios_read_config_byte(eni_dev->bus,eni_dev->dev_fn,
	    PCI_INTERRUPT_LINE,&eni_dev->irq)) || (error =
	    pcibios_read_config_byte(eni_dev->bus,eni_dev->dev_fn,
	    PCI_REVISION_ID,&revision))) {
		printk(KERN_ERR DEV_LABEL "(itf %d): init error %s\n",
		    dev->number,pcibios_strerror(error));
		return -EINVAL;
	}
	real_base &= MEM_VALID; /* strip flags */
	if ((error = pcibios_write_config_word(eni_dev->bus,eni_dev->dev_fn,
	    PCI_COMMAND,PCI_COMMAND_MEMORY | (eni_dev->asic ?
	    PCI_COMMAND_PARITY | PCI_COMMAND_SERR : 0)))) {
		printk(KERN_ERR DEV_LABEL "(itf %d): can't enable memory (%s)"
		    "\n",dev->number,pcibios_strerror(error));
		return error;
	}
	printk(KERN_NOTICE DEV_LABEL "(itf %d): rev.%d,base=0x%x,irq=%d,",
	    dev->number,revision,real_base,eni_dev->irq);
	if (!(base = (unsigned long) vremap(real_base,MAP_MAX_SIZE))) {
		printk("\n");
		printk(KERN_ERR DEV_LABEL "(itf %d): can't set up page "
		    "mapping\n",dev->number);
		return error;
	}
	eni_dev->base_diff = real_base-base;
	/* id may not be present in ASIC Tonga boards - check this @@@ */
	if (!eni_dev->asic) {
		eprom = (struct midway_eprom *) (base+EPROM_SIZE-sizeof(struct
		    midway_eprom));
		if (eprom->magic != ENI155_MAGIC) {
			printk("\n");
			printk(KERN_ERR KERN_ERR DEV_LABEL "(itf %d): bad "
			    "magic - expected 0x%X, got 0x%lX\n",dev->number,
			    ENI155_MAGIC,eprom->magic);
			return -EINVAL;
		}
	}
	eni_dev->phy = (volatile unsigned long *) (base+PHY_BASE);
	eni_dev->reg = (volatile unsigned long *) (base+REG_BASE);
	eni_dev->ram = (volatile unsigned long *) (base+RAM_BASE);
	last = (MAP_MAX_SIZE-RAM_BASE)/4;
	for (i = last-RAM_INCREMENT; i >= 0;
	    i -= RAM_INCREMENT) {
		eni_dev->ram[i] = 0x55555555;
		if (eni_dev->ram[i] != 0x55555555) last = i;
		else {
			eni_dev->ram[i] = 0xAAAAAAAA;
			if (eni_dev->ram[i] != 0xAAAAAAAA) last = i;
			else eni_dev->ram[i] = i;
		}
	}
	for (i = 0; i < last; i += RAM_INCREMENT)
		if (eni_dev->ram[i] != i) break;
	eni_dev->mem = i << 2;
	memset((void *) eni_dev->ram,0,eni_dev->mem);
	/* TODO: should shrink allocation now */
	printk("mem=%dkB (",eni_dev->mem >> 10);
	/* TODO: check for non-SUNI, check for TAXI ? */
	if (!(eni_dev->reg[MID_RES_ID_MCON] & 0x200) != !eni_dev->asic) {
		printk(")\n");
		printk(KERN_ERR DEV_LABEL "(itf %d): ERROR - wrong id 0x%0lx\n",
		    dev->number,eni_dev->reg[MID_RES_ID_MCON]);
		return -EINVAL;
	}
	error = eni_dev->asic ? get_esi_asic(dev) : get_esi_fpga(dev,base);
	if (error) return error;
	for (i = 0; i < ESI_LEN; i++)
		printk("%s%02X",i ? "-" : "",dev->esi[i]);
	printk(")\n");
	printk(KERN_NOTICE DEV_LABEL "(itf %d): %s,%s\n",dev->number,
	    eni_dev->reg[MID_RES_ID_MCON] & 0x200 ? "ASIC" : "FPGA",
	    media_name[eni_dev->reg[MID_RES_ID_MCON] & DAUGTHER_ID]);
	return suni_init(dev);
}


static int eni_start(struct atm_dev *dev)
{
	struct eni_dev *eni_dev;
	volatile unsigned long *buf;
	unsigned long buffer_mem;
	int error;

	DPRINTK(">eni_start\n");
	eni_dev = ENI_DEV(dev);
	if (request_irq(eni_dev->irq,&eni_int,0,DEV_LABEL,dev)) {
		printk(KERN_ERR DEV_LABEL "(itf %d): IRQ%d is already in use\n",
		    dev->number,eni_dev->irq);
		return -EAGAIN;
	}
	/* @@@ should release IRQ on error */
	if ((error = pcibios_write_config_word(eni_dev->bus,eni_dev->dev_fn,
	    PCI_COMMAND,PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER |
	    (eni_dev->asic ? PCI_COMMAND_PARITY | PCI_COMMAND_SERR : 0)))) {
		printk(KERN_ERR DEV_LABEL "(itf %d): can't enable memory+"
		    "master (%s)\n",dev->number,pcibios_strerror(error));
		return error;
	}
	if ((error = pcibios_write_config_byte(eni_dev->bus,eni_dev->dev_fn,
	    PCI_TONGA_CTRL,END_SWAP_DMA))) {
		printk(KERN_ERR DEV_LABEL "(itf %d): can't set endian swap "
		    "(%s)\n",dev->number,pcibios_strerror(error));
		return error;
	}
	/* determine addresses of internal tables */
	eni_dev->vci = eni_dev->ram;
	eni_dev->rx_dma = eni_dev->ram+NR_VCI*4;
	eni_dev->tx_dma = eni_dev->rx_dma+NR_DMA_RX*2;
	eni_dev->service = eni_dev->tx_dma+NR_DMA_TX*2;
	buf = eni_dev->service+NR_SERVICE;
	DPRINTK("vci 0x%p,rx 0x%p, tx 0x%p,srv 0x%p,buf 0x%lx\n",
	     eni_dev->vci,eni_dev->rx_dma,eni_dev->tx_dma,eni_dev->service,
	     buf);
	/* initialize memory management */
	buffer_mem = eni_dev->mem-((unsigned long) buf-
	    (unsigned long) eni_dev->ram);
	eni_dev->free_list_size = buffer_mem/MID_MIN_BUF_SIZE/2;
	eni_dev->free_list = (struct eni_free *) kmalloc(
	    sizeof(struct eni_free)*(eni_dev->free_list_size+1),GFP_KERNEL);
	if (!eni_dev->free_list) {
		printk(KERN_ERR DEV_LABEL "(itf %d): couldn't get free page\n",
		    dev->number);
		return -ENOMEM;
	}
	eni_dev->free_len = 0;
	eni_put_free(eni_dev,(unsigned long) buf,buffer_mem);
	memset((void *) eni_dev->vci,0,16*NR_VCI); /* clear VCI table */
	/*
	 * byte_addr  free (k)
	 * 0x00000000     512  VCI table
	 * 0x00004000	  496  RX DMA
	 * 0x00005000	  492  TX DMA
	 * 0x00006000	  488  service list
	 * 0x00007000	  484  buffers
	 * 0x00080000	    0  end (512kB)
	 */
	eni_dev->reg[MID_IE] = 0xffffffff;
#if 0
MID_TX_COMPLETE_0 | MID_TX_DMA_OVFL |
	    MID_TX_IDENT_MISM | MID_RX_DMA_COMPLETE | MID_TX_DMA_COMPLETE |
	    MID_SERVICE | MID_SUNI_INT | MID_STAT_OVFL; /* enable interrupts */
#endif
	error = start_tx(dev);
	if (error) return error;
	error = start_rx(dev);
	if (error) return error;
	error = dev->phy->start(dev);
	if (error) return error;
	eni_dev->reg[MID_MC_S] |= (1 << MID_INT_SEL_SHIFT) |
	    MID_TX_LOCK_MODE | MID_DMA_ENABLE | MID_TX_ENABLE | MID_RX_ENABLE;
	    /* Tonga uses SBus INTReq1 */
	(void) eni_dev->reg[MID_ISA]; /* clear Midway interrupts */
	return 0;
}


static void eni_close(struct atm_vcc *vcc)
{
	DPRINTK(">eni_close\n");
	if (!ENI_VCC(vcc)) return;
	vcc->flags &= ~ATM_VF_READY;
	close_rx(vcc);
	close_tx(vcc);
	DPRINTK("eni_close: done waiting\n");
	/* deallocate memory */
	kfree(ENI_VCC(vcc));
	ENI_VCC(vcc) = NULL;
	vcc->flags &= ~ATM_VF_ADDR;
	/*foo();*/
}


static int get_ci(struct atm_vcc *vcc,short *vpi,int *vci)
{
	struct atm_vcc *walk;

	if (*vpi == ATM_VPI_ANY) *vpi = 0;
	if (*vci == ATM_VCI_ANY) {
		for (*vci = ATM_NOT_RSV_VCI; *vci < NR_VCI; (*vci)++) {
			if (vcc->qos.rxtp.traffic_class != ATM_NONE &&
			    ENI_DEV(vcc->dev)->rx_map[*vci])
				continue;
			if (vcc->qos.txtp.traffic_class != ATM_NONE) {
				for (walk = vcc->dev->vccs; walk;
				    walk = walk->next)
					if ((walk->flags & ATM_VF_ADDR) &&
					    walk->vci == *vci &&
					    walk->qos.txtp.traffic_class !=
					    ATM_NONE)
						break;
				if (walk) continue;
			}
			break;
		}
		return *vci == NR_VCI ? -EADDRINUSE : 0;
	}
	if (*vci == ATM_VCI_UNSPEC) return 0;
	if (vcc->qos.rxtp.traffic_class != ATM_NONE &&
	    ENI_DEV(vcc->dev)->rx_map[*vci])
		return -EADDRINUSE;
	if (vcc->qos.txtp.traffic_class == ATM_NONE) return 0;
	for (walk = vcc->dev->vccs; walk; walk = walk->next)
		if ((walk->flags & ATM_VF_ADDR) && walk->vci == *vci &&
		    walk->qos.txtp.traffic_class != ATM_NONE)
			return -EADDRINUSE;
	return 0;
}


static int eni_open(struct atm_vcc *vcc,short vpi,int vci)
{
	struct eni_dev *eni_dev;
	struct eni_vcc *eni_vcc;
	int error;

	DPRINTK(">eni_open\n");
	EVENT("eni_open\n",0,0);
	if (!(vcc->flags & ATM_VF_PARTIAL)) ENI_VCC(vcc) = NULL;
	eni_dev = ENI_DEV(vcc->dev);
	error = get_ci(vcc,&vpi,&vci);
	if (error) return error;
	vcc->vpi = vpi;
	vcc->vci = vci;
	if (vci != ATM_VPI_UNSPEC && vpi != ATM_VCI_UNSPEC)
		vcc->flags |= ATM_VF_ADDR;
	if (vcc->aal != ATM_AAL0 && vcc->aal != ATM_AAL5) return -EINVAL;
	DPRINTK(DEV_LABEL "(itf %d): open %d.%d\n",vcc->dev->number,vcc->vpi,
	    vcc->vci);
	if (!(vcc->flags & ATM_VF_PARTIAL)) {
		eni_vcc = kmalloc(sizeof(struct eni_vcc),GFP_KERNEL);
		if (!eni_vcc) return -ENOMEM;
		ENI_VCC(vcc) = eni_vcc;
		eni_vcc->tx = NULL; /* for eni_close after open_rx */
		if ((error = open_rx_first(vcc))) {
			eni_close(vcc);
			return error;
		}
		if ((error = open_tx_first(vcc))) {
			eni_close(vcc);
			return error;
		}
	}
	if (vci == ATM_VPI_UNSPEC || vpi == ATM_VCI_UNSPEC) return 0;
	if ((error = open_rx_second(vcc))) {
		eni_close(vcc);
		return error;
	}
	if ((error = open_tx_second(vcc))) {
		eni_close(vcc);
		return error;
	}
	vcc->flags |= ATM_VF_READY;
	/* should power down SUNI while !ref_count @@@ */
	return 0;
}


static int eni_ioctl(struct atm_dev *dev,unsigned int cmd,unsigned long arg)
{
	if (cmd == ENI_MEMDUMP) {
		dump(dev);
		return 0;
	}
	if (cmd == ATM_SETCIRANGE) {
		struct atm_cirange ci;

		memcpy_fromfs(&ci,(void *) arg,sizeof(struct atm_cirange));
		if ((ci.vpi_bits == 0 || ci.vpi_bits == ATM_CI_MAX) &&
		    (ci.vci_bits == NR_VCI_LD || ci.vpi_bits == ATM_CI_MAX))
		    return 0;
		return -EINVAL;
	}
	if (!dev->phy->ioctl) return -EINVAL;
	return dev->phy->ioctl(dev,cmd,arg);
}


static int eni_getsockopt(struct atm_vcc *vcc,int level,int optname,
    char *optval,int *optlen)
{
#ifdef CONFIG_MMU_HACKS

static const struct atm_buffconst bctx = { PAGE_SIZE,0,PAGE_SIZE,0,0,0 };
static const struct atm_buffconst bcrx = { PAGE_SIZE,0,PAGE_SIZE,0,0,0 };

#else

static const struct atm_buffconst bctx = { 4,0,4,0,0,0 };
static const struct atm_buffconst bcrx = { 4,0,4,0,0,0 };

#endif
	int error;

	if (level == SOL_AAL && (optname == SO_BCTXOPT ||
	    optname == SO_BCRXOPT)) {
		if (get_fs_long(optlen) < sizeof(struct atm_buffconst))
			return -EINVAL;
		put_fs_long(sizeof(struct atm_buffconst),optlen);
		error = verify_area(VERIFY_WRITE,optval,
		    sizeof(struct atm_buffconst));
		if (error) return error;
		memcpy_tofs(optval,optname == SO_BCTXOPT ? &bctx : &bcrx,
		    sizeof(struct atm_buffconst));
		return 0;
	}
	return -EINVAL;
}


static int eni_setsockopt(struct atm_vcc *vcc,int level,int optname,
    char *optval,int optlen)
{
	return -EINVAL;
}


static int eni_send(struct atm_vcc *vcc,struct sk_buff *skb)
{
	unsigned long flags;

	DPRINTK(">eni_send\n");
	if (!ENI_VCC(vcc)->tx) {
		dev_kfree_skb(skb,FREE_WRITE);
		return -EINVAL;
	}
	if (!skb) {
		printk(KERN_CRIT "!skb in eni_send ?\n");
		dev_kfree_skb(skb,FREE_WRITE);
		return -EINVAL;
	}
	if (vcc->aal == ATM_AAL0) {
		if (skb->len != ATM_CELL_SIZE-1) {
			dev_kfree_skb(skb,FREE_WRITE);
			return -EINVAL;
		}
		*(unsigned long *) skb->data = htonl(*(unsigned long *)
		    skb->data);
	}
submitted++;
	skb->atm.vcc = vcc;
	save_flags(flags);
	cli(); /* brute force */
	if (skb_peek(&ENI_VCC(vcc)->tx->backlog) || do_tx(skb)) {
		skb_queue_tail(&ENI_VCC(vcc)->tx->backlog,skb);
		backlogged++;
	}
	restore_flags(flags);
	return 0;
}


static int eni_sg_send(struct atm_vcc *vcc,unsigned long start,
    unsigned long size)
{
	return vcc->aal == ATM_AAL5 && !((start | size) & 3);
		/* don't tolerate misalignment */
}


static void eni_phy_put(struct atm_dev *dev,unsigned char value,
    unsigned long addr)
{
	ENI_DEV(dev)->phy[addr] = value;
}



static unsigned char eni_phy_get(struct atm_dev *dev,unsigned long addr)
{
	volatile unsigned tmp; /* force 32 bit access */

	tmp = ENI_DEV(dev)->phy[addr];
	return tmp;
}


static const struct atmdev_ops ops = {
	eni_open,
	eni_close,
	eni_ioctl,
	eni_getsockopt,
	eni_setsockopt,
	eni_send,
	eni_sg_send,
	NULL,			/* no poll */
	NULL,			/* no send_oam */
	eni_phy_put,
	eni_phy_get,
	NULL,			/* no feedback */
	NULL,			/* no change_qos */
	NULL			/* no free_rx_skb */
};


int eni_detect(void)
{
	struct atm_dev *dev;
	struct eni_dev *eni_dev;
	int devs,type,index;

	if (!pcibios_present()) {
		printk(KERN_ERR DEV_LABEL " driver but no PCI BIOS ?\n");
		return 0;
	}
	eni_dev = (struct eni_dev *) kmalloc(sizeof(struct eni_dev),
	    GFP_KERNEL);
	if (!eni_dev) return -ENOMEM;
	devs = 0;
	for (type = 0; type < 2; type++) {
		index = 0;
		while (!pcibios_find_device(PCI_VENDOR_ID_EF,type ?
		    PCI_DEVICE_ID_EF_ATM_ASIC : PCI_DEVICE_ID_EF_ATM_FPGA,
		    index,&eni_dev->bus,&eni_dev->dev_fn)) {
			dev = atm_dev_register(DEV_LABEL,&ops,0);
			if (!dev) break;
			ENI_DEV(dev) = eni_dev;
			eni_dev->asic = type;
			if (eni_init(dev) || eni_start(dev)) {
				atm_dev_deregister(dev);
				break;
			}
			eni_dev->more = eni_boards;
			eni_boards = dev;
			index++;
			devs++;
			eni_dev = (struct eni_dev *) kmalloc(sizeof(struct
			    eni_dev),GFP_KERNEL);
			if (!eni_dev) break;
		}
	}
	return devs;
}


#ifdef MODULE

int init_module(void)
{
	if (!eni_detect()) {
		printk(KERN_ERR DEV_LABEL ": no adapter found\n");
		return -ENXIO;
	}
	MOD_INC_USE_COUNT;
	return 0;
}


void cleanup_module(void)
{
	/*
	 * Well, there's no way to get rid of the driver yet, so we don't
	 * have to clean up, right ? :-)
	 */
}

#endif
