/* nicstar.h, M. Welsh (matt.welsh@cl.cam.ac.uk)
 * Definitions for IDT77201.
 *
 * Copyright (c) 1996 University of Cambridge Computer Laboratory
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * M. Welsh, 6 July 1996
 *
 *
 *   Configuration notes:
 *     Driver can be configured by defining/undefining
 *     the configuration options below.
 *     Supported VPI/VCI range can be adjusted; search for
 *     CFG_VPIVCI for more info.
 *     A 128k x 32 SRAM board might require adjusting some constants
 *     below; see IDT 77201 NICStAR User's Manual
 *
 *     If you want to use a different size for the small free buffers,
 *     be sure to set SM_BUFSZ to the actual buffer size + 4
 *
 *     I'm not convinced that the CBR support is "accurate"; if you want
 *     to find out what's wrong, feel free.
 *
 * S. Daniel, Nov. 25, 1996
 * 
 */

#ifndef _LINUX_NICSTAR_H
#define _LINUX_NICSTAR_H

#include <linux/ioctl.h>
#include <asm/param.h>

/* User definitions *********************************************************/

#define NICSTAR_DEV_MAJOR 35

/* misc configuration options */
#define NICSTAR_FASTXMIT 1 /* Lies about buffer lengths to speed xmit
			       possibly insecure (see nicstar.c) */
#define NICSTAR_CBR 1    /* Constant bit rate support */
#undef NICSTAR_RCQ   /* Raw cell queue support; just prints out raw cells */
#define NICSTAR_PARANOID 1 /* adds some extra checks */
#undef TEST_LOOPBACK       /* enables PHY-level loopback */
#undef ATM_013             /* should be removed soon; compat w/ ATM rel 0.13 */
#undef NICSTAR_LBUFCNT     /* for debugging; keeps count of length of LGBUF_Q */

/* Kernel definitions ******************************************************/

#ifdef __KERNEL__

#define NICSTAR_MAX_DEVS 5
#define NICSTAR_IO_SIZE (4*1024)

#define SM_BUFSZ 52
#define SM_BUF_DATALEN (SM_BUFSZ - 4)
#define NUM_SM_BUFS 170
#define LG_BUFSZ 16384
#define LG_BUFMSK 0x06000000  /* 16k = 0x06000000
				  8k = 0x04000000
				  4k = 0x02000000
				  2k = 0x00000000 */
#define RAWCELLSZ 64
#define NUM_LG_BUFS 32
#define MAX_SDU_BUFS 32  /* Max number of buffers in an AAL5 SDU */
#define NUM_RX_SKB 32    /* Number of SKB's to allocate for receive */
#define NUM_SCQ_TRAIL 32 /* Number of outstanding SCQ entries */
#define TST_SIZE 200     /* Size of the Transmit Schedule Table
			    This determines resolution for CBR,
			    and thus lower bound on available BW */
#define IDT_25_PCR      (25600000/270*260/8/53)
			/* Link rate:  25600000 bps
			   SONET overhead: /270*260 (9 section, 1 path)
			   bits per cell:  /8/53
			   max cell rate:  58141.16 cells/sec */

/* TST entry values */
#define TSTE_NULL            0x0             /* force NULL cell xmit */
#define TSTE_CBR             0x20000000      /* transmit CBR cell    */
#define TSTE_VBR             0x40000000      /* transmit VBR cell    */
#define TSTE_JUMP            0x60000000      /* Jump to SRAM addr    */


enum nicstar_regs {
  DR0 = 0x00,
  DR1 = 0x04,
  DR2 = 0x08,
  DR3 = 0x0c,
  CMD = 0x10,
  CFG = 0x14,
  STAT = 0x18,
  RSQB = 0x1c,
  RSQT = 0x20,
  RSQH = 0x24,
  CDC = 0x28,
  VPEC = 0x2c,
  ICC = 0x30,
  RAWCT = 0x34,
  TMR = 0x38,
  TSTB = 0x3c,
  TSQB = 0x40,
  TSQT = 0x44,
  TSQH = 0x48,
  GP = 0x4c,
  VPM = 0x50
};

/* SRAM locations */
#define NICSTAR_VBR_SCD0 (0x1e7f4)
#define NICSTAR_VBR_SCD1 (0x1e7e8)
#define NICSTAR_VBR_SCD2 (0x1e7dc)
#define NICSTAR_TST_REGION (0x1c000)  /* For 32k SRAM */
#define NICSTAR_SCD_REGION (0x1d000)  /* Arbitrary value */

typedef struct nicstar_rcte {
  u32 status;
  u32 buf_handle;
  u32 dma_addr;
  u32 crc;
} nicstar_rcte;
#define RX_CTABLE_SIZE (16 * 1024) /* 16k words */

typedef struct nicstar_fbd {
  u32 buf_handle;
  u32 dma_addr;
} nicstar_fbd;

#define RSQE_GFC_MASK 0x00004000
#define RSQE_CLP_MASK 0x00000400

typedef struct nicstar_rsqe {
  u32 vpi_vci;
  u32 buf_handle;
  u32 crc;
  u32 status;
} nicstar_rsqe;
#define RX_STATQ_ENTRIES (512)

typedef struct nicstar_tbd {
  u32 status;
  u32 buf_addr;
  u32 ctl_len;
  u32 cell_hdr;
} nicstar_tbd;

#define TBD_SIZE     (16)
#ifdef NICSTAR_FASTXMIT
/* following apply to NIC_TBD_DESC */
#define TBD_ENDPDU          0x40000000      /* 1 => last buffer of
                                                    AAL5 CS_PDU */
#define TBD_AAL0            0x00000000      /*   000 = AAL0   */
#define TBD_AAL34           0x04000000      /*   001 = AAL3/4 */
#define TBD_AAL5            0x08000000      /*   010 = AAL5   */

#endif
                                           /* for CBR, set M and N to all 0 */
#define TBD_UBR_M               0x00800000      /* M value timer count, VBR */
#define TBD_UBR_N               0x00010000      /* N value timer count, VBR */

#define SCQ_SIZE (8192)    /* Number of bytes actually allocated */
#define SCQ_ENTRIES (SCQ_SIZE / TBD_SIZE)  /* Variable-rate SCQ */ /* XXX 511 */
#define CBRSCQ_SIZE         (1024)  /* space to reserve for each */
#define CBRSCQ_ENTRIES (CBRSCQ_SIZE / TBD_SIZE) /* # entries in a CBR SCQ */
#ifdef NICSTAR_CBR
/* magic #s must be non-zero; restrict to high two bits to support
   largest possible receive table (8 VPI bits, 6 VCI bits)
   Magic #s must be non-zero to distinguish them from the timestamp
   overflow */
#define SCQFULL_MAGIC (0xc0000000)
#define CBRSCQFULL_MAGIC (0x80000000)
#define CBRSCQFULL_MAGIC_CLOSE (0x40000000)
#else
#define SCQFULL_MAGIC (0xfeed0000)
#endif
#define SCQFULL_TIMEOUT (3 * HZ)

typedef struct nicstar_scd {
  u32 base_addr;
  u32 tail_addr;
  u32 crc;
  u32 rsvd;
  struct nicstar_tbd cache_a;
  struct nicstar_tbd cache_b;
} nicstar_scd;

typedef struct nicstar_tsi {
  u32 status;
  u32 timestamp;
} nicstar_tsi;
#define TX_STATQ_ENTRIES (1024)

typedef struct nicstar_scq_shadow {
#ifndef NICSTAR_FASTXMIT
  unsigned char cell[48];
#endif
  struct sk_buff *skb;
} nicstar_scq_shadow;

typedef struct nicstar_scq {
  struct nicstar_tbd *orig;
  struct nicstar_tbd *base;
  struct nicstar_tbd *next;
  struct nicstar_tbd *last;
  struct nicstar_tbd *tail;
  struct wait_queue *scqfull_waitq;
  u32 id;
  u32 scd; /* SRAM address of paired SCD */
  int closing;
  nicstar_scq_shadow scq_shadow[SCQ_ENTRIES];
  volatile int full;
} nicstar_scq;

typedef struct nicstar_vcimap {
  int tx:1;
  int rx:1;
  struct atm_vcc *tx_vcc;
  struct atm_vcc *rx_vcc;
  struct sk_buff *rx_skb;
  /* Pointer to SCQ for channel if CBR */
#ifdef NICSTAR_CBR
  nicstar_scq *scq;
#endif
} nicstar_vcimap;

/* Valid combinations for VPI/VCI:
               4k table, no VPI, 12b VCI (set CFG_VPIVCI to 0x00000)
               4k table, 1b VPI, 11b VCI (set CFG_VPIVCI to 0x40000)
               4k table, 2b VPI, 10b VCI (set CFG_VPIVCI to 0x80000)
               4k table, 8b VPI,  4b VCI (set CFG_VPIVCI to 0xc0000)
    *** below only for boards with 128k x 32b SRAM ***
               8k table, no VPI, 13b VCI (set CFG_VPIVCI to 0x10000)
               8k table, 1b VPI, 12b VCI (set CFG_VPIVCI to 0x50000)
               8k table, 2b VPI, 11b VCI (set CFG_VPIVCI to 0x90000)
               8k table, 8b VPI,  5b VCI (set CFG_VPIVCI to 0xd0000)
              16k table, no VPI, 14b VCI (set CFG_VPIVCI to 0x20000)
              16k table, 1b VPI, 13b VCI (set CFG_VPIVCI to 0x60000)
              16k table, 2b VPI, 12b VCI (set CFG_VPIVCI to 0xa0000)
              16k table, 8b VPI,  6b VCI (set CFG_VPIVCI to 0xe0000)
 You will need to define each of the below to match the desired config
 */
#define CFG_VPIVCI 0x40000
#define NUM_VCI (2048)  /* Size of Rx conn table, indexed by VCI only */
#define NUM_VCI_BITS (11)  /* Num sig bits */
#define NUM_VPI (2)     /* Only VPI 0 currently defined */
#define NUM_VPI_BITS (1) /* Number of bits required for above */

typedef struct nicstar_buf_list {
  caddr_t buf_addr;
  struct nicstar_buf_list *next;
} nicstar_buf_list;


#if NICSTAR_RCQ
typedef struct nicstar_rcq { /* Raw cell queue */
  u32 rcq_head; /* pointer to current head of queue */
  u32 rcq_base; /* pointer to current buffer */
} nicstar_rcq;
#endif

typedef struct sk_buff *skb_ptr;

typedef struct nicstar_dev {

  int index;                         /* Device index */
  unsigned long iobase, membase;     /* I/O and memory addresses */
  unsigned char modid, revid;        /* PCI module and revision ID */
  unsigned char pci_bus, pci_devfn;  /* PCI params */
  int irq;                           /* IRQ line */
  int inuse;
  int rev_C3;                        /* True if chip is rev. C3 */

  nicstar_rsqe *rx_statq, *rx_statq_last;
  volatile nicstar_rsqe *rx_statq_next;
  caddr_t rx_statq_orig;
  nicstar_tsi *tx_statq, *tx_statq_next, *tx_statq_last;
  caddr_t tx_statq_orig;

  u32 *host_tst;
  int available_slots;   /* VBR slots in TST */
  int max_pcr;           /* Maximum PCR for the PHY */

#if NICSTAR_RCQ
  nicstar_rcq rcq;
#endif

  caddr_t sm_bufs, lg_bufs;
  struct sk_buff_head rx_skb_queue;

  nicstar_scq *scq;
  caddr_t scq_orig;

  skb_ptr rx_lg_skb[NUM_LG_BUFS];

  struct atm_dev *dev;
  nicstar_vcimap vci_map[NUM_VPI * NUM_VCI];
  unsigned char tmp_cell[48];

#ifdef NICSTAR_LBUFCNT
  int lbuf_cnt;
#endif
  
} nicstar_dev, *nicstar_devp;

#endif /* __KERNEL__ */

#endif /* _LINUX_NICSTAR_H_ */
