/* linux/mm/bigphysarea.c, M. Welsh (mdw@cs.cornell.edu)
 * Copyright (c) 1996 by Matt Welsh.
 *
 * This is a set of routines which allow you to reserve a large (?) 
 * amount of physical memory at boot-time, which can be allocated/deallocated
 * by drivers. This memory is intended to be used for devices such as 
 * video framegrabbers which need a lot of physical RAM (above the amount
 * allocated by kmalloc). This is by no means efficient or recommended;
 * to be used only in extreme circumstances.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/config.h>
#include <linux/ptrace.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/errno.h>

static int bigphysarea_pages = 0;
static unsigned char *bigphysarea_map;
caddr_t bigphysarea;

#define BIGPHYS_FREE 0
#define BIGPHYS_INUSE 1

void bigphysarea_setup(char *str, int *ints) {
  if (ints[0] != 1) {
    printk("bigphysarea_setup: Usage: bigphysarea=<number of pages>\n");
  } else {
    bigphysarea_pages = ints[1];
  }
}

unsigned long bigphysarea_init(unsigned long mem_start, unsigned long mem_end) {
  int ncp, i;

  if (bigphysarea_pages == 0) return mem_start;

  ncp = (bigphysarea_pages / PAGE_SIZE) + 1;
  bigphysarea_map = (caddr_t)((mem_start + (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1));
  bigphysarea = bigphysarea_map + (ncp * PAGE_SIZE);

  printk("bigphysarea: Allocated %d pages at 0x%lx.\n",
  		bigphysarea_pages, (unsigned long)bigphysarea);

  for (i = 0; i < bigphysarea_pages; i++) {
    bigphysarea_map[i] = BIGPHYS_FREE;
  }

  return (unsigned long)(bigphysarea + (bigphysarea_pages * PAGE_SIZE));
}

caddr_t bigphysarea_alloc(int size) {
  int i, j, n;

  n = (size / PAGE_SIZE) + 1;

  i = 0;

try_again:
  /* Get the first unallocated page */
  for (; i < bigphysarea_pages; i++) {
    if (bigphysarea_map[i] == BIGPHYS_FREE) break;
  }
  if (i == bigphysarea_pages) return NULL;

  /* Check out the region */
  for (j = i; (j < bigphysarea_pages) && (j < (i + n)); j++) {
    if (bigphysarea_map[i] != BIGPHYS_FREE) break;
  }

  if (j == (i + n)) {
    /* Got it */
    for (j = i; (j < (i + n)); j++) {
      bigphysarea_map[j] = BIGPHYS_INUSE;
    }
    return bigphysarea + (i * PAGE_SIZE);
  }

  /* Try again ... */
  i = j;
  goto try_again;

}

void bigphysarea_free(caddr_t addr, int size) {
  int i, j;
  int n = (size / PAGE_SIZE) + 1;

  i = (((unsigned int)addr - (unsigned int)bigphysarea)/PAGE_SIZE);
  for (j = 0; j < n; j++) {
    if (bigphysarea_map[i+j] != BIGPHYS_INUSE) return;
    bigphysarea_map[i+j] = BIGPHYS_FREE;
  }
}
