/* net/atm/proc.c - ATM /proc interface */

/* Written 1995-1997 by Werner Almesberger, EPFL LRC */


#include <linux/config.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/proc_fs.h>
#include <linux/errno.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/netdevice.h>
#include <linux/atmclip.h>
#include <linux/atmarp.h>
#include <linux/if_arp.h>
#include <linux/arequipa.h> /* to get aqd */
#include <asm/param.h> /* for HZ */

#include "static.h" /* ugly */
#include "signaling.h" /* to get sigd - ugly too */
#include "ipcommon.h"
#include "atmarp.h"

#ifdef CONFIG_ATM_LANE
#include "lec.h"
#include "lec_arpc.h"
extern struct device *dev_lec[MAX_LEC_ITF];
#endif

#ifdef CONFIG_AREQUIPA
void atm_push_arequipa(struct atm_vcc *vcc,struct sk_buff *skb);
#endif

/*extern void eni_proc(int i,char *buf); - not yet @@@ */
extern void atm_push_clip(struct atm_vcc *vcc,struct sk_buff *skb);


static int atm_header(ino_t ino,char *buf)
{
	switch (ino) {
		case PROC_ATM_DEVICES:
			sprintf(buf,"Itf Type    ESI/\"MAC\"addr "
			    "AAL(TX,err,RX,err,drop) ...\n");
			break;
		case PROC_ATM_ARP:
			sprintf(buf,"IPitf TypeEncp Idle IP address      "
			    "ATM address\n");
			break;
		case PROC_ATM_SVC:
			sprintf(buf,"Itf VPI VCI   State      Remote\n");
			break;
		case PROC_ATM_PVC:
			sprintf(buf,"Itf VPI VCI   AAL RX(PCR,Class) "
			    "TX(PCR,Class)\n");
			break;
#ifdef CONFIG_ATM_LANE
                case PROC_ATM_LEC:
                        sprintf(buf,"Itf  MAC          ATM destination                          Status            Flags VPI/VCI Recv VPI/VCI\n");
                        break;
#endif
#ifdef CONFIG_AREQUIPA
		case PROC_ATM_AREQUIPA:
			sprintf(buf,"Itf VPI VCI   State    Sock# Inode\n");
			break;
#endif
		default:
			return -EINVAL;
	}
	return strlen(buf);
}


static void add_stats(char *buf,const char *aal,
  const struct atm_aal_stats *stats)
{
	sprintf(strchr(buf,0),"%s ( %ld %ld %ld %ld %ld )",aal,stats->tx,
	    stats->tx_err,stats->rx,stats->rx_err,stats->rx_drop);
}


static void dev_info(const struct atm_dev *dev,char *buf)
{
	int off,i;

	off = sprintf(buf,"%3d %-8s",dev->number,dev->type);
	for (i = 0; i < ESI_LEN; i++)
		off += sprintf(buf+off,"%02x",dev->esi[i]);
	strcat(buf,"  ");
	add_stats(buf,"0",&dev->stats.aal0);
	strcat(buf,"  ");
	add_stats(buf,"5",&dev->stats.aal5);
	strcat(buf,"\n");
}


#ifdef CONFIG_ATM_ATMARP


static int svc_addr(char *buf,struct sockaddr_atmsvc *addr)
{
	static int code[] = { 1,2,10,6,1,0 };
	static int e164[] = { 1,8,4,6,1,0 };
	int *fields;
	int len,i,j,pos;

	len = 0;
	if (*addr->sas_addr.pub) {
		strcpy(buf,addr->sas_addr.pub);
		len = strlen(addr->sas_addr.pub);
		buf += len;
		if (*addr->sas_addr.pub) {
			*buf += '+';
			len++;
		}
	}
	else if (!*addr->sas_addr.prv) {
			strcpy(buf,"(none)");
			return strlen(buf);
		}
	if (*addr->sas_addr.prv) {
		len += 44;
		pos = 0;
		fields = *addr->sas_addr.prv == ATM_AFI_E164 ? e164 : code;
		for (i = 0; fields[i]; i++) {
			for (j = fields[i]; j; j--) {
				sprintf(buf,"%02X",addr->sas_addr.prv[pos++]);
				buf += 2;
			}
			if (fields[i+1]) *buf++ = '.';
		}
	}
	return len;
}


static void atmarp_info(struct device *dev,struct atmarp_entry *entry,
    char *buf)
{
	unsigned char *ip;
	int svc,off,ip_len;

	svc = !entry->vcc || entry->vcc->family == AF_ATMSVC;
	off = sprintf(buf,"%-6s%-4s%-4s%5ld ",dev->name,svc ? "SVC" : "PVC",
	    entry->encap ? "LLC" : "NULL",(jiffies-entry->last_use)/HZ);
	ip = (unsigned char *) &entry->ip;
	ip_len = sprintf(buf+off,"%d.%d.%d.%d",ip[0],ip[1],ip[2],ip[3]);
	off += ip_len;
	while (ip_len++ < 16) buf[off++] = ' ';
	if (!entry->vcc) strcpy(buf+off,"(incomplete)\n");
	else if (!svc)
			sprintf(buf+off,"%d.%d.%d\n",entry->vcc->dev->number,
			    entry->vcc->vpi,entry->vcc->vci);
		else {
			off += svc_addr(buf+off,&entry->vcc->remote);
			strcpy(buf+off,"\n");
		}
}


#endif


static void pvc_info(struct atm_vcc *vcc,char *buf)
{
	static const char *class_name[] = { "off","UBR","CBR","VBR","ABR" };
	static const char *aal_name[] = {
		"0",	"1",	"2",	"3/4",	/*  0- 3 */
		"???",	"5",	"???",	"???" };/*  4- 7 */
	int off;

	off = sprintf(buf,"%3d %3d %5d %-3s %7d %-5s %7d %-6s",
	    vcc->dev->number,vcc->vpi,vcc->vci,
	    vcc->aal >= sizeof(aal_name)/sizeof(aal_name[0]) ? "err" :
	    aal_name[vcc->aal],vcc->qos.rxtp.min_pcr,
	    class_name[vcc->qos.rxtp.traffic_class],vcc->qos.txtp.min_pcr,
	    class_name[vcc->qos.txtp.traffic_class]);
#ifdef CONFIG_ATM_CLIP
	if (vcc->push == atm_push_clip) {
		struct device *dev;

		dev = (struct device *) vcc->proto_data;
		off += sprintf(buf+off,"CLIP, Itf:%s, Encap:",dev->name);
		if (dev->hard_header_len == RFC1483LLC_LEN)
		    off += sprintf(buf+off,"LLC/SNAP");
		else if (dev->hard_header_len == 0)
		    off += sprintf(buf+off,"None");
		else off += sprintf(buf+off,"Unknown");
	}
#endif
	strcpy(buf+off,"\n");
}


static const char *vcc_state(struct atm_vcc *vcc)
{
    if (vcc->flags & ATM_VF_READY) return "CONNECTED";
    if (vcc->flags & ATM_VF_RELEASED) return "CLOSING";
    if (vcc->flags & ATM_VF_LISTEN) return "LISTEN";
    if (vcc->flags & ATM_VF_REGIS) return "INUSE";
    if (vcc->flags & ATM_VF_BOUND) return "BOUND";
    return "IDLE";
}


static void svc_info(struct atm_vcc *vcc,char *buf)
{
	char *here;
	int i;

	if (!vcc->dev) sprintf(buf,"Unassigned    ");
	else sprintf(buf,"%3d %3d %5d ",vcc->dev->number,vcc->vpi,vcc->vci);
	here = strchr(buf,0);
	here += sprintf(here,"%-10s ",vcc == sigd ? "Signaling" :
#ifdef CONFIG_ATM_ATMARP
	    vcc == atmarpd ? "ATMARPctrl" :
#endif
#ifdef CONFIG_AREQUIPA
	    vcc == aqd ? "Arequipa" :
#endif
	    vcc_state(vcc));
	here += sprintf(here,"%s%s",vcc->remote.sas_addr.pub,
	    *vcc->remote.sas_addr.pub && *vcc->remote.sas_addr.prv ? "+" : "");
	if (*vcc->remote.sas_addr.prv)
		for (i = 0; i < ATM_ESA_LEN; i++)
			here += sprintf(here,"%02x",
			    vcc->remote.sas_addr.prv[i]);
	strcat(here,"\n");
}


#ifdef CONFIG_AREQUIPA


static const char *arequipa_state(const struct atm_vcc *vcc)
{
	if (!(vcc->flags & ATM_VF_REGIS) && vcc->family != PF_ATMPVC)
		return "DOOMED";
	if (vcc->upper) return "ATTACHED";
	return "DANGLING";
}


static void arequipa_info(struct atm_vcc *vcc,char *buf)
{
	char *here;
 
	if (!vcc->dev) sprintf(buf,"Unassigned    ");
	else sprintf(buf,"%3d %3d %5d ",vcc->dev->number,vcc->vpi,vcc->vci);
        here = strchr(buf,0);
        here += sprintf(here,"%-8s ",arequipa_state(vcc));
	if (vcc->upper)
		here += sprintf(here,"%5d %ld",vcc->upper->num,
		    vcc->upper->socket && SOCK_INODE(vcc->upper->socket) ?
		    SOCK_INODE(vcc->upper->socket)->i_ino : 0);
	strcat(here,"\n");
}


#endif


#ifdef CONFIG_ATM_LANE

static char*
lec_arp_get_status_string(unsigned char status)
{
  switch(status) {
  case ESI_UNKNOWN:
    return "ESI_UNKNOWN       ";
  case ESI_ARP_PENDING:
    return "ESI_ARP_PENDING   ";
  case ESI_VC_PENDING:
    return "ESI_VC_PENDING    ";
  case ESI_FLUSH_PENDING:
    return "ESI_FLUSH_PENDING ";
  case ESI_FORWARD_DIRECT:
    return "ESI_FORWARD_DIRECT";
  default:
    return "<Unknown>         ";
  }
}

static void 
lec_info(struct lec_arp_table *entry, char *buf)
{
        int j, offset=0;
        

        for(j=0;j<ETH_ALEN;j++) {
                offset+=sprintf(buf+offset,"%2.2x",0xff&entry->mac_addr[j]);
        }
        offset+=sprintf(buf+offset, " ");
        for(j=0;j<ATM_ESA_LEN;j++) {
                offset+=sprintf(buf+offset,"%2.2x",0xff&entry->atm_addr[j]);
        }
        offset+=sprintf(buf+offset, " %s %4.4x",
                        lec_arp_get_status_string(entry->status),
                        entry->flags&0xffff);
        if (entry->vcc) {
                offset+=sprintf(buf+offset, "%3d %3d ", entry->vcc->vpi, 
                                entry->vcc->vci);                
        } else
                offset+=sprintf(buf+offset, "        ");
        if (entry->recv_vcc) {
                offset+=sprintf(buf+offset, "     %3d %3d", 
                                entry->recv_vcc->vpi, entry->recv_vcc->vci);
        }

        sprintf(buf+offset,"\n");
}

#endif


static int atm_info(ino_t ino,loff_t *pos,char *buf)
{
	switch (ino) {
		case PROC_ATM_DEVICES:
			while (*pos <= MAX_ATM_ITF)
				if (atm_dev[*pos-1].ops) break;
				else (*pos)++;
			if (*pos > MAX_ATM_ITF) return 0;
			dev_info(atm_dev+*pos-1,buf);
			break;
#ifdef CONFIG_ATM_ATMARP
		case PROC_ATM_ARP:
			{
				struct device *dev;
				struct atmarp_entry *entry;
				int count;

				count = *pos;
				for (dev = clip_devs; dev;
				    dev = PRIV(dev)->next)
					for (entry = PRIV(dev)->table; entry;
					    entry = entry->next)
						if (!--count) {
							atmarp_info(dev,entry,
							    buf);
							return strlen(buf);
						}
				return 0;
			}
			return 0;
#endif
		case PROC_ATM_SVC:
			while (*pos <= MAX_ATM_VCC)
				if (atm_vcc[*pos-1].family == PF_ATMSVC) break;
				else (*pos)++;
			if (*pos > MAX_ATM_VCC) return 0;
			svc_info(atm_vcc+*pos-1,buf);
			break;
		case PROC_ATM_PVC:
			while (*pos <= MAX_ATM_VCC)
				if (atm_vcc[*pos-1].family == PF_ATMPVC &&
				    atm_vcc[*pos-1].dev) break;
				else (*pos)++;
			if (*pos > MAX_ATM_VCC) return 0;
			pvc_info(atm_vcc+*pos-1,buf);
			break;
#ifdef CONFIG_AREQUIPA
		case PROC_ATM_AREQUIPA:
			while (*pos <= MAX_ATM_VCC)
				if ((atm_vcc[*pos-1].family == PF_ATMPVC ||
				    atm_vcc[*pos-1].family == PF_ATMSVC) &&
				    atm_vcc[*pos-1].push == atm_push_arequipa)
					break;
				else (*pos)++;
			if (*pos > MAX_ATM_VCC) return 0;
			arequipa_info(atm_vcc+*pos-1,buf);
			break;
#endif
#ifdef CONFIG_ATM_LANE

               case PROC_ATM_LEC: {
                       struct lec_priv *priv;
                       struct lec_arp_table *entry;
                       int i, count, d, e;
                       
                       count = *pos;
                       for(d=0;d<MAX_LEC_ITF;d++) {
                               if (dev_lec[d] &&
                                   (priv = (struct lec_priv *)
                                    dev_lec[d]->priv)) {
                                       for(i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                                               entry = priv->lec_arp_tables[i];
                                               for(;entry;entry=entry->next) {
                                                       if (!--count) {
                                                               e=sprintf(buf,"%s ",dev_lec[d]->name);
                                                               lec_info(entry,buf+e);
                                                               return strlen(buf);
                                                       }
                                               }
                                       }
                                       for(entry=priv->lec_arp_empty_ones;
                                           entry; entry=entry->next) {
                                               if (!--count) {
                                                       e=sprintf(buf,"%s ",
                                                                 dev_lec[d]->name);
                                                       lec_info(entry, buf+e);
                                                       return strlen(buf);
                                               }
                                       }
                                       for(entry=priv->lec_no_forward;
                                           entry; entry=entry->next) {
                                               if (!--count) {
                                                       e=sprintf(buf,"%s ",
                                                                 dev_lec[d]->name);
                                                       lec_info(entry, buf+e);
                                                       return strlen(buf);
                                               }
                                       }
                               }
                       }
                       return 0;
               }
               
#endif
		default:
			return -EINVAL;
	}
	return strlen(buf);
}


static int proc_atm_read(struct inode *inode,struct file *file,char *buf,
    int count)
{
	unsigned long page;
	int length;

	if (count < 0) return -EINVAL;
	page = get_free_page(GFP_KERNEL);
	if (!page) return -ENOMEM;
	if (file->f_pos)
		length = atm_info(inode->i_ino,&file->f_pos,(char *) page);
	else length = atm_header(inode->i_ino,(char *) page);
	if (length > count) length = -EINVAL;
	if (length >= 0) {
		memcpy_tofs(buf,(char *) page,length);
		file->f_pos++;
	}
	free_page(page);
	return length;
}


static struct file_operations proc_atm_operations = {
	NULL,			/* lseek */
	proc_atm_read,		/* read */
	NULL,			/* write */
	NULL,			/* readdir */
	NULL,			/* select */
	NULL,			/* ioctl */
	NULL,			/* mmap */
	NULL,			/* no special open code */
	NULL,			/* no special release */
	NULL			/* can't fsync */
};

struct inode_operations proc_atm_inode_operations = {
	&proc_atm_operations,	/* default ATM directory file-ops */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};


#define FILE(ino,name,len) &proc_atm, \
    (&(struct proc_dir_entry) { ino, len, name, \
    S_IFREG | S_IRUGO, 1, 0, 0, 0, &proc_atm_inode_operations, NULL })
 
 
void atm_proc_init(void)
{
	proc_register(FILE(PROC_ATM_DEVICES,"devices",7));
	proc_register(FILE(PROC_ATM_ARP,"arp",3));
	proc_register(FILE(PROC_ATM_SVC,"svc",3));
	proc_register(FILE(PROC_ATM_PVC,"pvc",3));
#ifdef CONFIG_ATM_LANE
        proc_register(FILE(PROC_ATM_LEC,"lec",3));
#endif
#ifdef CONFIG_AREQUIPA
	proc_register(FILE(PROC_ATM_AREQUIPA,"arequipa",8));
#endif
}
