/* net/atm/svc.c - ATM SVC sockets */

/* Written 1995,1996 by Werner Almesberger, EPFL LRC */


#include <linux/string.h>
#include <linux/net.h>		/* struct socket, struct net_proto,
				   struct proto_ops */
#include <linux/errno.h>	/* error codes */
#include <linux/kernel.h>	/* printk */
#include <linux/skbuff.h>
#include <linux/wait.h>
#include <linux/sched.h>	/* jiffies and HZ */
#include <linux/fcntl.h>	/* O_NONBLOCK */
#include <linux/atm.h>		/* ATM stuff */
#include <linux/atmsap.h>
#include <linux/atmsvc.h>
#include <linux/atmdev.h>

#include "static.h"
#include "common.h"		/* common for PVCs and SVCs */
#include "signaling.h"


#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


/*
 * Note: since all this is still nicely synchronized with the signaling demon,
 *       there's no need to protect sleep loops with clis. If signaling is
 *       moved into the kernel, that would change.
 */


void svc_callback(struct atm_vcc *vcc)
{
	wake_up(&vcc->sleep);
}


static int svc_create(struct socket *sock,int protocol)
{
	int error;

	error = atm_create(sock,protocol);
	if (error) return error;
	ATM_SD(sock)->callback = svc_callback;
	return 0;
}


static int svc_dup(struct socket *newsock,struct socket *oldsock)
{
	return svc_create(newsock,ATM_SD(oldsock)->aal);
}


static int svc_shutdown(struct socket *sock,int how)
{
	return -EOPNOTSUPP;
}


static void svc_disconnect(struct atm_vcc *vcc)
{
	struct sk_buff *skb;

	DPRINTK("svc_disconnect\n");
	if (vcc->flags & ATM_VF_REGIS) {
		sigd_enq(vcc,as_close,NULL,NULL,NULL);
		while (!(vcc->flags & ATM_VF_RELEASED) && sigd)
		    sleep_on(&vcc->sleep);
	}
	while ((skb = skb_dequeue(&vcc->listenq))) {
		vcc->flags &= ~ATM_VF_RELEASED;
		DPRINTK("LISTEN REL\n");
		sigd_enq(NULL,as_reject,vcc,NULL,NULL); /* @@@ should include
							       the reason */
		dev_kfree_skb(skb,FREE_WRITE);
	}
	vcc->flags &= ~(ATM_VF_REGIS | ATM_VF_RELEASED); /* may retry later */
}


static int svc_release(struct socket *sock,struct socket *peer)
{
	DPRINTK("svc_release\n");
	if (!ATM_SD(sock)) return 0;
	ATM_SD(sock)->flags &= ~ATM_VF_READY;
	atm_release_vcc(ATM_SD(sock),0);
	svc_disconnect(ATM_SD(sock));
	    /* VCC pointer is used as a reference, so we must not free it
	       (thereby subjecting it to re-use) before all pending connections
	        are closed */
	free_atm_vcc(ATM_SD(sock));
	return 0;
}


int copy_svc_addr(struct sockaddr_atmsvc *to,struct sockaddr_atmsvc *from)
{
	struct atm_blli *walk,**link,*next;
	int error,bllis;

	*to = *from;
	link = &to->sas_addr.blli;
	error = bllis = 0;
	for (walk = from->sas_addr.blli; walk; walk = (struct atm_blli *)
	    get_fs_long((unsigned long) &walk->next)) {
		if (++bllis > ATM_MAX_BLLI) {
			error = -E2BIG;
			break;
		}
		*link = kmalloc(sizeof(struct atm_blli),GFP_KERNEL);
		if (!*link) {
			error = -ENOMEM;
			break;
		}
		error = verify_area(VERIFY_READ,walk,sizeof(struct atm_blli));
		    /* kmalloc may swap out user pages */
		if (error) break;
		memcpy_fromfs(*link,walk,sizeof(struct atm_blli));
		link = &(*link)->next;
	}
	*link = NULL;
	if (!error) return 0;
	for (walk = to->sas_addr.blli; walk; walk = next) {
		next = walk->next;
		kfree(walk);
	}
	return error;
}


static int svc_bind(struct socket *sock,struct sockaddr *sockaddr,
    int sockaddr_len)
{
	struct sockaddr_atmsvc *addr;
	struct atm_vcc *vcc;
	int error;

	if (sockaddr_len != sizeof(struct sockaddr_atmsvc)) return -EINVAL;
	if (sock->state == SS_CONNECTED) return -EISCONN;
	if (sock->state != SS_UNCONNECTED) return -EINVAL;
	addr = (struct sockaddr_atmsvc *) sockaddr;
	if (addr->sas_family != AF_ATMSVC) return -EAFNOSUPPORT;
	vcc = ATM_SD(sock);
	vcc->flags &= ~ATM_VF_BOUND; /* failing rebind will kill old binding */
	/* @@@ check memory (de)allocation on rebind */
	if (!(vcc->flags & ATM_VF_HASQOS)) return -EBADFD;
	error = copy_svc_addr(&vcc->local,addr);
	if (error) return error;
	vcc->reply = WAITING;
	sigd_enq(vcc,as_bind,NULL,NULL,&vcc->local);
	while (vcc->reply == WAITING && sigd) sleep_on(&vcc->sleep);
	vcc->flags &= ~ATM_VF_REGIS; /* doesn't count */
	if (!sigd) return -EUNATCH; /* good code ? @@@ */
        if (!vcc->reply) vcc->flags |= ATM_VF_BOUND;
	return vcc->reply;
}


static int svc_connect(struct socket *sock,struct sockaddr *sockaddr,
    int sockaddr_len,int flags)
{
	struct sockaddr_atmsvc *addr;
	struct atm_vcc *vcc;
	int error;

	DPRINTK("svc_connect\n");
	if (sockaddr_len != sizeof(struct sockaddr_atmsvc)) return -EINVAL;
	if (sock->state == SS_CONNECTED) return -EISCONN;
	vcc = ATM_SD(sock);
	if (sock->state == SS_CONNECTING) {
		if (vcc->reply == WAITING) return -EALREADY;
		sock->state = SS_UNCONNECTED;
		if (vcc->reply) return vcc->reply;
	}
	else {
		if (sock->state != SS_UNCONNECTED) return -EINVAL;
		addr = (struct sockaddr_atmsvc *) sockaddr;
		if (addr->sas_family != AF_ATMSVC) return -EAFNOSUPPORT;
		if (!(vcc->flags & ATM_VF_HASQOS)) return -EBADFD;
		if (vcc->qos.txtp.traffic_class == ATM_ANYCLASS ||
		    vcc->qos.rxtp.traffic_class == ATM_ANYCLASS)
			return -EINVAL;
		if (!vcc->qos.txtp.traffic_class &&
		    !vcc->qos.rxtp.traffic_class) return -EINVAL;
		error = copy_svc_addr(&vcc->remote,addr);
		if (error) return error;
		vcc->reply = WAITING;
		sigd_enq(vcc,as_connect,NULL,NULL,&vcc->remote);
		if (flags & O_NONBLOCK) {
			sock->state = SS_CONNECTING;
			return -EINPROGRESS;
		}
		while (vcc->reply == WAITING && sigd) {
			interruptible_sleep_on(&vcc->sleep);
			if (current->signal & ~current->blocked) {
				DPRINTK("*ABORT*\n");
				/*
				 * This is tricky:
				 *   Kernel -------close------> Demon
				 *   Kernel <-----close(0)----- Demon
			         * or
				 *   Kernel -------close------> Demon
				 *   Kernel <--close(error)---- Demon
				 * or
				 *   Kernel -------close------> Demon
				 *   Kernel <------okay-------- Demon
				 *   Kernel <-----close(0)----- Demon
				 */
				vcc->flags &= ~ATM_VF_RELEASED;
				sigd_enq(vcc,as_close,NULL,NULL,NULL);
				while (vcc->reply == WAITING && sigd)
					sleep_on(&vcc->sleep);
				if (!vcc->reply)
					while (!(vcc->flags & ATM_VF_RELEASED)
					    && sigd) sleep_on(&vcc->sleep);
				vcc->flags &= ~(ATM_VF_REGIS | ATM_VF_RELEASED);
				    /* we're gone now but may connect later */
				return -EINTR;
			}
		}
		if (!sigd) return -EUNATCH;
		if (vcc->reply) return vcc->reply;
	}
/*
 * Not supported yet
 *
 * #ifndef CONFIG_SINGLE_SIGITF
 */
	vcc->qos.txtp.max_pcr = SELECT_TOP_PCR(vcc->qos.txtp);
	vcc->qos.txtp.min_pcr = 0;
/*
 * #endif
 */
	if (!(error = atm_connect(sock,vcc->itf,vcc->vpi,vcc->vci)))
		sock->state = SS_CONNECTED;
	else (void) svc_disconnect(ATM_SD(sock));
	return error;
}


static int svc_listen(struct socket *sock,int backlog)
{
	struct atm_vcc *vcc;

	DPRINTK("svc_listen\n");
	/* let server handle listen on unbound sockets */
	vcc = ATM_SD(sock);
	vcc->reply = WAITING;
	sigd_enq(vcc,as_listen,NULL,NULL,&vcc->local);
	while (vcc->reply == WAITING && sigd) sleep_on(&vcc->sleep);
	if (!sigd) return -EUNATCH;
	vcc->flags |= ATM_VF_LISTEN;
	vcc->backlog_quota = backlog > 0 ? backlog : ATM_BACKLOG_DEFAULT;
	return vcc->reply;
}


static int svc_accept(struct socket *sock,struct socket *newsock,int flags)
{
	struct sk_buff *skb;
	struct atmsvc_msg *msg;
	struct atm_vcc *old_vcc,*new_vcc;
	int error;

	DPRINTK("svc_accept\n");
	old_vcc = ATM_SD(sock);
	new_vcc = ATM_SD(newsock);
	while (1) {
		while (!(skb = skb_dequeue(&old_vcc->listenq)) && sigd) {
			if (flags & O_NONBLOCK) return 0; /* not -EAGAIN ? */
			interruptible_sleep_on(&old_vcc->sleep);
			if (current->signal & ~current->blocked)
				return -ERESTARTSYS;
		}
		if (!skb) return -EUNATCH;
		msg = (struct atmsvc_msg *) skb->data;
		new_vcc->qos = msg->qos;
		new_vcc->flags |= ATM_VF_HASQOS;
		new_vcc->remote = msg->svc;
		/* copy BLLI @@@ */
		new_vcc->remote.sas_addr.blli = NULL;
		error = atm_connect(newsock,msg->pvc.sap_addr.itf,
		    msg->pvc.sap_addr.vpi,msg->pvc.sap_addr.vci);
		dev_kfree_skb(skb,FREE_WRITE);
		old_vcc->backlog_quota++;
		if (error) {
			sigd_enq(NULL,as_reject,old_vcc,NULL,NULL);
				/* @@@ should include the reason */
			return error == -EAGAIN ? -EBUSY : error;
		}
		/* wait should be short, so we ignore the non-blocking flag */
		new_vcc->reply = WAITING;
		sigd_enq(new_vcc,as_accept,old_vcc,NULL,NULL);
		while (new_vcc->reply == WAITING && sigd)
			sleep_on(&new_vcc->sleep);
		if (!sigd) {
			atm_release_vcc(new_vcc,1);
			return -EUNATCH;
		}
		if (!new_vcc->reply) break;
		if (new_vcc->reply != -ERESTARTSYS) {
			atm_release_vcc(new_vcc,1);
			return new_vcc->reply;
		}
	}
	newsock->state = SS_CONNECTED;
	return 0;
}


static int svc_getname(struct socket *sock,struct sockaddr *sockaddr,
    int *sockaddr_len,int peer)
{
	struct sockaddr_atmsvc *addr;

	/* this will be fun ... we have: public and private
	   address, bhli and possibly a lot of bllis. Now address buffers
	   are static ... argl */
	/* The solution: use pointers to link bllis */
	/* naturally, we still need some place to put all those nasty
	   little bllis ... */
#if 0
	/* sigh ... */
	if (*sockaddr_len < sizeof(struct sockaddr_atmsvc)) return -EINVAL;
#endif
	*sockaddr_len = sizeof(struct sockaddr_atmsvc);
	addr = (struct sockaddr_atmsvc *) sockaddr;
	memcpy(addr,peer ? &ATM_SD(sock)->remote : &ATM_SD(sock)->local,
	    sizeof(struct sockaddr_atmsvc));
	addr->sas_addr.blli = NULL; /* @@@ no - copy it */
	return 0;
}


static int check_addr(struct sockaddr_atmsvc *addr)
{
	int i;

	if (addr->sas_family != AF_ATMSVC) return -EAFNOSUPPORT;
	if (!*addr->sas_addr.pub)
		if (!*addr->sas_addr.prv) return -EINVAL;
		else return 0;
	for (i = 1; i < ATM_E164_LEN+1; i++) /* make sure it's \0-terminated */
		if (!addr->sas_addr.pub[i]) return 0;
	return -EINVAL;
}


static int identical(struct sockaddr_atmsvc *a,struct sockaddr_atmsvc *b)
{
	if (*a->sas_addr.prv)
		if (memcmp(a->sas_addr.prv,b->sas_addr.prv,ATM_ESA_LEN))
			return 0;
	if (!*a->sas_addr.pub) return !*b->sas_addr.pub;
	if (!*b->sas_addr.pub) return 0;
	return !strcmp(a->sas_addr.pub,b->sas_addr.pub);
}


/*
 * Avoid modification of any list of local interfaces while reading it
 * (which may involve page faults and therefore rescheduling)
 */


static volatile int local_lock = 0;
static struct wait_queue *local_wait = NULL;


static void lock_local(void)
{
	while (local_lock) sleep_on(&local_wait);
	local_lock = 1;
}


static void unlock_local(void)
{
	local_lock = 0;
	wake_up(&local_wait);
}


static void notify_sigd(struct atm_dev *dev)
{
	struct sockaddr_atmpvc pvc;

	pvc.sap_addr.itf = dev->number;
	sigd_enq(NULL,as_itf_notify,NULL,&pvc,NULL);
}


static void reset_addr(struct atm_dev *dev)
{
	struct atm_dev_addr *this;

	lock_local();
	while (dev->local) {
		this = dev->local;
		dev->local = this->next;
		kfree(this);
	}
	unlock_local();
	notify_sigd(dev);
}


static int add_addr(struct atm_dev *dev,struct sockaddr_atmsvc *addr)
{
	struct atm_dev_addr **walk;

	lock_local();
	for (walk = &dev->local; *walk; walk = &(*walk)->next)
		if (identical(&(*walk)->addr,addr)) {
			unlock_local();
			return -EEXIST;
		}
	*walk = kmalloc(sizeof(struct atm_dev_addr),GFP_KERNEL);
	if (!*walk) {
		unlock_local();
		return -ENOMEM;
	}
	(*walk)->addr = *addr;
	(*walk)->next = NULL;
	unlock_local();
	notify_sigd(dev);
	return 0;
}


static int del_addr(struct atm_dev *dev,struct sockaddr_atmsvc *addr)
{
	struct atm_dev_addr **walk,*this;

	lock_local();
	for (walk = &dev->local; *walk; walk = &(*walk)->next)
		if (identical(&(*walk)->addr,addr)) break;
	if (!*walk) {
		unlock_local();
		return -ENOENT;
	}
	this = *walk;
	*walk = this->next;
	kfree(this);
	unlock_local();
	notify_sigd(dev);
	return 0;
}


static int get_addr(struct atm_dev *dev,struct sockaddr_atmsvc *buf,int size)
{
	struct atm_dev_addr *walk;
	int total;

	lock_local();
	total = 0;
	for (walk = dev->local; walk; walk = walk->next) {
		total += sizeof(struct sockaddr_atmsvc);
		if (total > size) {
			unlock_local();
			return -E2BIG;
		}
		memcpy_tofs(buf,&walk->addr,sizeof(struct sockaddr_atmsvc));
		buf++;
	}
	unlock_local();
	return total;
}


static int svc_ioctl(struct socket *sock,unsigned int cmd,unsigned long arg)
{
	struct atm_dev *dev;
	void *buf;
	int rsize,wsize,len;
	int error;

	rsize = wsize = 0;
	switch (cmd) {
		case ATM_ADDADDR:
		case ATM_DELADDR:
			rsize = sizeof(struct sockaddr_atmsvc);
		case ATM_RSTADDR:
			if (!suser()) return -EPERM;
			break;
		case ATM_GETADDR:
			wsize = -1;
			break;
		default:
			return atm_ioctl(sock,cmd,arg);
	}
	error = verify_area(VERIFY_READ,(void *) arg,
	    sizeof(struct atmif_sioc));
	if (error) return error;
	if (wsize) {
		error = verify_area(VERIFY_WRITE,(void *) arg,
		    sizeof(struct atmif_sioc));
		if (error) return error;
	}
	if (!(dev = atm_find_dev(get_fs_long(&((struct atmif_sioc *) arg)->
	    number)))) return -ENODEV;
	len = get_fs_long(&((struct atmif_sioc *) arg)->length);
	buf = (void *) get_fs_long(&((struct atmif_sioc *) arg)->arg);
	if (!buf && (rsize || wsize)) return -EINVAL;
	if (rsize > 0) {
		if (len != rsize) return -EINVAL;
		error = verify_area(VERIFY_READ,buf,len);
		if (error) return error;
	}
	if (wsize > 0) {
		if (len != wsize) return -EINVAL;
		error = verify_area(VERIFY_WRITE,buf,len);
		if (error) return error;
		put_fs_long(wsize,&((struct atmif_sioc *) arg)->length);
	}
	switch (cmd) {
		case ATM_RSTADDR:
			reset_addr(dev);
			return 0;
		case ATM_ADDADDR:
		case ATM_DELADDR:
			{
				struct sockaddr_atmsvc addr;

				memcpy_fromfs(&addr,buf,sizeof(addr));
				error = check_addr(&addr);
				if (error) return error;
				if (cmd == ATM_ADDADDR)
					return add_addr(dev,&addr);
				else return del_addr(dev,&addr);
			}
		case ATM_GETADDR:
			error = get_addr(dev,buf,len);
			if (error < 0) return error;
			put_fs_long(error,
				&((struct atmif_sioc *) arg)->length);
			return error;
	}
	return -EINVAL; /* actually, we're in trouble if we end up here ... */
}


static int svc_setsockopt(struct socket *sock,int level,int optname,
    char *optval,int optlen)
{
	int error;

	error = verify_area(VERIFY_READ,optval,optlen);
	if (error) return -EINVAL;
	/* stuff for SVCs */
	return atm_setsockopt(sock,level,optname,optval,optlen);
}


static int svc_getsockopt(struct socket *sock,int level,int optname,
    char *optval,int *optlen)
{
	/* stuff for SVCs */
	return atm_getsockopt(sock,level,optname,optval,optlen);
}


static struct proto_ops svc_proto_ops = {
	PF_ATMSVC,
	svc_create,
	svc_dup,
	svc_release,
	svc_bind,
	svc_connect,
	NULL,			/* no socketpair */
	svc_accept,
	svc_getname,
	atm_select,
	svc_ioctl,
	svc_listen,
	svc_shutdown,
	svc_setsockopt,
	svc_getsockopt,
	NULL,			/* no fcntl */
	atm_sendmsg,
	atm_recvmsg
};


/*
 *	Initialize the ATM SVC protocol family
 */

void atmsvc_proto_init(struct net_proto *pro)
{
	if (sock_register(svc_proto_ops.family,&svc_proto_ops) < 0) {
		printk(KERN_ERR "ATMSVC: can't register");
		return;
	}
}
