/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5FSprivate_H
#define H5FSprivate_H

#include "H5Fprivate.h"  
#include "H5FLprivate.h" 

#define H5FS_CLS_GHOST_OBJ                                                                                   \
    0x01 
#define H5FS_CLS_SEPAR_OBJ                                                                                   \
    0x02 
#define H5FS_CLS_MERGE_SYM                                                                                   \
    0x04 
#define H5FS_CLS_ADJUST_OK                                                                                   \
    0x08 

#define H5FS_ADD_DESERIALIZING                                                                               \
    0x01 
#define H5FS_ADD_RETURNED_SPACE                                                                              \
    0x02 
#define H5FS_ADD_SKIP_VALID                                                                                  \
    0x04 

#define H5FS_PAGE_END_NO_ADD                                                                                 \
    0x08 

#define H5FS_DESERIALIZE_NO_ADD                                                                              \
    0x01 

typedef struct H5FS_t H5FS_t;

typedef struct H5FS_section_info_t H5FS_section_info_t;

typedef struct H5FS_section_class_t {
    
    const unsigned type;        
    size_t         serial_size; 
    unsigned       flags;       
    void          *cls_private; 

    
    herr_t (*init_cls)(struct H5FS_section_class_t *,
                       void *);                        
    herr_t (*term_cls)(struct H5FS_section_class_t *); 

    
    herr_t (*add)(H5FS_section_info_t **, unsigned *,
                  void *); 
    herr_t (*serialize)(const struct H5FS_section_class_t *, const H5FS_section_info_t *,
                        uint8_t *); 
    H5FS_section_info_t *(*deserialize)(
        const struct H5FS_section_class_t *, const uint8_t *, haddr_t, hsize_t,
        unsigned *); 
    htri_t (*can_merge)(const H5FS_section_info_t *, const H5FS_section_info_t *,
                        void *); 
    herr_t (*merge)(H5FS_section_info_t **, H5FS_section_info_t *, void *); 
    htri_t (*can_shrink)(const H5FS_section_info_t *,
                         void *);                     
    herr_t (*shrink)(H5FS_section_info_t **, void *); 
    herr_t (*free)(H5FS_section_info_t *);            
    herr_t (*valid)(const struct H5FS_section_class_t *,
                    const H5FS_section_info_t *); 
    H5FS_section_info_t *(*split)(H5FS_section_info_t *, hsize_t); 
    herr_t (*debug)(const H5FS_section_info_t *, FILE *, int,
                    int); 
} H5FS_section_class_t;

typedef enum H5FS_section_state_t {
    H5FS_SECT_LIVE,      
    H5FS_SECT_SERIALIZED 
} H5FS_section_state_t;

struct H5FS_section_info_t {
    haddr_t              addr;  
    hsize_t              size;  
    unsigned             type;  
    H5FS_section_state_t state; 
};

typedef enum H5FS_client_t {
    H5FS_CLIENT_FHEAP_ID = 0, 
    H5FS_CLIENT_FILE_ID,      
    H5FS_NUM_CLIENT_ID        
} H5FS_client_t;

typedef struct H5FS_create_t {
    H5FS_client_t client;         
    unsigned      shrink_percent; 
    unsigned      expand_percent; 
    unsigned      max_sect_addr;  
    hsize_t       max_sect_size;  
} H5FS_create_t;

typedef struct H5FS_stat_t {
    hsize_t tot_space;         
    hsize_t tot_sect_count;    
    hsize_t serial_sect_count; 
    hsize_t ghost_sect_count;  
    haddr_t addr;              
    hsize_t hdr_size;          
    haddr_t sect_addr;         
    hsize_t alloc_sect_size;   
    hsize_t sect_size;         
} H5FS_stat_t;

typedef herr_t (*H5FS_operator_t)(H5FS_section_info_t *sect, void *operator_data );

H5FL_SEQ_EXTERN(H5FS_section_class_t);

H5_DLL herr_t H5FS_init(void);

H5_DLL H5FS_t *H5FS_create(H5F_t *f, haddr_t *fs_addr, const H5FS_create_t *fs_create, uint16_t nclasses,
                           const H5FS_section_class_t *classes[], void *cls_init_udata, hsize_t alignment,
                           hsize_t threshold);
H5_DLL H5FS_t *H5FS_open(H5F_t *f, haddr_t fs_addr, uint16_t nclasses, const H5FS_section_class_t *classes[],
                         void *cls_init_udata, hsize_t alignment, hsize_t threshold);
H5_DLL herr_t  H5FS_size(const H5FS_t *fspace, hsize_t *meta_size);
H5_DLL herr_t  H5FS_delete(H5F_t *f, haddr_t fs_addr);
H5_DLL herr_t  H5FS_close(H5F_t *f, H5FS_t *fspace);
H5_DLL herr_t  H5FS_alloc_hdr(H5F_t *f, H5FS_t *fspace, haddr_t *fs_addr);
H5_DLL herr_t  H5FS_alloc_sect(H5F_t *f, H5FS_t *fspace);
H5_DLL herr_t  H5FS_free(H5F_t *f, H5FS_t *fspace, bool free_file_space);

H5_DLL herr_t H5FS_sect_add(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *node, unsigned flags,
                            void *op_data);
H5_DLL htri_t H5FS_sect_try_merge(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect, unsigned flags,
                                  void *op_data);
H5_DLL htri_t H5FS_sect_try_extend(H5F_t *f, H5FS_t *fspace, haddr_t addr, hsize_t size,
                                   hsize_t extra_requested, unsigned flags, void *op_data);
H5_DLL herr_t H5FS_sect_remove(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *node);
H5_DLL htri_t H5FS_sect_find(H5F_t *f, H5FS_t *fspace, hsize_t request, H5FS_section_info_t **node);
H5_DLL herr_t H5FS_sect_iterate(H5F_t *f, H5FS_t *fspace, H5FS_operator_t op, void *op_data);
H5_DLL herr_t H5FS_sect_stats(const H5FS_t *fspace, hsize_t *tot_space, hsize_t *nsects);
H5_DLL herr_t H5FS_sect_change_class(H5F_t *f, H5FS_t *fspace, H5FS_section_info_t *sect, uint16_t new_class);
H5_DLL htri_t H5FS_sect_try_shrink_eoa(H5F_t *f, H5FS_t *fspace, void *op_data);

H5_DLL herr_t H5FS_stat_info(const H5F_t *f, const H5FS_t *frsp, H5FS_stat_t *stats);
H5_DLL herr_t H5FS_get_sect_count(const H5FS_t *frsp, hsize_t *tot_sect_count);

H5_DLL herr_t H5FS_vfd_alloc_hdr_and_section_info_if_needed(H5F_t *f, H5FS_t *fspace, haddr_t *fs_addr_ptr);

H5_DLL herr_t H5FS_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth);
H5_DLL herr_t H5FS_sects_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, haddr_t fs_addr,
                               haddr_t client_addr);
H5_DLL herr_t H5FS_sect_debug(const H5FS_t *fspace, const H5FS_section_info_t *sect, FILE *stream, int indent,
                              int fwidth);

#endif 
