% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{covdepGE}
\alias{covdepGE}
\alias{covdepGE-method}
\title{Covariate Dependent Graph Estimation}
\usage{
covdepGE(
  X,
  Z = NULL,
  hp_method = "hybrid",
  ssq = NULL,
  sbsq = NULL,
  pip = NULL,
  nssq = 5,
  nsbsq = 5,
  npip = 5,
  ssq_mult = 1.5,
  ssq_lower = 1e-05,
  snr_upper = 25,
  sbsq_lower = 1e-05,
  pip_lower = 1e-05,
  pip_upper = NULL,
  tau = NULL,
  norm = 2,
  center_X = TRUE,
  scale_Z = TRUE,
  alpha_tol = 1e-05,
  max_iter_grid = 10,
  max_iter = 100,
  edge_threshold = 0.5,
  sym_method = "mean",
  parallel = FALSE,
  num_workers = NULL,
  prog_bar = TRUE
)
}
\arguments{
\item{X}{\eqn{n \times p}{n x p} numeric matrix; data matrix. For best
results, \eqn{n} should be greater than \eqn{p}}

\item{Z}{\code{NULL} OR \eqn{n \times q}{n x q} numeric matrix; extraneous
covariates. If \code{NULL}, \code{Z} will be treated as constant for all observations,
i.e.:

\if{html}{\out{<div class="sourceCode">}}\preformatted{Z <- rep(0, nrow(X))
}\if{html}{\out{</div>}}

If \code{Z} is constant, the estimated graph will be homogeneous throughout the
data. \code{NULL} by default}

\item{hp_method}{\code{character} in \code{c("grid_search","model_average","hybrid")};
method for selecting hyperparameters from the the hyperparameter grid. The
grid will be generated as the Cartesian product of \code{ssq}, \code{sbsq}, and \code{pip}.
Fix \eqn{X_j}{Xj}, the \eqn{j}-th column of \code{X}, as the response; then, the
hyperparameters will be selected as follows:

\itemize{
\item If \code{"grid_search"}, the point in the hyperparameter grid that
maximizes the total ELBO summed across all \eqn{n} regressions will be
selected
\item If \code{"model_average"}, then all posterior quantities will be an
average of the variational estimates resulting from the model fit for each
point in the hyperparameter grid. The unnormalized averaging weights for
each of the \eqn{n} regressions are the exponentiated ELBO
\item If \code{"hybrid"}, then models will be averaged over \code{pip} as in
\code{"model_average"}, with \eqn{\sigma^2}{sigma^2} and
\eqn{\sigma_\beta^2}{sigma_beta^2} chosen for each \eqn{\pi}{pi} in \code{pip}
by maximizing the total ELBO over the grid defined by the Cartesian
product of \code{ssq} and \code{sbsq} as in \code{"grid_search"}
}

\code{"hybrid"} by default}

\item{ssq}{\code{NULL} OR numeric vector with positive entries; candidate values
of the hyperparameter \eqn{\sigma^2}{sigma^2} (prior residual variance). If
\code{NULL}, \code{ssq} will be generated for each variable \eqn{X_j}{Xj} fixed as the
response as:

\if{html}{\out{<div class="sourceCode">}}\preformatted{ssq <- seq(ssq_lower, ssq_upper, length.out = nssq)
}\if{html}{\out{</div>}}

\code{NULL} by default}

\item{sbsq}{\code{NULL} OR numeric vector with positive entries; candidate values
of the hyperparameter \eqn{\sigma_\beta^2}{sigma_beta^2} (prior slab
variance). If \code{NULL}, \code{sbsq} will be generated for each variable
\eqn{X_j}{Xj} fixed as the response as:

\if{html}{\out{<div class="sourceCode">}}\preformatted{sbsq <- seq(sbsq_lower, sbsq_upper, length.out = nsbsq)
}\if{html}{\out{</div>}}

\code{NULL} by default}

\item{pip}{\code{NULL} OR numeric vector with entries in \eqn{(0, 1)}; candidate
values of the hyperparameter \eqn{\pi}{pi} (prior inclusion probability). If
\code{NULL}, \code{pip} will be generated for each variable \eqn{X_j}{Xj} fixed as the
response as:

\if{html}{\out{<div class="sourceCode">}}\preformatted{pip <- seq(pip_lower, pi_upper, length.out = npip)
}\if{html}{\out{</div>}}

\code{NULL} by default}

\item{nssq}{positive integer; number of points to generate for \code{ssq} if
\code{ssq} is \code{NULL}. \code{5} by default}

\item{nsbsq}{positive integer; number of points to generate for \code{sbsq} if
\code{sbsq} is \code{NULL}. \code{5} by default}

\item{npip}{positive integer; number of points to generate for \code{pip} if \code{pip}
is \code{NULL}. \code{5} by default}

\item{ssq_mult}{positive numeric; if \code{ssq} is \code{NULL}, then for each variable
\eqn{X_j}{Xj} fixed as the response:

\if{html}{\out{<div class="sourceCode">}}\preformatted{ssq_upper <- ssq_mult * stats::var(X_j)
}\if{html}{\out{</div>}}

Then, \code{ssq_upper} will be the greatest value in \code{ssq} for variable
\eqn{X_j}{Xj}. \code{1.5} by default}

\item{ssq_lower}{positive numeric; if \code{ssq} is \code{NULL}, then \code{ssq_lower} will
be the least value in \code{ssq}. \code{1e-5} by default}

\item{snr_upper}{positive numeric; upper bound on the signal-to-noise ratio.
If \code{sbsq} is \code{NULL}, then for each variable \eqn{X_j}{Xj} fixed as the
response:

\if{html}{\out{<div class="sourceCode">}}\preformatted{s2_sum <- sum(apply(X, 2, stats::var))
sbsq_upper <- snr_upper / (pip_upper * s2_sum)
}\if{html}{\out{</div>}}

Then, \code{sbsq_upper} will be the greatest value in \code{sbsq}. \code{25} by default}

\item{sbsq_lower}{positive numeric; if \code{sbsq} is \code{NULL}, then \code{sbsq_lower}
will be the least value in \code{sbsq}. \code{1e-5} by default}

\item{pip_lower}{numeric in \eqn{(0, 1)}; if \code{pip} is \code{NULL}, then
\code{pip_lower} will be the least value in \code{pip}. \code{1e-5} by default}

\item{pip_upper}{\code{NULL} OR  numeric in \eqn{(0, 1)}; if \code{pip} is \code{NULL}, then
\code{pip_upper} will be the greatest value in \code{pip}. If \code{sbsq} is \code{NULL},
\code{pip_upper} will be used to calculate \code{sbsq_upper}. If \code{NULL}, \code{pip_upper}
will be calculated for each variable \eqn{X_j}{Xj} fixed as the response as:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lasso <- glmnet::cv.glmnet(X, X_j)
non0 <- sum(glmnet::coef.glmnet(lasso, s = "lambda.1se")[-1] != 0)
non0 <- min(max(non0, 1), p - 1)
pip_upper <- non0 / p
}\if{html}{\out{</div>}}

\code{NULL} by default}

\item{tau}{\code{NULL} OR positive numeric OR numeric vector of length \eqn{n}
with positive entries; bandwidth parameter. Greater values allow for more
information to be shared between observations. Allows for global or
observation-specific specification. If \code{NULL}, use 2-step KDE methodology as
described in (2) to calculate observation-specific bandwidths. \code{NULL} by
default}

\item{norm}{numeric in \eqn{[1, \infty]}{[1, Inf]}; norm to use when
calculating weights. \code{Inf} results in infinity norm. \code{2} by default}

\item{center_X}{logical; if \code{TRUE}, center \code{X} column-wise to mean \eqn{0}.
\code{TRUE} by default}

\item{scale_Z}{logical; if \code{TRUE}, center and scale \code{Z} column-wise to mean
\eqn{0}, standard deviation \eqn{1} prior to calculating the weights. \code{TRUE}
by default}

\item{alpha_tol}{positive numeric; end CAVI when the Frobenius norm of the
change in the alpha matrix is within \code{alpha_tol}. \code{1e-5} by default}

\item{max_iter_grid}{positive integer; if tolerance criteria has not been
met by \code{max_iter_grid} iterations during grid search, end CAVI. After grid
search has completed, CAVI is performed with the final hyperparameters
selected by grid search for at most \code{max_iter} iterations. Does not apply to
\code{hp_method = "model_average"}. \code{10} by default}

\item{max_iter}{positive integer; if tolerance criteria has not been met by
\code{max_iter} iterations, end CAVI. \code{100} by default}

\item{edge_threshold}{numeric in \eqn{(0, 1)}; a graph for each observation
will be constructed by including an edge between variable \eqn{i} and
variable \eqn{j} if, and only if, the \eqn{(i, j)} entry of the symmetrized
posterior inclusion probability matrix corresponding to the observation is
greater than \code{edge_threshold}. \code{0.5} by default}

\item{sym_method}{\code{character} in \code{c("mean","max","min")}; to symmetrize
the posterior inclusion probability matrix for each observation, the
\eqn{(i, j)} and \eqn{(j, i)} entries will be post-processed as \code{sym_method}
applied to the \eqn{(i, j)} and \eqn{(j, i)} entries. \code{"mean"} by default}

\item{parallel}{logical; if \code{TRUE}, hyperparameter selection and CAVI for
each of the \eqn{p} variables will be performed in parallel using \code{foreach}.
Parallel backend may be registered prior to making a call to \code{covdepGE}. If
no active parallel backend can be detected, then parallel backend will be
automatically registered using:

\if{html}{\out{<div class="sourceCode">}}\preformatted{doParallel::registerDoParallel(num_workers)
}\if{html}{\out{</div>}}

\code{FALSE} by default}

\item{num_workers}{\code{NULL} OR positive integer less than or equal to
\code{parallel::detectCores()}; argument to \code{doParallel::registerDoParallel} if
\code{parallel = TRUE} and no parallel backend is detected. If \code{NULL}, then:

\if{html}{\out{<div class="sourceCode">}}\preformatted{num_workers <- floor(parallel::detectCores() / 2)
}\if{html}{\out{</div>}}

\code{NULL} by default}

\item{prog_bar}{logical; if \code{TRUE}, then a progress bar will be displayed
denoting the number of remaining variables to fix as the response and perform
CAVI. If \code{parallel}, no progress bar will be displayed. \code{TRUE} by default}
}
\value{
Returns object of class \code{covdepGE} with the following values:

\item{graphs}{list with the following values:

\itemize{
\item \code{graphs}: list of \eqn{n} numeric matrices of dimension
\eqn{p \times p}{p x p}; the \eqn{l}-th matrix is the adjacency matrix
for the \eqn{l}-th observation
\item \code{unique_graphs}: list; the \eqn{l}-th element is a list containing
the \eqn{l}-th unique graph and the indices of the observation(s)
corresponding to this graph
\item \code{inclusion_probs_sym}: list of \eqn{n} numeric matrices of
dimension \eqn{p \times p}{p x p}; the \eqn{l}-th matrix is the
symmetrized posterior inclusion probability matrix for the \eqn{l}-th
observation
\item \code{inclusion_probs_asym}: list of \eqn{n} numeric matrices of
dimension \eqn{p \times p}{p x p}; the \eqn{l}-th matrix is the
posterior inclusion probability matrix for the \eqn{l}-th observation
prior to symmetrization
}
}

\item{variational_params}{list with the following values:

\itemize{
\item \code{alpha}: list of \eqn{p} numeric matrices of dimension
\eqn{n \times (p - 1)}{n x (p - 1)}; the \eqn{(i, j)} entry of the
\eqn{k}-th matrix is the variational approximation to the posterior
inclusion probability of the \eqn{j}-th variable in a weighted
regression with variable \eqn{k} fixed as the response, where the
weights are taken with respect to observation \eqn{i}
\item \code{mu}: list of \eqn{p} numeric matrices of dimension
\eqn{n \times (p - 1)}{n x (p - 1)}; the \eqn{(i, j)} entry of the
\eqn{k}-th matrix is the variational approximation to the posterior slab
mean for the \eqn{j}-th variable in a weighted regression with variable
\eqn{k} fixed as the response, where the weights are taken with respect
to observation \eqn{i}
\item \code{ssq_var}: list of \eqn{p} numeric
matrices of dimension \eqn{n \times (p - 1)}{n x (p - 1)}; the
\eqn{(i, j)} entry of the \eqn{k}-th matrix is the variational
approximation to the posterior slab variance for the \eqn{j}-th variable
in a weighted regression with variable \eqn{k} fixed as the response,
where the weights are taken with respect to observation \eqn{i}
}
}

\item{hyperparameters}{list of \eqn{p} lists; the \eqn{j}-th list has the
following values for variable \eqn{j} fixed as the response:

\itemize{
\item \code{grid}: matrix of candidate hyperparameter values, corresponding
ELBO, and iterations to converge
\item \code{final}: the final hyperparameters chosen by grid search and the
ELBO and iterations to converge for these hyperparameters
}
}

\item{model_details}{list with the following values:

\itemize{
\item \code{elapsed}: amount of time to fit the model
\item \code{n}: number of observations
\item \code{p}: number of variables
\item \code{ELBO}: ELBO summed across all observations and variables. If
\code{hp_method} is \code{"model_average"} or \code{"hybrid"}, this ELBO is averaged
across the hyperparameter grid using the model averaging weights for
each variable
\item \code{num_unique}: number of unique graphs
\item \code{grid_size}: number of points in the hyperparameter grid
\item \code{args}: list containing all passed arguments of length \eqn{1}
}
}

\item{weights}{list with the following values:

\itemize{
\item \code{weights}: \eqn{n\times n}{n x n} numeric matrix. The \eqn{(i, j)}
entry is the similarity weight of the \eqn{i}-th observation with
respect to the \eqn{j}-th observation using the \eqn{j}-th observation's
bandwidth
\item \code{bandwidths}: numeric vector of length \eqn{n}. The \eqn{i}-th
entry is the bandwidth for the \eqn{i}-th observation
}
}
}
\description{
Model the conditional dependence structure of \code{X} as a function
of \code{Z} as described in (1)
}
\examples{
\dontrun{
library(ggplot2)

# get the data
set.seed(12)
data <- generateData()
X <- data$X
Z <- data$Z
interval <- data$interval
prec <- data$true_precision

# get overall and within interval sample sizes
n <- nrow(X)
n1 <- sum(interval == 1)
n2 <- sum(interval == 2)
n3 <- sum(interval == 3)

# visualize the distribution of the extraneous covariate
ggplot(data.frame(Z = Z, interval = as.factor(interval))) +
  geom_histogram(aes(Z, fill = interval), color = "black", bins = n \%/\% 5)

# visualize the true precision matrices in each of the intervals

# interval 1
matViz(prec[[1]], incl_val = TRUE) +
  ggtitle(paste0("True precision matrix, interval 1, observations 1,...,", n1))

# interval 2 (varies continuously with Z)
cat("\nInterval 2, observations ", n1 + 1, ",...,", n1 + n2, sep = "")
int2_mats <- prec[interval == 2]
int2_inds <- c(5, n2 \%/\% 2, n2 - 5)
lapply(int2_inds, function(j) matViz(int2_mats[[j]], incl_val = TRUE) +
         ggtitle(paste("True precision matrix, interval 2, observation", j + n1)))

# interval 3
matViz(prec[[length(prec)]], incl_val = TRUE) +
  ggtitle(paste0("True precision matrix, interval 3, observations ",
                 n1 + n2 + 1, ",...,", n1 + n2 + n3))

# fit the model and visualize the estimated graphs
(out <- covdepGE(X, Z))
plot(out)

# visualize the posterior inclusion probabilities for variables (1, 3) and (1, 2)
inclusionCurve(out, 1, 2)
inclusionCurve(out, 1, 3)
}
}
\references{
(1) Sutanoy Dasgupta, Peng Zhao, Prasenjit Ghosh, Debdeep Pati, and Bani
Mallick. An approximate Bayesian approach to covariate-dependent graphical
modeling. pages 1–59, 2022.

(2) Sutanoy Dasgupta, Debdeep Pati, and Anuj Srivastava. A Two-Step Geometric
Framework For Density Modeling. \emph{Statistica Sinica}, 30(4):2155–2177, 2020.
}
