% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extractB.R
\name{extractB}
\alias{extractB}
\title{Extract the community matrix (B)}
\usage{
extractB(marss_res, states_names = NULL)
}
\arguments{
\item{marss_res}{MARSS object returned by \code{\link[MARSS]{MARSS}}}

\item{states_names}{a string vector containing the names of species/groups
for which interactions were estimated.}
}
\value{
A named square matrix (B) with one row (and column) per species/group
listed in \code{states_names} (which should also have been used in
\code{\link[MARSS]{MARSS}}). Row and column names are included (named after
\code{states_names}). The values are interaction strengths between the
species/groups estimated by MARSS.
}
\description{
Extract the community matrix (B) estimated by a MARSS model. The matrix is
necessary to calculate the Jacobian metrics.
}
\examples{
library(MARSS)

\donttest{
  # smaller dataset for example:
  # 3 functional groups and two insecticide concentrations besides control
  data_df <- subset(
    aquacomm_fgps,
    treat \%in\% c(0.0, 0.9, 6) &
      time >= 1 & time <= 28,
    select = c(time, treat, repli, herb, carn, detr)
  )

  # estimate z-score transformation and replace zeros with NA
  data_df[, c("herb", "carn", "detr")] <- lapply(data_df[, c("herb", "carn", "detr")],
                                                 MARSS::zscore)
  data_df[, c("herb", "carn", "detr")] <- lapply(data_df[, c("herb", "carn", "detr")],
                                                 function(x) replace(x, x == 0, NA))

  # reshape data from wide to long format
  data_z_ldf <- reshape(
    data_df,
    varying = list(c("herb", "carn", "detr")),
    v.names = "abund_z",
    timevar = "fgp",
    times = c("herb", "carn", "detr"),
    direction = "long",
    idvar = c("time", "treat", "repli")
  )

  data_z_ldf <- data_z_ldf[order(data_z_ldf$time, data_z_ldf$treat, data_z_ldf$fgp),]

  # summarize mean and sd
  data_z_summldf <- aggregate(abund_z ~ time + treat + fgp, data_z_ldf, function(x)
    c(mean = mean(x, na.rm = TRUE), sd = sd(x, na.rm = TRUE)))
  data_z_summldf <- do.call(data.frame, data_z_summldf)
  names(data_z_summldf)[4:5] <- c("abundz_mu", "abundz_sd")

  # split dataframe per functional groups
  # into list to apply the MARSS model more easily
  split_data_z <- split(data_z_summldf[, c("time", "fgp", "abundz_mu")], data_z_summldf$treat)

  reshape_to_wide <- function(df) {
    df_wide <- reshape(df,
                       idvar = "fgp",
                       timevar = "time",
                       direction = "wide")
    rownames(df_wide) <- df_wide$fgp
    df_wide <- df_wide[, -1]  # Remove the 'fgp' column
    as.matrix(df_wide)
  }

  data_z_summls <- lapply(split_data_z, reshape_to_wide)

  # fit MARSS models
  data.marssls <- list(
    MARSS(
      data_z_summls[[1]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    ),
    MARSS(
      data_z_summls[[2]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    ),
    MARSS(
      data_z_summls[[3]],
      model = list(
        B = "unconstrained",
        U = "zero",
        A = "zero",
        Z = "identity",
        Q = "diagonal and equal",
        R = matrix(0, 3, 3),
        tinitx = 1
      ),
      method = "BFGS"
    )
  )

  # identify experiments
  names(data.marssls) <- paste0("Conc. = ", c("0", "0.9", "44"), " micro g/L")

  # extract community matrices (B)
  data.Bls <- data.marssls |>
    lapply(extractB,
           states_names = c("Herbivores", "Carnivores", "Detrivores"))

  print(data.Bls)

}
}
\references{
Holmes, E. E., Ward, E. J., Scheuerell, M. D., & Wills, K. (2024). MARSS: Multivariate Autoregressive State-Space Modeling (Version 3.11.9). \doi{10.32614/CRAN.package.MARSS}

Holmes, E. E., Scheuerell, M. D., & Ward, E. J. (2024). Analysis of multivariate time-series using the MARSS package. Version 3.11.9. \doi{10.5281/zenodo.5781847}

Holmes, E. E., Ward, E. J., & Wills, K. (2012). MARSS: Multivariate autoregressive state-space models for analyzing time-series data. The R Journal, 4(1), 30. \doi{10.32614/RJ-2012-002}
}
