/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Zmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 
#include "H5Zpkg.h"      

typedef enum {
    H5Z_XFORM_ERROR,
    H5Z_XFORM_INTEGER, 
    H5Z_XFORM_FLOAT,   
    H5Z_XFORM_SYMBOL,
    H5Z_XFORM_PLUS,
    H5Z_XFORM_MINUS,
    H5Z_XFORM_MULT,
    H5Z_XFORM_DIVIDE,
    H5Z_XFORM_LPAREN,
    H5Z_XFORM_RPAREN,
    H5Z_XFORM_END
} H5Z_token_type;

typedef struct {
    unsigned int num_ptrs;
    void       **ptr_dat_val;
} H5Z_datval_ptrs;

typedef union {
    void  *dat_val;
    long   int_val;
    double float_val;
} H5Z_num_val;

typedef struct H5Z_node {
    struct H5Z_node *lchild;
    struct H5Z_node *rchild;
    H5Z_token_type   type;
    H5Z_num_val      value;
} H5Z_node;

struct H5Z_data_xform_t {
    char            *xform_exp;
    H5Z_node        *parse_root;
    H5Z_datval_ptrs *dat_val_pointers;
};

typedef struct result {
    H5Z_token_type type;
    H5Z_num_val    value;
} H5Z_result;

typedef struct {
    const char *tok_expr; 

    
    H5Z_token_type tok_type;  
    const char    *tok_begin; 
    const char    *tok_end;   

    
    H5Z_token_type tok_last_type;  
    const char    *tok_last_begin; 
    const char    *tok_last_end;   
} H5Z_token;

static H5Z_token *H5Z__get_token(H5Z_token *current);
static H5Z_node  *H5Z__parse_expression(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers);
static H5Z_node  *H5Z__parse_term(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers);
static H5Z_node  *H5Z__parse_factor(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers);
static H5Z_node  *H5Z__new_node(H5Z_token_type type);
static void       H5Z__do_op(H5Z_node *tree);
static bool       H5Z__op_is_numbs(H5Z_node *_tree);
static bool       H5Z__op_is_numbs2(H5Z_node *_tree);
static hid_t      H5Z__xform_find_type(const H5T_t *type);
static herr_t     H5Z__xform_eval_full(H5Z_node *tree, size_t array_size, hid_t array_type, H5Z_result *res);
static void       H5Z__xform_destroy_parse_tree(H5Z_node *tree);
static void      *H5Z__xform_parse(const char *expression, H5Z_datval_ptrs *dat_val_pointers);
static void      *H5Z__xform_copy_tree(H5Z_node *tree, H5Z_datval_ptrs *dat_val_pointers,
                                       H5Z_datval_ptrs *new_dat_val_pointers);
static void       H5Z__xform_reduce_tree(H5Z_node *tree);

#define H5Z_XFORM_DO_OP1(RESL, RESR, TYPE, OP, SIZE)                                                         \
    {                                                                                                        \
        size_t u;                                                                                            \
                                                                                                             \
        if (((RESL).type == H5Z_XFORM_SYMBOL) && ((RESR).type != H5Z_XFORM_SYMBOL)) {                        \
            TYPE  *p;                                                                                        \
            double tree_val;                                                                                 \
                                                                                                             \
            tree_val =                                                                                       \
                ((RESR).type == H5Z_XFORM_INTEGER ? (double)(RESR).value.int_val : (RESR).value.float_val);  \
            p = (TYPE *)(RESL).value.dat_val;                                                                \
                                                                                                             \
            for (u = 0; u < (SIZE); u++) {                                                                   \
                *p = (TYPE)((double)*p OP tree_val);                                                         \
                p++;                                                                                         \
            }                                                                                                \
        }                                                                                                    \
        else if (((RESR).type == H5Z_XFORM_SYMBOL) && ((RESL).type != H5Z_XFORM_SYMBOL)) {                   \
            TYPE  *p;                                                                                        \
            double tree_val;                                                                                 \
                                                                                                             \
                                               \
            if ((RESL).type == H5Z_XFORM_ERROR)                                                              \
                tree_val = 0;                                                                                \
            else                                                                                             \
                tree_val = ((RESL).type == H5Z_XFORM_INTEGER ? (double)(RESL).value.int_val                  \
                                                             : (RESL).value.float_val);                      \
                                                                                                             \
            p = (TYPE *)(RESR).value.dat_val;                                                                \
            for (u = 0; u < (SIZE); u++) {                                                                   \
                *p = (TYPE)(tree_val OP(double) * p);                                                        \
                p++;                                                                                         \
            }                                                                                                \
        }                                                                                                    \
        else if (((RESL).type == H5Z_XFORM_SYMBOL) && ((RESR).type == H5Z_XFORM_SYMBOL)) {                   \
            TYPE *pl = (TYPE *)(RESL).value.dat_val;                                                         \
            TYPE *pr = (TYPE *)(RESR).value.dat_val;                                                         \
                                                                                                             \
            for (u = 0; u < (SIZE); u++) {                                                                   \
                *pl = (TYPE)(*pl OP * pr);                                                                   \
                pl++;                                                                                        \
                pr++;                                                                                        \
            }                                                                                                \
        }                                                                                                    \
        else                                                                                                 \
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Unexpected type conversion operation");               \
    }

#if CHAR_MIN >= 0
#define H5Z_XFORM_TYPE_OP(RESL, RESR, TYPE, OP, SIZE)                                                        \
    {                                                                                                        \
        if ((TYPE) == H5T_NATIVE_CHAR)                                                                       \
            H5Z_XFORM_DO_OP1((RESL), (RESR), char, OP, (SIZE))                                               \
        else if ((TYPE) == H5T_NATIVE_SCHAR)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), signed char, OP, (SIZE))                                        \
        else if ((TYPE) == H5T_NATIVE_SHORT)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), short, OP, (SIZE))                                              \
        else if ((TYPE) == H5T_NATIVE_USHORT)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned short, OP, (SIZE))                                     \
        else if ((TYPE) == H5T_NATIVE_INT)                                                                   \
            H5Z_XFORM_DO_OP1((RESL), (RESR), int, OP, (SIZE))                                                \
        else if ((TYPE) == H5T_NATIVE_UINT)                                                                  \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned int, OP, (SIZE))                                       \
        else if ((TYPE) == H5T_NATIVE_LONG)                                                                  \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long, OP, (SIZE))                                               \
        else if ((TYPE) == H5T_NATIVE_ULONG)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned long, OP, (SIZE))                                      \
        else if ((TYPE) == H5T_NATIVE_LLONG)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long long, OP, (SIZE))                                          \
        else if ((TYPE) == H5T_NATIVE_ULLONG)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned long long, OP, (SIZE))                                 \
        else if ((TYPE) == H5T_NATIVE_FLOAT)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), float, OP, (SIZE))                                              \
        else if ((TYPE) == H5T_NATIVE_DOUBLE)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), double, OP, (SIZE))                                             \
        else if ((TYPE) == H5T_NATIVE_LDOUBLE)                                                               \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long double, OP, (SIZE))                                        \
    }
#else 
#define H5Z_XFORM_TYPE_OP(RESL, RESR, TYPE, OP, SIZE)                                                        \
    {                                                                                                        \
        if ((TYPE) == H5T_NATIVE_CHAR)                                                                       \
            H5Z_XFORM_DO_OP1((RESL), (RESR), char, OP, (SIZE))                                               \
        else if ((TYPE) == H5T_NATIVE_UCHAR)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned char, OP, (SIZE))                                      \
        else if ((TYPE) == H5T_NATIVE_SHORT)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), short, OP, (SIZE))                                              \
        else if ((TYPE) == H5T_NATIVE_USHORT)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned short, OP, (SIZE))                                     \
        else if ((TYPE) == H5T_NATIVE_INT)                                                                   \
            H5Z_XFORM_DO_OP1((RESL), (RESR), int, OP, (SIZE))                                                \
        else if ((TYPE) == H5T_NATIVE_UINT)                                                                  \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned int, OP, (SIZE))                                       \
        else if ((TYPE) == H5T_NATIVE_LONG)                                                                  \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long, OP, (SIZE))                                               \
        else if ((TYPE) == H5T_NATIVE_ULONG)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned long, OP, (SIZE))                                      \
        else if ((TYPE) == H5T_NATIVE_LLONG)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long long, OP, (SIZE))                                          \
        else if ((TYPE) == H5T_NATIVE_ULLONG)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), unsigned long long, OP, (SIZE))                                 \
        else if ((TYPE) == H5T_NATIVE_FLOAT)                                                                 \
            H5Z_XFORM_DO_OP1((RESL), (RESR), float, OP, (SIZE))                                              \
        else if ((TYPE) == H5T_NATIVE_DOUBLE)                                                                \
            H5Z_XFORM_DO_OP1((RESL), (RESR), double, OP, (SIZE))                                             \
        else if ((TYPE) == H5T_NATIVE_LDOUBLE)                                                               \
            H5Z_XFORM_DO_OP1((RESL), (RESR), long double, OP, (SIZE))                                        \
    }
#endif 

#define H5Z_XFORM_DO_OP3(OP)                                                                                 \
    {                                                                                                        \
        if ((tree->lchild->type == H5Z_XFORM_INTEGER) && (tree->rchild->type == H5Z_XFORM_INTEGER)) {        \
            tree->type          = H5Z_XFORM_INTEGER;                                                         \
            tree->value.int_val = tree->lchild->value.int_val OP tree->rchild->value.int_val;                \
            H5MM_xfree(tree->lchild);                                                                        \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->lchild = NULL;                                                                             \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
        else if (((tree->lchild->type == H5Z_XFORM_FLOAT) || (tree->lchild->type == H5Z_XFORM_INTEGER)) &&   \
                 ((tree->rchild->type == H5Z_XFORM_FLOAT) || (tree->rchild->type == H5Z_XFORM_INTEGER))) {   \
            tree->type = H5Z_XFORM_FLOAT;                                                                    \
            tree->value.float_val =                                                                          \
                ((tree->lchild->type == H5Z_XFORM_FLOAT) ? tree->lchild->value.float_val                     \
                                                         : (double)tree->lchild->value.int_val)              \
                    OP((tree->rchild->type == H5Z_XFORM_FLOAT) ? tree->rchild->value.float_val               \
                                                               : (double)tree->rchild->value.int_val);       \
            H5MM_xfree(tree->lchild);                                                                        \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->lchild = NULL;                                                                             \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
    }

#define H5Z_XFORM_DO_OP4(TYPE)                                                                               \
    {                                                                                                        \
        if ((ret_value = (H5Z_node *)H5MM_malloc(sizeof(H5Z_node))) == NULL)                                 \
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Ran out of memory trying to copy parse tree");     \
        else {                                                                                               \
            ret_value->type = (TYPE);                                                                        \
            if (tree->lchild)                                                                                \
                ret_value->lchild =                                                                          \
                    (H5Z_node *)H5Z__xform_copy_tree(tree->lchild, dat_val_pointers, new_dat_val_pointers);  \
            else                                                                                             \
                ret_value->lchild = NULL;                                                                    \
            if (tree->rchild)                                                                                \
                ret_value->rchild =                                                                          \
                    (H5Z_node *)H5Z__xform_copy_tree(tree->rchild, dat_val_pointers, new_dat_val_pointers);  \
            else                                                                                             \
                ret_value->rchild = NULL;                                                                    \
        }                                                                                                    \
    }

#define H5Z_XFORM_DO_OP5(TYPE, SIZE)                                                                         \
    {                                                                                                        \
        TYPE val =                                                                                           \
            ((tree->type == H5Z_XFORM_INTEGER) ? (TYPE)tree->value.int_val : (TYPE)tree->value.float_val);   \
        H5VM_array_fill(array, &val, sizeof(TYPE), (SIZE));                                                  \
    }

#define H5Z_XFORM_DO_OP6(OP)                                                                                 \
    {                                                                                                        \
        if (!tree->lchild && (tree->rchild->type == H5Z_XFORM_INTEGER)) {                                    \
            tree->type          = H5Z_XFORM_INTEGER;                                                         \
            tree->value.int_val = OP tree->rchild->value.int_val;                                            \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
        else if (!tree->lchild && (tree->rchild->type == H5Z_XFORM_FLOAT)) {                                 \
            tree->type            = H5Z_XFORM_FLOAT;                                                         \
            tree->value.float_val = OP tree->rchild->value.float_val;                                        \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
        else if ((tree->lchild->type == H5Z_XFORM_INTEGER) && (tree->rchild->type == H5Z_XFORM_INTEGER)) {   \
            tree->type          = H5Z_XFORM_INTEGER;                                                         \
            tree->value.int_val = tree->lchild->value.int_val OP tree->rchild->value.int_val;                \
            H5MM_xfree(tree->lchild);                                                                        \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->lchild = NULL;                                                                             \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
        else if (((tree->lchild->type == H5Z_XFORM_FLOAT) || (tree->lchild->type == H5Z_XFORM_INTEGER)) &&   \
                 ((tree->rchild->type == H5Z_XFORM_FLOAT) || (tree->rchild->type == H5Z_XFORM_INTEGER))) {   \
            tree->type = H5Z_XFORM_FLOAT;                                                                    \
            tree->value.float_val =                                                                          \
                ((tree->lchild->type == H5Z_XFORM_FLOAT) ? tree->lchild->value.float_val                     \
                                                         : (double)tree->lchild->value.int_val)              \
                    OP((tree->rchild->type == H5Z_XFORM_FLOAT) ? tree->rchild->value.float_val               \
                                                               : (double)tree->rchild->value.int_val);       \
            H5MM_xfree(tree->lchild);                                                                        \
            H5MM_xfree(tree->rchild);                                                                        \
            tree->lchild = NULL;                                                                             \
            tree->rchild = NULL;                                                                             \
        }                                                                                                    \
    }

static void
H5Z__unget_token(H5Z_token *current)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(current);

    current->tok_type  = current->tok_last_type;
    current->tok_begin = current->tok_last_begin;
    current->tok_end   = current->tok_last_end;

    FUNC_LEAVE_NOAPI_VOID
}

static H5Z_token *
H5Z__get_token(H5Z_token *current)
{
    H5Z_token *ret_value = current;

    FUNC_ENTER_PACKAGE

    
    assert(current);

    
    current->tok_last_type  = current->tok_type;
    current->tok_last_begin = current->tok_begin;
    current->tok_last_end   = current->tok_end;

    current->tok_begin = current->tok_end;

    while (current->tok_begin[0] != '\0') {
        if (isspace(current->tok_begin[0])) {
            
        }
        else if (isdigit(current->tok_begin[0]) || current->tok_begin[0] == '.') {
            current->tok_end = current->tok_begin;

            
            if (current->tok_end[0] != '.') {
                
                current->tok_type = H5Z_XFORM_INTEGER;

                while (isdigit(current->tok_end[0]))
                    ++current->tok_end;
            }

            
            if (current->tok_end[0] == '.' || current->tok_end[0] == 'e' || current->tok_end[0] == 'E') {
                current->tok_type = H5Z_XFORM_FLOAT;

                if (current->tok_end[0] == '.')
                    do {
                        ++current->tok_end;
                    } while (isdigit(current->tok_end[0]));

                if (current->tok_end[0] == 'e' || current->tok_end[0] == 'E') {
                    ++current->tok_end;

                    if (current->tok_end[0] == '-' || current->tok_end[0] == '+')
                        ++current->tok_end;

                    if (!isdigit(current->tok_end[0])) {
                        current->tok_type = H5Z_XFORM_ERROR;
                        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, current,
                                    "Invalidly formatted floating point number");
                    }

                    while (isdigit(current->tok_end[0]))
                        ++current->tok_end;
                }

                
                if (isalpha(current->tok_end[0]) || current->tok_end[0] == '.') {
                    current->tok_type = H5Z_XFORM_ERROR;
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, current, "Invalidly formatted floating point number");
                }
            }

            break;
        }
        else if (isalpha(current->tok_begin[0])) {
            
            current->tok_type = H5Z_XFORM_SYMBOL;
            current->tok_end  = current->tok_begin;

            while (isalnum(current->tok_end[0]))
                ++current->tok_end;

            break;
        }
        else {
            
            switch (current->tok_begin[0]) {
                case '+':
                    current->tok_type = H5Z_XFORM_PLUS;
                    break;
                case '-':
                    current->tok_type = H5Z_XFORM_MINUS;
                    break;
                case '*':
                    current->tok_type = H5Z_XFORM_MULT;
                    break;
                case '/':
                    current->tok_type = H5Z_XFORM_DIVIDE;
                    break;
                case '(':
                    current->tok_type = H5Z_XFORM_LPAREN;
                    break;
                case ')':
                    current->tok_type = H5Z_XFORM_RPAREN;
                    break;
                default:
                    current->tok_type = H5Z_XFORM_ERROR;
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, current,
                                "Unknown H5Z_token in data transform expression ");
            }

            current->tok_end = current->tok_begin + 1;
            break;
        }

        ++current->tok_begin;
    }

    if (current->tok_begin[0] == '\0')
        current->tok_type = H5Z_XFORM_END;

    
    ret_value = current;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static void
H5Z__xform_destroy_parse_tree(H5Z_node *tree)
{
    FUNC_ENTER_PACKAGE_NOERR

    if (tree) {
        H5Z__xform_destroy_parse_tree(tree->lchild);
        H5Z__xform_destroy_parse_tree(tree->rchild);
        H5MM_xfree(tree);
        tree = NULL;
    }

    FUNC_LEAVE_NOAPI_VOID
}

static void *
H5Z__xform_parse(const char *expression, H5Z_datval_ptrs *dat_val_pointers)
{
    H5Z_token tok;
    void     *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (!expression)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "No expression provided?");

    
    tok.tok_expr = tok.tok_begin = tok.tok_end = expression;

    ret_value = (void *)H5Z__parse_expression(&tok, dat_val_pointers);

    H5Z__xform_reduce_tree((H5Z_node *)ret_value);

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static H5Z_node *
H5Z__parse_expression(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers)
{
    H5Z_node *expr;
    H5Z_node *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    expr = H5Z__parse_term(current, dat_val_pointers);

    for (;;) {
        H5Z_node *new_node;

        current = H5Z__get_token(current);

        switch (current->tok_type) {
            case H5Z_XFORM_PLUS:
                new_node = H5Z__new_node(H5Z_XFORM_PLUS);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(expr);
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
                }

                new_node->lchild = expr;
                new_node->rchild = H5Z__parse_term(current, dat_val_pointers);

                if (!new_node->rchild) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                expr = new_node;
                break;

            case H5Z_XFORM_MINUS:
                new_node = H5Z__new_node(H5Z_XFORM_MINUS);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(expr);
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
                }

                new_node->lchild = expr;
                new_node->rchild = H5Z__parse_term(current, dat_val_pointers);

                if (!new_node->rchild) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                expr = new_node;
                break;

            case H5Z_XFORM_RPAREN:
                H5Z__unget_token(current);
                HGOTO_DONE(expr);

            case H5Z_XFORM_END:
                HGOTO_DONE(expr);

            case H5Z_XFORM_ERROR:
            case H5Z_XFORM_INTEGER:
            case H5Z_XFORM_FLOAT:
            case H5Z_XFORM_SYMBOL:
            case H5Z_XFORM_MULT:
            case H5Z_XFORM_DIVIDE:
            case H5Z_XFORM_LPAREN:
            default:
                H5Z__xform_destroy_parse_tree(expr);
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static H5Z_node *
H5Z__parse_term(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers)
{
    H5Z_node *term      = NULL;
    H5Z_node *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    term = H5Z__parse_factor(current, dat_val_pointers);

    for (;;) {
        H5Z_node *new_node;

        current = H5Z__get_token(current);

        switch (current->tok_type) {
            case H5Z_XFORM_MULT:
                new_node = H5Z__new_node(H5Z_XFORM_MULT);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(term);
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
                }

                new_node->lchild = term;
                new_node->rchild = H5Z__parse_factor(current, dat_val_pointers);

                if (!new_node->rchild) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                term = new_node;
                break;

            case H5Z_XFORM_DIVIDE:
                new_node = H5Z__new_node(H5Z_XFORM_DIVIDE);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(term);
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
                }

                new_node->lchild = term;
                new_node->rchild = H5Z__parse_factor(current, dat_val_pointers);
                term             = new_node;

                if (!new_node->rchild) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }
                break;

            case H5Z_XFORM_RPAREN:
                H5Z__unget_token(current);
                HGOTO_DONE(term);

            case H5Z_XFORM_END:
                HGOTO_DONE(term);

            case H5Z_XFORM_INTEGER:
            case H5Z_XFORM_FLOAT:
            case H5Z_XFORM_SYMBOL:
            case H5Z_XFORM_PLUS:
            case H5Z_XFORM_MINUS:
            case H5Z_XFORM_LPAREN:
                H5Z__unget_token(current);
                HGOTO_DONE(term);

            case H5Z_XFORM_ERROR:
            default:
                H5Z__xform_destroy_parse_tree(term);
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL,
                            "bad transform type passed to data transform expression");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static H5Z_node *
H5Z__parse_factor(H5Z_token *current, H5Z_datval_ptrs *dat_val_pointers)
{
    H5Z_node *factor = NULL;
    H5Z_node *new_node;
    H5Z_node *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    current = H5Z__get_token(current);

    switch (current->tok_type) {
        case H5Z_XFORM_INTEGER:
            factor = H5Z__new_node(H5Z_XFORM_INTEGER);

            if (!factor)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
            sscanf(current->tok_begin, "%ld", &factor->value.int_val);
            break;

        case H5Z_XFORM_FLOAT:
            factor = H5Z__new_node(H5Z_XFORM_FLOAT);

            if (!factor)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");
            sscanf(current->tok_begin, "%lf", &factor->value.float_val);
            break;

        case H5Z_XFORM_SYMBOL:
            factor = H5Z__new_node(H5Z_XFORM_SYMBOL);

            if (!factor)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");

            factor->value.dat_val = &(dat_val_pointers->ptr_dat_val[dat_val_pointers->num_ptrs]);
            dat_val_pointers->num_ptrs++;
            break;

        case H5Z_XFORM_LPAREN:
            factor = H5Z__parse_expression(current, dat_val_pointers);

            if (!factor)
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Unable to allocate new node");

            current = H5Z__get_token(current);

            if (current->tok_type != H5Z_XFORM_RPAREN) {
                H5Z__xform_destroy_parse_tree(factor);
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Syntax error in data transform expression");
            }
            break;

        case H5Z_XFORM_RPAREN:
            
            H5Z__xform_destroy_parse_tree(factor);
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Syntax error: unexpected ')' ");

        case H5Z_XFORM_PLUS:
            
            new_node = H5Z__parse_factor(current, dat_val_pointers);

            if (new_node) {
                if (new_node->type != H5Z_XFORM_INTEGER && new_node->type != H5Z_XFORM_FLOAT &&
                    new_node->type != H5Z_XFORM_SYMBOL) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    H5Z__xform_destroy_parse_tree(factor);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                factor   = new_node;
                new_node = H5Z__new_node(H5Z_XFORM_PLUS);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(factor);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                new_node->rchild = factor;
                factor           = new_node;
            }
            else {
                H5Z__xform_destroy_parse_tree(factor);
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
            }
            break;

        case H5Z_XFORM_MINUS:
            
            new_node = H5Z__parse_factor(current, dat_val_pointers);

            if (new_node) {
                if (new_node->type != H5Z_XFORM_INTEGER && new_node->type != H5Z_XFORM_FLOAT &&
                    new_node->type != H5Z_XFORM_SYMBOL) {
                    H5Z__xform_destroy_parse_tree(new_node);
                    H5Z__xform_destroy_parse_tree(factor);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                factor   = new_node;
                new_node = H5Z__new_node(H5Z_XFORM_MINUS);

                if (!new_node) {
                    H5Z__xform_destroy_parse_tree(factor);
                    HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
                }

                new_node->rchild = factor;
                factor           = new_node;
            }
            else {
                H5Z__xform_destroy_parse_tree(factor);
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error parsing data transform expression");
            }
            break;

        case H5Z_XFORM_END:
            break;

        case H5Z_XFORM_MULT:
        case H5Z_XFORM_DIVIDE:
        case H5Z_XFORM_ERROR:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL,
                        "Invalid token while parsing data transform expression");
    }

    
    ret_value = factor;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static H5Z_node *
H5Z__new_node(H5Z_token_type type)
{
    H5Z_node *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (NULL == (ret_value = (H5Z_node *)H5MM_calloc(sizeof(H5Z_node))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "Ran out of memory trying to allocate space for nodes in the parse tree");

    ret_value->type = type;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5Z_xform_eval(H5Z_data_xform_t *data_xform_prop, void *array, size_t array_size, const H5T_t *buf_type)
{
    H5Z_node  *tree;
    hid_t      array_type;
    H5Z_result res;
    size_t     i;
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(data_xform_prop);

    tree = data_xform_prop->parse_root;

    
    if ((array_type = H5Z__xform_find_type(buf_type)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Cannot perform data transform on this type.");

    

    
    if (tree->type == H5Z_XFORM_INTEGER || tree->type == H5Z_XFORM_FLOAT) {
        if (array_type == H5T_NATIVE_CHAR)
            H5Z_XFORM_DO_OP5(char, array_size)
#if CHAR_MIN >= 0
        else if (array_type == H5T_NATIVE_SCHAR)
            H5Z_XFORM_DO_OP5(signed char, array_size)
#else  
        else if (array_type == H5T_NATIVE_UCHAR)
            H5Z_XFORM_DO_OP5(unsigned char, array_size)
#endif 
        else if (array_type == H5T_NATIVE_SHORT)
            H5Z_XFORM_DO_OP5(short, array_size)
        else if (array_type == H5T_NATIVE_USHORT)
            H5Z_XFORM_DO_OP5(unsigned short, array_size)
        else if (array_type == H5T_NATIVE_INT)
            H5Z_XFORM_DO_OP5(int, array_size)
        else if (array_type == H5T_NATIVE_UINT)
            H5Z_XFORM_DO_OP5(unsigned int, array_size)
        else if (array_type == H5T_NATIVE_LONG)
            H5Z_XFORM_DO_OP5(long, array_size)
        else if (array_type == H5T_NATIVE_ULONG)
            H5Z_XFORM_DO_OP5(unsigned long, array_size)
        else if (array_type == H5T_NATIVE_LLONG)
            H5Z_XFORM_DO_OP5(long long, array_size)
        else if (array_type == H5T_NATIVE_ULLONG)
            H5Z_XFORM_DO_OP5(unsigned long long, array_size)
        else if (array_type == H5T_NATIVE_FLOAT)
            H5Z_XFORM_DO_OP5(float, array_size)
        else if (array_type == H5T_NATIVE_DOUBLE)
            H5Z_XFORM_DO_OP5(double, array_size)
        else if (array_type == H5T_NATIVE_LDOUBLE)
            H5Z_XFORM_DO_OP5(long double, array_size)

    } 
    
    else {
        
        if (data_xform_prop->dat_val_pointers->num_ptrs == 1)
            data_xform_prop->dat_val_pointers->ptr_dat_val[0] = array;

        
        else {
            for (i = 0; i < data_xform_prop->dat_val_pointers->num_ptrs; i++) {
                if (NULL == (data_xform_prop->dat_val_pointers->ptr_dat_val[i] = (void *)H5MM_malloc(
                                 array_size * H5T_get_size((H5T_t *)H5I_object(array_type)))))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                "Ran out of memory trying to allocate space for data in data transform");

                H5MM_memcpy(data_xform_prop->dat_val_pointers->ptr_dat_val[i], array,
                            array_size * H5T_get_size((H5T_t *)H5I_object(array_type)));
            } 
        }     

        if (H5Z__xform_eval_full(tree, array_size, array_type, &res) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "error while performing data transform");

        if (data_xform_prop->dat_val_pointers->num_ptrs > 1)
            H5MM_memcpy(array, res.value.dat_val, array_size * H5T_get_size((H5T_t *)H5I_object(array_type)));

        
        if (data_xform_prop->dat_val_pointers->num_ptrs > 1)
            for (i = 0; i < data_xform_prop->dat_val_pointers->num_ptrs; i++)
                H5MM_xfree(data_xform_prop->dat_val_pointers->ptr_dat_val[i]);
    } 

done:
    if (ret_value < 0) {
        
        if (data_xform_prop->dat_val_pointers->num_ptrs > 1)
            for (i = 0; i < data_xform_prop->dat_val_pointers->num_ptrs; i++)
                if (data_xform_prop->dat_val_pointers->ptr_dat_val[i])
                    H5MM_xfree(data_xform_prop->dat_val_pointers->ptr_dat_val[i]);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5Z__xform_eval_full(H5Z_node *tree, const size_t array_size, const hid_t array_type, H5Z_result *res)
{
    H5Z_result resl, resr;
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(tree);

    memset(&resl, 0, sizeof(H5Z_result));
    memset(&resr, 0, sizeof(H5Z_result));

    if (tree->type == H5Z_XFORM_INTEGER) {
        res->type          = H5Z_XFORM_INTEGER;
        res->value.int_val = tree->value.int_val;
    } 
    else if (tree->type == H5Z_XFORM_FLOAT) {
        res->type            = H5Z_XFORM_FLOAT;
        res->value.float_val = tree->value.float_val;
    } 
    else if (tree->type == H5Z_XFORM_SYMBOL) {
        res->type = H5Z_XFORM_SYMBOL;

        
        res->value.dat_val = *((void **)(tree->value.dat_val));
    } 
    else {
        if (tree->lchild && H5Z__xform_eval_full(tree->lchild, array_size, array_type, &resl) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "error while performing data transform");
        if (H5Z__xform_eval_full(tree->rchild, array_size, array_type, &resr) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "error while performing data transform");

        res->type = H5Z_XFORM_SYMBOL;

        
        switch (tree->type) {
            case H5Z_XFORM_PLUS:
                H5Z_XFORM_TYPE_OP(resl, resr, array_type, +, array_size)
                break;

            case H5Z_XFORM_MINUS:
                H5Z_XFORM_TYPE_OP(resl, resr, array_type, -, array_size)
                break;

            case H5Z_XFORM_MULT:
                H5Z_XFORM_TYPE_OP(resl, resr, array_type, *, array_size)
                break;

            case H5Z_XFORM_DIVIDE:
                H5Z_XFORM_TYPE_OP(resl, resr, array_type, /, array_size)
                break;

            case H5Z_XFORM_ERROR:
            case H5Z_XFORM_INTEGER:
            case H5Z_XFORM_FLOAT:
            case H5Z_XFORM_SYMBOL:
            case H5Z_XFORM_LPAREN:
            case H5Z_XFORM_RPAREN:
            case H5Z_XFORM_END:
            default:
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Invalid expression tree");
        } 

        
        
        if (resl.type == H5Z_XFORM_SYMBOL)
            res->value.dat_val = resl.value.dat_val;
        else if (resr.type == H5Z_XFORM_SYMBOL)
            res->value.dat_val = resr.value.dat_val;
        else
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "error during transform evaluation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static hid_t
H5Z__xform_find_type(const H5T_t *type)
{
    H5T_t *tmp;                 
    hid_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(type);

    
    if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_SHORT)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_SHORT);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_INT)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_INT);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_LONG)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_LONG);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_LLONG)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_LLONG);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_UCHAR)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_UCHAR);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_CHAR)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_CHAR);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_SCHAR)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_SCHAR);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_USHORT)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_USHORT);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_UINT)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_UINT);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_ULONG)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_ULONG);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_ULLONG)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_ULLONG);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_FLOAT);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_DOUBLE)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_DOUBLE);
    
    else if ((tmp = (H5T_t *)H5I_object(H5T_NATIVE_LDOUBLE)) && 0 == H5T_cmp(type, tmp, false))
        HGOTO_DONE(H5T_NATIVE_LDOUBLE);
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "could not find matching type");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5Z__xform_copy_tree(H5Z_node *tree, H5Z_datval_ptrs *dat_val_pointers, H5Z_datval_ptrs *new_dat_val_pointers)
{
    H5Z_node *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(tree);

    if (tree->type == H5Z_XFORM_INTEGER) {
        if ((ret_value = (H5Z_node *)H5MM_malloc(sizeof(H5Z_node))) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Ran out of memory trying to copy parse tree");
        else {
            ret_value->type          = H5Z_XFORM_INTEGER;
            ret_value->value.int_val = tree->value.int_val;
            ret_value->lchild        = NULL;
            ret_value->rchild        = NULL;
        }
    }
    else if (tree->type == H5Z_XFORM_FLOAT) {
        if ((ret_value = (H5Z_node *)H5MM_malloc(sizeof(H5Z_node))) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Ran out of memory trying to copy parse tree");
        else {
            ret_value->type            = H5Z_XFORM_FLOAT;
            ret_value->value.float_val = tree->value.float_val;
            ret_value->lchild          = NULL;
            ret_value->rchild          = NULL;
        }
    }
    else if (tree->type == H5Z_XFORM_SYMBOL) {
        if ((ret_value = (H5Z_node *)H5MM_malloc(sizeof(H5Z_node))) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "Ran out of memory trying to copy parse tree");
        else {
            ret_value->type = H5Z_XFORM_SYMBOL;

            ret_value->value.dat_val = &(new_dat_val_pointers->ptr_dat_val[new_dat_val_pointers->num_ptrs]);
            new_dat_val_pointers->num_ptrs++;
            ret_value->lchild = NULL;
            ret_value->rchild = NULL;
        }
    }
    else if (tree->type == H5Z_XFORM_MULT)
        H5Z_XFORM_DO_OP4(H5Z_XFORM_MULT)
    else if (tree->type == H5Z_XFORM_PLUS)
        H5Z_XFORM_DO_OP4(H5Z_XFORM_PLUS)
    else if (tree->type == H5Z_XFORM_MINUS)
        H5Z_XFORM_DO_OP4(H5Z_XFORM_MINUS)
    else if (tree->type == H5Z_XFORM_DIVIDE)
        H5Z_XFORM_DO_OP4(H5Z_XFORM_DIVIDE)
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "Error in parse tree while trying to copy");

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

static bool
H5Z__op_is_numbs(H5Z_node *_tree)
{
    bool ret_value = false;

    FUNC_ENTER_PACKAGE_NOERR

    assert(_tree);

    if (((_tree->lchild->type == H5Z_XFORM_INTEGER) || (_tree->lchild->type == H5Z_XFORM_FLOAT)) &&
        ((_tree->rchild->type == H5Z_XFORM_INTEGER) || (_tree->rchild->type == H5Z_XFORM_FLOAT)))
        ret_value = true;

    FUNC_LEAVE_NOAPI(ret_value)
}

static bool
H5Z__op_is_numbs2(H5Z_node *_tree)
{
    bool ret_value = false;

    FUNC_ENTER_PACKAGE_NOERR

    assert(_tree);

    if ((!_tree->lchild &&
         ((_tree->rchild->type == H5Z_XFORM_INTEGER) || (_tree->rchild->type == H5Z_XFORM_FLOAT))) ||
        ((_tree->lchild &&
          ((_tree->lchild->type == H5Z_XFORM_INTEGER) || (_tree->lchild->type == H5Z_XFORM_FLOAT))) &&
         (_tree->rchild &&
          ((_tree->rchild->type == H5Z_XFORM_INTEGER) || (_tree->rchild->type == H5Z_XFORM_FLOAT)))))
        ret_value = true;

    FUNC_LEAVE_NOAPI(ret_value)
}

static void
H5Z__xform_reduce_tree(H5Z_node *tree)
{
    FUNC_ENTER_PACKAGE_NOERR

    if (tree) {
        if ((tree->type == H5Z_XFORM_DIVIDE) || (tree->type == H5Z_XFORM_MULT)) {
            if (H5Z__op_is_numbs(tree))
                H5Z__do_op(tree);
            else {
                H5Z__xform_reduce_tree(tree->lchild);
                if (H5Z__op_is_numbs(tree))
                    H5Z__do_op(tree);
                else {
                    H5Z__xform_reduce_tree(tree->rchild);
                    if (H5Z__op_is_numbs(tree))
                        H5Z__do_op(tree);
                }
            }
        }
        else if ((tree->type == H5Z_XFORM_PLUS) || (tree->type == H5Z_XFORM_MINUS)) {
            if (H5Z__op_is_numbs2(tree))
                H5Z__do_op(tree);
            else {
                H5Z__xform_reduce_tree(tree->lchild);
                if (H5Z__op_is_numbs2(tree))
                    H5Z__do_op(tree);
                else {
                    H5Z__xform_reduce_tree(tree->rchild);
                    if (H5Z__op_is_numbs2(tree))
                        H5Z__do_op(tree);
                }
            }
        }
    }

    FUNC_LEAVE_NOAPI_VOID
}

static void
H5Z__do_op(H5Z_node *tree)
{
    FUNC_ENTER_PACKAGE_NOERR

    if (tree->type == H5Z_XFORM_DIVIDE)
        H5Z_XFORM_DO_OP3(/)
    else if (tree->type == H5Z_XFORM_MULT)
        H5Z_XFORM_DO_OP3(*)
    else if (tree->type == H5Z_XFORM_PLUS)
        H5Z_XFORM_DO_OP6(+)
    else if (tree->type == H5Z_XFORM_MINUS)
        H5Z_XFORM_DO_OP6(-)

    FUNC_LEAVE_NOAPI_VOID
}

H5Z_data_xform_t *
H5Z_xform_create(const char *expr)
{
    H5Z_data_xform_t *data_xform_prop = NULL;
    unsigned int      i;
    unsigned int      count     = 0;
    H5Z_data_xform_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    assert(expr);

    
    if (NULL == (data_xform_prop = (H5Z_data_xform_t *)H5MM_calloc(sizeof(H5Z_data_xform_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to allocate memory for data transform info");

    if (NULL == (data_xform_prop->dat_val_pointers = (H5Z_datval_ptrs *)H5MM_malloc(sizeof(H5Z_datval_ptrs))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "unable to allocate memory for data transform array storage");

    
    if (NULL == (data_xform_prop->xform_exp = (char *)H5MM_xstrdup(expr)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "unable to allocate memory for data transform expression");

    
    for (i = 0; i < strlen(expr); i++) {
        if (isalpha(expr[i])) {
            if ((i > 0) && (i < (strlen(expr) - 1))) {
                if (((expr[i] == 'E') || (expr[i] == 'e')) &&
                    (isdigit(expr[i - 1]) || (expr[i - 1] == '.')) &&
                    (isdigit(expr[i + 1]) || (expr[i + 1] == '-') || (expr[i + 1] == '+')))
                    continue;
            } 

            count++;
        } 
    }     

    
    if (count > 0)
        if (NULL ==
            (data_xform_prop->dat_val_pointers->ptr_dat_val = (void **)H5MM_calloc(count * sizeof(void *))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                        "unable to allocate memory for pointers in transform array");

    
    data_xform_prop->dat_val_pointers->num_ptrs = 0;

    
    if ((data_xform_prop->parse_root =
             (H5Z_node *)H5Z__xform_parse(expr, data_xform_prop->dat_val_pointers)) == NULL)
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to generate parse tree from expression");

    
    if (count != data_xform_prop->dat_val_pointers->num_ptrs)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL,
                    "error copying the parse tree, did not find correct number of \"variables\"");

    
    ret_value = data_xform_prop;

done:
    
    if (ret_value == NULL) {
        if (data_xform_prop) {
            if (data_xform_prop->parse_root)
                H5Z__xform_destroy_parse_tree(data_xform_prop->parse_root);
            if (data_xform_prop->xform_exp)
                H5MM_xfree(data_xform_prop->xform_exp);
            if (count > 0 && data_xform_prop->dat_val_pointers->ptr_dat_val)
                H5MM_xfree(data_xform_prop->dat_val_pointers->ptr_dat_val);
            if (data_xform_prop->dat_val_pointers)
                H5MM_xfree(data_xform_prop->dat_val_pointers);
            H5MM_xfree(data_xform_prop);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Z_xform_destroy(H5Z_data_xform_t *data_xform_prop)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (data_xform_prop) {
        
        H5Z__xform_destroy_parse_tree(data_xform_prop->parse_root);

        
        H5MM_xfree(data_xform_prop->xform_exp);

        
        if (data_xform_prop->dat_val_pointers->num_ptrs > 0)
            H5MM_xfree(data_xform_prop->dat_val_pointers->ptr_dat_val);

        
        H5MM_xfree(data_xform_prop->dat_val_pointers);

        
        H5MM_xfree(data_xform_prop);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Z_xform_copy(H5Z_data_xform_t **data_xform_prop)
{
    unsigned int      i;
    unsigned int      count               = 0;
    H5Z_data_xform_t *new_data_xform_prop = NULL;
    herr_t            ret_value           = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    if (*data_xform_prop) {
        
        if (NULL == (new_data_xform_prop = (H5Z_data_xform_t *)H5MM_calloc(sizeof(H5Z_data_xform_t))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate memory for data transform info");

        
        if (NULL == (new_data_xform_prop->xform_exp = (char *)H5MM_xstrdup((*data_xform_prop)->xform_exp)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "unable to allocate memory for data transform expression");

        if (NULL ==
            (new_data_xform_prop->dat_val_pointers = (H5Z_datval_ptrs *)H5MM_malloc(sizeof(H5Z_datval_ptrs))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "unable to allocate memory for data transform array storage");

        
        for (i = 0; i < strlen(new_data_xform_prop->xform_exp); i++)
            if (isalpha(new_data_xform_prop->xform_exp[i]))
                count++;

        if (count > 0)
            if (NULL == (new_data_xform_prop->dat_val_pointers->ptr_dat_val =
                             (void **)H5MM_calloc(count * sizeof(void *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                            "unable to allocate memory for pointers in transform array");

        
        new_data_xform_prop->dat_val_pointers->num_ptrs = 0;

        
        if ((new_data_xform_prop->parse_root = (H5Z_node *)H5Z__xform_copy_tree(
                 (*data_xform_prop)->parse_root, (*data_xform_prop)->dat_val_pointers,
                 new_data_xform_prop->dat_val_pointers)) == NULL)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "error copying the parse tree");

        
        if (count != new_data_xform_prop->dat_val_pointers->num_ptrs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL,
                        "error copying the parse tree, did not find correct number of \"variables\"");

        
        *data_xform_prop = new_data_xform_prop;
    } 

done:
    
    if (ret_value < 0) {
        if (new_data_xform_prop) {
            if (new_data_xform_prop->parse_root)
                H5Z__xform_destroy_parse_tree(new_data_xform_prop->parse_root);
            if (new_data_xform_prop->xform_exp)
                H5MM_xfree(new_data_xform_prop->xform_exp);
            H5MM_xfree(new_data_xform_prop);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5Z_xform_noop(const H5Z_data_xform_t *data_xform_prop)
{
    bool ret_value = true; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (data_xform_prop) {
        ret_value = false;

        
        if ((strlen(data_xform_prop->xform_exp) == 1) && data_xform_prop->dat_val_pointers &&
            (data_xform_prop->dat_val_pointers->num_ptrs == 1)) {
            ret_value = true;
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

const char *
H5Z_xform_extract_xform_str(const H5Z_data_xform_t *data_xform_prop)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(data_xform_prop);

    FUNC_LEAVE_NOAPI(data_xform_prop->xform_exp)
} 
