/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"  
#include "H5Mprivate.h" 
#include "H5Eprivate.h" 
#include "H5Ppkg.h"     

#define H5M_ACS_KEY_PREFETCH_SIZE_SIZE sizeof(size_t)
#define H5M_ACS_KEY_PREFETCH_SIZE_DEF  (size_t)(16 * 1024)
#define H5M_ACS_KEY_PREFETCH_SIZE_ENC  H5P__encode_size_t
#define H5M_ACS_KEY_PREFETCH_SIZE_DEC  H5P__decode_size_t

#define H5M_ACS_KEY_ALLOC_SIZE_SIZE sizeof(size_t)
#define H5M_ACS_KEY_ALLOC_SIZE_DEF  (size_t)(1024 * 1024)
#define H5M_ACS_KEY_ALLOC_SIZE_ENC  H5P__encode_size_t
#define H5M_ACS_KEY_ALLOC_SIZE_DEC  H5P__decode_size_t

static herr_t H5P__macc_reg_prop(H5P_genclass_t *pclass);

const H5P_libclass_t H5P_CLS_MACC[1] = {{
    "map access",        
    H5P_TYPE_MAP_ACCESS, 

    &H5P_CLS_LINK_ACCESS_g,   
    &H5P_CLS_MAP_ACCESS_g,    
    &H5P_CLS_MAP_ACCESS_ID_g, 
    &H5P_LST_MAP_ACCESS_ID_g, 
    H5P__macc_reg_prop,       

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static herr_t
H5P__macc_reg_prop(H5P_genclass_t *pclass)
{
    size_t key_prefetch_size = H5M_ACS_KEY_PREFETCH_SIZE_DEF; 
    size_t key_alloc_size =
        H5M_ACS_KEY_ALLOC_SIZE_DEF; 
    herr_t ret_value = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5M_ACS_KEY_PREFETCH_SIZE_NAME, H5M_ACS_KEY_PREFETCH_SIZE_SIZE,
                           &key_prefetch_size, NULL, NULL, NULL, H5M_ACS_KEY_PREFETCH_SIZE_ENC,
                           H5M_ACS_KEY_PREFETCH_SIZE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5M_ACS_KEY_ALLOC_SIZE_NAME, H5M_ACS_KEY_ALLOC_SIZE_SIZE, &key_alloc_size,
                           NULL, NULL, NULL, H5M_ACS_KEY_ALLOC_SIZE_ENC, H5M_ACS_KEY_ALLOC_SIZE_DEC, NULL,
                           NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_MAP_API

herr_t
H5Pset_map_iterate_hints(hid_t mapl_id, size_t key_prefetch_size, size_t key_alloc_size)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(mapl_id, H5P_MAP_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5M_ACS_KEY_PREFETCH_SIZE_NAME, &key_prefetch_size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set key prefetch size");
    if (H5P_set(plist, H5M_ACS_KEY_ALLOC_SIZE_NAME, &key_alloc_size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set key allocation size");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_map_iterate_hints(hid_t mapl_id, size_t *key_prefetch_size , size_t *key_alloc_size )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(mapl_id, H5P_MAP_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (key_prefetch_size) {
        if (H5P_get(plist, H5M_ACS_KEY_PREFETCH_SIZE_NAME, key_prefetch_size) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get key prefetch size");
    } 
    if (key_alloc_size) {
        if (H5P_get(plist, H5M_ACS_KEY_ALLOC_SIZE_NAME, key_alloc_size) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get key allocation size");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif
