/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND      
#include "H5PBmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5PBpkg.h"     
#include "H5SLprivate.h" 

#define H5PB__PREPEND(page_ptr, head_ptr, tail_ptr, len)                                                     \
    {                                                                                                        \
        if ((head_ptr) == NULL) {                                                                            \
            (head_ptr) = (page_ptr);                                                                         \
            (tail_ptr) = (page_ptr);                                                                         \
        }                                                                                        \
        else {                                                                                               \
            (head_ptr)->prev = (page_ptr);                                                                   \
            (page_ptr)->next = (head_ptr);                                                                   \
            (head_ptr)       = (page_ptr);                                                                   \
        }                                                                                      \
        (len)++;                                                                                             \
    } 

#define H5PB__REMOVE(page_ptr, head_ptr, tail_ptr, len)                                                      \
    {                                                                                                        \
        if ((head_ptr) == (page_ptr)) {                                                                      \
            (head_ptr) = (page_ptr)->next;                                                                   \
            if ((head_ptr) != NULL)                                                                          \
                (head_ptr)->prev = NULL;                                                                     \
        }                                                                                        \
        else                                                                                                 \
            (page_ptr)->prev->next = (page_ptr)->next;                                                       \
        if ((tail_ptr) == (page_ptr)) {                                                                      \
            (tail_ptr) = (page_ptr)->prev;                                                                   \
            if ((tail_ptr) != NULL)                                                                          \
                (tail_ptr)->next = NULL;                                                                     \
        }                                                                                        \
        else                                                                                                 \
            (page_ptr)->next->prev = (page_ptr)->prev;                                                       \
        page_ptr->next = NULL;                                                                               \
        page_ptr->prev = NULL;                                                                               \
        (len)--;                                                                                             \
    }

#define H5PB__INSERT_LRU(page_buf, page_ptr)                                                                 \
    {                                                                                                        \
        assert(page_buf);                                                                                    \
        assert(page_ptr);                                                                                    \
                                                              \
        H5PB__PREPEND((page_ptr), (page_buf)->LRU_head_ptr, (page_buf)->LRU_tail_ptr,                        \
                      (page_buf)->LRU_list_len)                                                              \
    }

#define H5PB__REMOVE_LRU(page_buf, page_ptr)                                                                 \
    {                                                                                                        \
        assert(page_buf);                                                                                    \
        assert(page_ptr);                                                                                    \
                                                                        \
        H5PB__REMOVE((page_ptr), (page_buf)->LRU_head_ptr, (page_buf)->LRU_tail_ptr,                         \
                     (page_buf)->LRU_list_len)                                                               \
    }

#define H5PB__MOVE_TO_TOP_LRU(page_buf, page_ptr)                                                            \
    {                                                                                                        \
        assert(page_buf);                                                                                    \
        assert(page_ptr);                                                                                    \
                                                       \
        H5PB__REMOVE((page_ptr), (page_buf)->LRU_head_ptr, (page_buf)->LRU_tail_ptr,                         \
                     (page_buf)->LRU_list_len)                                                               \
        H5PB__PREPEND((page_ptr), (page_buf)->LRU_head_ptr, (page_buf)->LRU_tail_ptr,                        \
                      (page_buf)->LRU_list_len)                                                              \
    }

typedef struct {
    H5PB_t *page_buf;
    bool    actual_slist;
} H5PB_ud1_t;

static herr_t H5PB__insert_entry(H5PB_t *page_buf, H5PB_entry_t *page_entry);
static htri_t H5PB__make_space(H5F_shared_t *f_sh, H5PB_t *page_buf, H5FD_mem_t inserted_type);
static herr_t H5PB__write_entry(H5F_shared_t *f_sh, H5PB_entry_t *page_entry);

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE_STATIC(H5PB_t);

H5FL_DEFINE_STATIC(H5PB_entry_t);

herr_t
H5PB_reset_stats(H5PB_t *page_buf)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(page_buf);

    page_buf->accesses[0]  = 0;
    page_buf->accesses[1]  = 0;
    page_buf->hits[0]      = 0;
    page_buf->hits[1]      = 0;
    page_buf->misses[0]    = 0;
    page_buf->misses[1]    = 0;
    page_buf->evictions[0] = 0;
    page_buf->evictions[1] = 0;
    page_buf->bypasses[0]  = 0;
    page_buf->bypasses[1]  = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5PB_get_stats(const H5PB_t *page_buf, unsigned accesses[2], unsigned hits[2], unsigned misses[2],
               unsigned evictions[2], unsigned bypasses[2])
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(page_buf);

    accesses[0]  = page_buf->accesses[0];
    accesses[1]  = page_buf->accesses[1];
    hits[0]      = page_buf->hits[0];
    hits[1]      = page_buf->hits[1];
    misses[0]    = page_buf->misses[0];
    misses[1]    = page_buf->misses[1];
    evictions[0] = page_buf->evictions[0];
    evictions[1] = page_buf->evictions[1];
    bypasses[0]  = page_buf->bypasses[0];
    bypasses[1]  = page_buf->bypasses[1];

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5PB_print_stats(const H5PB_t *page_buf)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(page_buf);

    Rprintf("PAGE BUFFER STATISTICS:\n");

    Rprintf("******* METADATA\n");
    Rprintf("\t Total Accesses: %u\n", page_buf->accesses[0]);
    Rprintf("\t Hits: %u\n", page_buf->hits[0]);
    Rprintf("\t Misses: %u\n", page_buf->misses[0]);
    Rprintf("\t Evictions: %u\n", page_buf->evictions[0]);
    Rprintf("\t Bypasses: %u\n", page_buf->bypasses[0]);
    Rprintf("\t Hit Rate = %f%%\n",
           ((double)page_buf->hits[0] / (page_buf->accesses[0] - page_buf->bypasses[0])) * 100);
    Rprintf("*****************\n\n");

    Rprintf("******* RAWDATA\n");
    Rprintf("\t Total Accesses: %u\n", page_buf->accesses[1]);
    Rprintf("\t Hits: %u\n", page_buf->hits[1]);
    Rprintf("\t Misses: %u\n", page_buf->misses[1]);
    Rprintf("\t Evictions: %u\n", page_buf->evictions[1]);
    Rprintf("\t Bypasses: %u\n", page_buf->bypasses[1]);
    Rprintf("\t Hit Rate = %f%%\n",
           ((double)page_buf->hits[1] / (page_buf->accesses[1] - page_buf->bypasses[0])) * 100);
    Rprintf("*****************\n\n");

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5PB_create(H5F_shared_t *f_sh, size_t size, unsigned page_buf_min_meta_perc, unsigned page_buf_min_raw_perc)
{
    H5PB_t *page_buf  = NULL;
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);

    
    if (f_sh->fs_strategy != H5F_FSPACE_STRATEGY_PAGE)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL,
                    "Enabling Page Buffering requires PAGE file space strategy");
    
    else if (size > f_sh->fs_page_size) {
        hsize_t temp_size;

        temp_size = (size / f_sh->fs_page_size) * f_sh->fs_page_size;
        H5_CHECKED_ASSIGN(size, size_t, temp_size, hsize_t);
    } 
    else if (0 != size % f_sh->fs_page_size)
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTINIT, FAIL, "Page Buffer size must be >= to the page size");

    
    if (NULL == (page_buf = H5FL_CALLOC(H5PB_t)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_NOSPACE, FAIL, "memory allocation failed");

    page_buf->max_size = size;
    H5_CHECKED_ASSIGN(page_buf->page_size, size_t, f_sh->fs_page_size, hsize_t);
    page_buf->min_meta_perc = page_buf_min_meta_perc;
    page_buf->min_raw_perc  = page_buf_min_raw_perc;

    
    page_buf->min_meta_count = (unsigned)((size * page_buf_min_meta_perc) / (f_sh->fs_page_size * 100));
    page_buf->min_raw_count  = (unsigned)((size * page_buf_min_raw_perc) / (f_sh->fs_page_size * 100));

    if (NULL == (page_buf->slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTCREATE, FAIL, "can't create skip list");
    if (NULL == (page_buf->mf_slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTCREATE, FAIL, "can't create skip list");

    if (NULL == (page_buf->page_fac = H5FL_fac_init(page_buf->page_size)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTINIT, FAIL, "can't create page factory");

    f_sh->page_buf = page_buf;

done:
    if (ret_value < 0) {
        if (page_buf != NULL) {
            if (page_buf->slist_ptr != NULL)
                H5SL_close(page_buf->slist_ptr);
            if (page_buf->mf_slist_ptr != NULL)
                H5SL_close(page_buf->mf_slist_ptr);
            if (page_buf->page_fac != NULL)
                H5FL_fac_term(page_buf->page_fac);
            page_buf = H5FL_FREE(H5PB_t, page_buf);
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PB__flush_cb(void *item, void H5_ATTR_UNUSED *key, void *_op_data)
{
    H5PB_entry_t *page_entry = (H5PB_entry_t *)item; 
    H5F_shared_t *f_sh       = (H5F_shared_t *)_op_data;
    herr_t        ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(page_entry);
    assert(f_sh);

    
    if (page_entry->is_dirty)
        if (H5PB__write_entry(f_sh, page_entry) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_WRITEERROR, FAIL, "file write failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_flush(H5F_shared_t *f_sh)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);

    
    if (f_sh->page_buf && (H5F_ACC_RDWR & H5F_SHARED_INTENT(f_sh))) {
        H5PB_t *page_buf = f_sh->page_buf;

        
        if (H5SL_iterate(page_buf->slist_ptr, H5PB__flush_cb, f_sh))
            HGOTO_ERROR(H5E_PAGEBUF, H5E_BADITER, FAIL, "can't flush page buffer skip list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PB__dest_cb(void *item, void H5_ATTR_UNUSED *key, void *_op_data)
{
    H5PB_entry_t *page_entry = (H5PB_entry_t *)item; 
    H5PB_ud1_t   *op_data    = (H5PB_ud1_t *)_op_data;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(page_entry);
    assert(op_data);
    assert(op_data->page_buf);

    
    if (op_data->actual_slist) {
        H5PB__REMOVE_LRU(op_data->page_buf, page_entry)
        page_entry->page_buf_ptr = H5FL_FAC_FREE(op_data->page_buf->page_fac, page_entry->page_buf_ptr);
    } 

    
    page_entry = H5FL_FREE(H5PB_entry_t, page_entry);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5PB_dest(H5F_shared_t *f_sh)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);

    
    if (f_sh->page_buf) {
        H5PB_t    *page_buf = f_sh->page_buf;
        H5PB_ud1_t op_data; 

        if (H5PB_flush(f_sh) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTFLUSH, FAIL, "can't flush page buffer");

        
        op_data.page_buf = page_buf;

        
        op_data.actual_slist = true;
        if (H5SL_destroy(page_buf->slist_ptr, H5PB__dest_cb, &op_data))
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTCLOSEOBJ, FAIL, "can't destroy page buffer skip list");

        
        op_data.actual_slist = false;
        if (H5SL_destroy(page_buf->mf_slist_ptr, H5PB__dest_cb, &op_data))
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTCLOSEOBJ, FAIL, "can't destroy page buffer skip list");

        
        if (H5FL_fac_term(page_buf->page_fac) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTRELEASE, FAIL, "can't destroy page buffer page factory");

        f_sh->page_buf = H5FL_FREE(H5PB_t, page_buf);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_add_new_page(H5F_shared_t *f_sh, H5FD_mem_t type, haddr_t page_addr)
{
    H5PB_t       *page_buf;             
    H5PB_entry_t *page_entry = NULL;    
    herr_t        ret_value  = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    page_buf = f_sh->page_buf;
    assert(page_buf);

    
    
    if (NULL == H5SL_search(page_buf->mf_slist_ptr, &(page_addr))) {
        
        if (NULL == (page_entry = H5FL_CALLOC(H5PB_entry_t)))
            HGOTO_ERROR(H5E_PAGEBUF, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        page_entry->addr     = page_addr;
        page_entry->type     = (H5F_mem_page_t)type;
        page_entry->is_dirty = false;

        
        if (H5SL_insert(page_buf->mf_slist_ptr, page_entry, &(page_entry->addr)) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_BADVALUE, FAIL, "Can't insert entry in skip list");
    } 

done:
    if (ret_value < 0)
        if (page_entry)
            page_entry = H5FL_FREE(H5PB_entry_t, page_entry);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_update_entry(H5PB_t *page_buf, haddr_t addr, size_t size, const void *buf)
{
    H5PB_entry_t *page_entry; 
    haddr_t       page_addr;

    FUNC_ENTER_NOAPI_NOERR

    
    assert(page_buf);
    assert(size <= page_buf->page_size);
    assert(buf);

    
    page_addr = (addr / page_buf->page_size) * page_buf->page_size;

    
    page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&page_addr));
    if (page_entry) {
        haddr_t offset;

        assert(addr + size <= page_addr + page_buf->page_size);
        offset = addr - page_addr;
        H5MM_memcpy((uint8_t *)page_entry->page_buf_ptr + offset, buf, size);

        
        H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5PB_remove_entry(const H5F_shared_t *f_sh, haddr_t addr)
{
    H5PB_t       *page_buf;             
    H5PB_entry_t *page_entry = NULL;    
    herr_t        ret_value  = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    page_buf = f_sh->page_buf;
    assert(page_buf);

    
    page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&addr));

    
    if (page_entry) {
        assert(page_entry->type != H5F_MEM_PAGE_DRAW);
        if (NULL == H5SL_remove(page_buf->slist_ptr, &(page_entry->addr)))
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Page Entry is not in skip list");

        
        H5PB__REMOVE_LRU(page_buf, page_entry)
        assert(H5SL_count(page_buf->slist_ptr) == page_buf->LRU_list_len);

        page_buf->meta_count--;

        page_entry->page_buf_ptr = H5FL_FAC_FREE(page_buf->page_fac, page_entry->page_buf_ptr);
        page_entry               = H5FL_FREE(H5PB_entry_t, page_entry);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_read(H5F_shared_t *f_sh, H5FD_mem_t type, haddr_t addr, size_t size, void *buf )
{
    H5PB_t       *page_buf;                        
    H5PB_entry_t *page_entry;                      
    H5FD_t       *file;                            
    haddr_t       first_page_addr, last_page_addr; 
    haddr_t       offset;
    haddr_t       search_addr;       
    hsize_t       num_touched_pages; 
    size_t        access_size = 0;
    bool          bypass_pb   = false; 
    hsize_t       i;                   
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);
    assert(type != H5FD_MEM_GHEAP);

    
    page_buf = f_sh->page_buf;

#ifdef H5_HAVE_PARALLEL
    if (H5F_SHARED_HAS_FEATURE(f_sh, H5FD_FEAT_HAS_MPI)) {
#if 1
        bypass_pb = true;
#else
        
        int mpi_size;

        if ((mpi_size = H5F_shared_mpi_get_size(f_sh)) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "can't retrieve MPI communicator size");
        if (1 != mpi_size)
            bypass_pb = true;
#endif
    } 
#endif

    
    if (NULL == page_buf || size >= page_buf->page_size || (bypass_pb && H5FD_MEM_DRAW == type)) {
        if (H5F__accum_read(f_sh, type, addr, size, buf) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_READERROR, FAIL, "read through metadata accumulator failed");

        
        if (page_buf) {
            if (type == H5FD_MEM_DRAW)
                page_buf->bypasses[1]++;
            else
                page_buf->bypasses[0]++;
        } 

        
        if (NULL == page_buf || (size >= page_buf->page_size && H5FD_MEM_DRAW != type) ||
            (bypass_pb && H5FD_MEM_DRAW == type))
            HGOTO_DONE(SUCCEED);
    } 

    
    if (page_buf) {
        if (type == H5FD_MEM_DRAW)
            page_buf->accesses[1]++;
        else
            page_buf->accesses[0]++;
    } 

    
    first_page_addr = (addr / page_buf->page_size) * page_buf->page_size;

    
    if (H5FD_MEM_DRAW == type) {
        last_page_addr = ((addr + size - 1) / page_buf->page_size) * page_buf->page_size;

        
        num_touched_pages =
            (last_page_addr / page_buf->page_size + 1) - (first_page_addr / page_buf->page_size);
        if (first_page_addr == last_page_addr) {
            assert(1 == num_touched_pages);
            last_page_addr = HADDR_UNDEF;
        } 
    }     
    
    else {
        num_touched_pages = 1;
        last_page_addr    = HADDR_UNDEF;
    } 

    
    file = f_sh->lf;

    
    if (H5FD_MEM_DRAW == type && size >= page_buf->page_size) {
        H5SL_node_t *node;

        
        node = H5SL_find(page_buf->slist_ptr, (void *)(&first_page_addr));
        for (i = 0; i < num_touched_pages; i++) {
            search_addr = i * page_buf->page_size + first_page_addr;

            
            if (!node && i != 0)
                node = H5SL_find(page_buf->slist_ptr, (void *)(&search_addr));

            
            if (node) {
                page_entry = (H5PB_entry_t *)H5SL_item(node);

                assert(page_entry);

                
                if (page_entry->addr >= addr + size)
                    break;

                assert(page_entry->addr == search_addr);

                if (page_entry->is_dirty) {
                    
                    if (i == 0 && first_page_addr != addr) {
                        offset = addr - first_page_addr;
                        assert(page_buf->page_size > offset);

                        H5MM_memcpy(buf, (uint8_t *)page_entry->page_buf_ptr + offset,
                                    page_buf->page_size - (size_t)offset);

                        
                        H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)
                    } 
                    
                    else if (num_touched_pages > 1 && i == num_touched_pages - 1 &&
                             search_addr < addr + size) {
                        offset = (num_touched_pages - 2) * page_buf->page_size +
                                 (page_buf->page_size - (addr - first_page_addr));

                        H5MM_memcpy((uint8_t *)buf + offset, page_entry->page_buf_ptr,
                                    (size_t)((addr + size) - last_page_addr));

                        
                        H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)
                    } 
                    
                    else {
                        offset = i * page_buf->page_size;

                        H5MM_memcpy((uint8_t *)buf + (i * page_buf->page_size), page_entry->page_buf_ptr,
                                    page_buf->page_size);
                    } 
                }     
                node = H5SL_next(node);
            } 
        }     
    }         
    else {
        
        assert(1 == num_touched_pages || 2 == num_touched_pages);
        for (i = 0; i < num_touched_pages; i++) {
            haddr_t buf_offset;

            
            search_addr = (0 == i ? first_page_addr : last_page_addr);

            
            if (1 == num_touched_pages)
                access_size = size;
            else
                access_size = (0 == i ? (size_t)((first_page_addr + page_buf->page_size) - addr)
                                      : (size - access_size));

            
            page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&search_addr));

            
            if (page_entry) {
                offset     = (0 == i ? addr - page_entry->addr : 0);
                buf_offset = (0 == i ? 0 : size - access_size);

                
                if (offset + access_size > page_buf->page_size)
                    access_size = page_buf->page_size - offset;

                
                H5MM_memcpy((uint8_t *)buf + buf_offset, (uint8_t *)page_entry->page_buf_ptr + offset,
                            access_size);

                
                H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)

                
                if (type == H5FD_MEM_DRAW)
                    page_buf->hits[1]++;
                else
                    page_buf->hits[0]++;
            } 
            
            else {
                void   *new_page_buf = NULL;
                size_t  page_size    = page_buf->page_size;
                haddr_t eoa;

                
                if ((H5SL_count(page_buf->slist_ptr) * page_buf->page_size) >= page_buf->max_size) {
                    htri_t can_make_space;

                    
                    if ((can_make_space = H5PB__make_space(f_sh, page_buf, type)) < 0)
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_NOSPACE, FAIL, "make space in Page buffer Failed");

                    
                    if (0 == can_make_space) {
                        
                        assert(0 == i);

                        
                        if (H5FD_read(file, type, addr, size, buf) < 0)
                            HGOTO_ERROR(H5E_PAGEBUF, H5E_READERROR, FAIL, "driver read request failed");

                        
                        break;
                    } 
                }     

                
                if (NULL == (new_page_buf = H5FL_FAC_MALLOC(page_buf->page_fac)))
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTALLOC, FAIL,
                                "memory allocation failed for page buffer entry");

                

                
                if (HADDR_UNDEF == (eoa = H5F_shared_get_eoa(f_sh, type)))
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "driver get_eoa request failed");

                
                if (search_addr > eoa)
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_BADVALUE, FAIL,
                                "reading an entire page that is outside the file EOA");

                
                if (search_addr + page_size > eoa)
                    page_size = (size_t)(eoa - search_addr);

                
                if (H5FD_read(file, type, search_addr, page_size, new_page_buf) < 0)
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_READERROR, FAIL, "driver read request failed");

                
                offset     = (0 == i ? addr - search_addr : 0);
                buf_offset = (0 == i ? 0 : size - access_size);

                
                if (offset + access_size > page_buf->page_size)
                    access_size = page_buf->page_size - offset;

                H5MM_memcpy((uint8_t *)buf + buf_offset, (uint8_t *)new_page_buf + offset, access_size);

                
                if (NULL == (page_entry = H5FL_CALLOC(H5PB_entry_t)))
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_NOSPACE, FAIL, "memory allocation failed");

                page_entry->page_buf_ptr = new_page_buf;
                page_entry->addr         = search_addr;
                page_entry->type         = (H5F_mem_page_t)type;
                page_entry->is_dirty     = false;

                
                if (H5PB__insert_entry(page_buf, page_entry) < 0)
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTSET, FAIL, "error inserting new page in page buffer");

                
                if (type == H5FD_MEM_DRAW)
                    page_buf->misses[1]++;
                else
                    page_buf->misses[0]++;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_write(H5F_shared_t *f_sh, H5FD_mem_t type, haddr_t addr, size_t size, const void *buf)
{
    H5PB_t       *page_buf;                        
    H5PB_entry_t *page_entry;                      
    H5FD_t       *file;                            
    haddr_t       first_page_addr, last_page_addr; 
    haddr_t       offset;
    haddr_t       search_addr;       
    hsize_t       num_touched_pages; 
    size_t        access_size = 0;
    bool          bypass_pb   = false; 
    hsize_t       i;                   
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f_sh);

    
    page_buf = f_sh->page_buf;

#ifdef H5_HAVE_PARALLEL
    if (H5F_SHARED_HAS_FEATURE(f_sh, H5FD_FEAT_HAS_MPI)) {
#if 1
        bypass_pb = true;
#else
        
        int mpi_size;

        if ((mpi_size = H5F_shared_mpi_get_size(f_sh)) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "can't retrieve MPI communicator size");
        if (1 != mpi_size)
            bypass_pb = true;
#endif
    } 
#endif

    
    if (NULL == page_buf || size >= page_buf->page_size || bypass_pb) {
        if (H5F__accum_write(f_sh, type, addr, size, buf) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_WRITEERROR, FAIL, "write through metadata accumulator failed");

        
        if (page_buf) {
            if (type == H5FD_MEM_DRAW || type == H5FD_MEM_GHEAP)
                page_buf->bypasses[1]++;
            else
                page_buf->bypasses[0]++;
        } 

        
        if (NULL == page_buf || (size >= page_buf->page_size && H5FD_MEM_DRAW != type) ||
            (bypass_pb && H5FD_MEM_DRAW == type))
            HGOTO_DONE(SUCCEED);

#ifdef H5_HAVE_PARALLEL
        if (bypass_pb) {
            if (H5PB_update_entry(page_buf, addr, size, buf) > 0)
                HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTUPDATE, FAIL, "failed to update PB with metadata cache");
            HGOTO_DONE(SUCCEED);
        } 
#endif
    } 

    
    if (page_buf) {
        if (type == H5FD_MEM_DRAW || type == H5FD_MEM_GHEAP)
            page_buf->accesses[1]++;
        else
            page_buf->accesses[0]++;
    } 

    
    first_page_addr = (addr / page_buf->page_size) * page_buf->page_size;

    
    if (H5FD_MEM_DRAW == type) {
        last_page_addr = (addr + size - 1) / page_buf->page_size * page_buf->page_size;

        
        num_touched_pages =
            (last_page_addr / page_buf->page_size + 1) - (first_page_addr / page_buf->page_size);
        if (first_page_addr == last_page_addr) {
            assert(1 == num_touched_pages);
            last_page_addr = HADDR_UNDEF;
        } 
    }     
    
    else {
        num_touched_pages = 1;
        last_page_addr    = HADDR_UNDEF;
    } 

    
    file = f_sh->lf;

    
    if (H5FD_MEM_DRAW == type && size >= page_buf->page_size) {
        
        for (i = 0; i < num_touched_pages; i++) {
            search_addr = i * page_buf->page_size + first_page_addr;

            
            if (i == 0 && first_page_addr != addr) {
                
                page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&search_addr));
                if (page_entry) {
                    offset = addr - first_page_addr;
                    assert(page_buf->page_size > offset);

                    
                    H5MM_memcpy((uint8_t *)page_entry->page_buf_ptr + offset, buf,
                                page_buf->page_size - (size_t)offset);

                    
                    page_entry->is_dirty = true;
                    H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)
                } 
            }     
            
            else if (num_touched_pages > 1 && i == (num_touched_pages - 1) &&
                     (search_addr + page_buf->page_size) != (addr + size)) {
                assert(search_addr + page_buf->page_size > addr + size);

                
                page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&search_addr));
                if (page_entry) {
                    offset = (num_touched_pages - 2) * page_buf->page_size +
                             (page_buf->page_size - (addr - first_page_addr));

                    
                    H5MM_memcpy(page_entry->page_buf_ptr, (const uint8_t *)buf + offset,
                                (size_t)((addr + size) - last_page_addr));

                    
                    page_entry->is_dirty = true;
                    H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)
                } 
            }     
            
            else {
                page_entry = (H5PB_entry_t *)H5SL_remove(page_buf->slist_ptr, (void *)(&search_addr));
                if (page_entry) {
                    
                    H5PB__REMOVE_LRU(page_buf, page_entry)

                    
                    if (H5F_MEM_PAGE_DRAW == page_entry->type || H5F_MEM_PAGE_GHEAP == page_entry->type)
                        page_buf->raw_count--;
                    else
                        page_buf->meta_count--;

                    
                    page_entry->page_buf_ptr = H5FL_FAC_FREE(page_buf->page_fac, page_entry->page_buf_ptr);
                    page_entry               = H5FL_FREE(H5PB_entry_t, page_entry);
                } 
            }     
        }         
    }             
    else {
        
        assert(1 == num_touched_pages || 2 == num_touched_pages);
        for (i = 0; i < num_touched_pages; i++) {
            haddr_t buf_offset;

            
            search_addr = (0 == i ? first_page_addr : last_page_addr);

            
            if (1 == num_touched_pages)
                access_size = size;
            else
                access_size =
                    (0 == i ? (size_t)(first_page_addr + page_buf->page_size - addr) : (size - access_size));

            
            page_entry = (H5PB_entry_t *)H5SL_search(page_buf->slist_ptr, (void *)(&search_addr));

            
            if (page_entry) {
                offset     = (0 == i ? addr - page_entry->addr : 0);
                buf_offset = (0 == i ? 0 : size - access_size);

                
                H5MM_memcpy((uint8_t *)page_entry->page_buf_ptr + offset, (const uint8_t *)buf + buf_offset,
                            access_size);

                
                page_entry->is_dirty = true;
                H5PB__MOVE_TO_TOP_LRU(page_buf, page_entry)

                
                if (type == H5FD_MEM_DRAW || type == H5FD_MEM_GHEAP)
                    page_buf->hits[1]++;
                else
                    page_buf->hits[0]++;
            } 
            
            else {
                void  *new_page_buf;
                size_t page_size = page_buf->page_size;

                
                if ((H5SL_count(page_buf->slist_ptr) * page_buf->page_size) >= page_buf->max_size) {
                    htri_t can_make_space;

                    
                    if ((can_make_space = H5PB__make_space(f_sh, page_buf, type)) < 0)
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_NOSPACE, FAIL, "make space in Page buffer Failed");

                    
                    if (0 == can_make_space) {
                        assert(0 == i);

                        
                        if (H5FD_write(file, type, addr, size, buf) < 0)
                            HGOTO_ERROR(H5E_PAGEBUF, H5E_WRITEERROR, FAIL, "driver write request failed");

                        
                        break;
                    } 
                }     

                
                if (H5F_ACC_RDWR & H5F_SHARED_INTENT(f_sh))
                    
                    page_entry = (H5PB_entry_t *)H5SL_remove(page_buf->mf_slist_ptr, (void *)(&search_addr));

                
                offset     = (0 == i ? addr - search_addr : 0);
                buf_offset = (0 == i ? 0 : size - access_size);

                
                if (page_entry) {
                    
                    if (NULL == (new_page_buf = H5FL_FAC_MALLOC(page_buf->page_fac)))
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for page buffer entry");
                    memset(new_page_buf, 0, (size_t)offset);
                    memset((uint8_t *)new_page_buf + offset + access_size, 0,
                           page_size - ((size_t)offset + access_size));

                    page_entry->page_buf_ptr = new_page_buf;

                    
                    if (type == H5FD_MEM_DRAW || type == H5FD_MEM_GHEAP)
                        page_buf->hits[1]++;
                    else
                        page_buf->hits[0]++;
                } 
                
                else {
                    haddr_t eoa, eof = HADDR_UNDEF;

                    
                    if (NULL == (new_page_buf = H5FL_FAC_CALLOC(page_buf->page_fac)))
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTALLOC, FAIL,
                                    "memory allocation failed for page buffer entry");

                    
                    if (NULL == (page_entry = H5FL_CALLOC(H5PB_entry_t)))
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTALLOC, FAIL, "memory allocation failed");

                    page_entry->page_buf_ptr = new_page_buf;
                    page_entry->addr         = search_addr;
                    page_entry->type         = (H5F_mem_page_t)type;

                    
                    if (HADDR_UNDEF == (eoa = H5F_shared_get_eoa(f_sh, type)))
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "driver get_eoa request failed");

                    
                    if (search_addr > eoa)
                        HGOTO_ERROR(H5E_PAGEBUF, H5E_BADVALUE, FAIL,
                                    "writing to a page that is outside the file EOA");

                    
                    if (!H5F_SHARED_HAS_FEATURE(f_sh, H5FD_FEAT_HAS_MPI))
                        if (HADDR_UNDEF == (eof = H5FD_get_eof(f_sh->lf, H5FD_MEM_DEFAULT)))
                            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "driver get_eof request failed");

                    
                    if (search_addr + page_size > eoa)
                        page_size = (size_t)(eoa - search_addr);

                    if (search_addr < eof) {
                        if (H5FD_read(file, type, search_addr, page_size, new_page_buf) < 0)
                            HGOTO_ERROR(H5E_PAGEBUF, H5E_READERROR, FAIL, "driver read request failed");

                        
                        if (type == H5FD_MEM_DRAW || type == H5FD_MEM_GHEAP)
                            page_buf->misses[1]++;
                        else
                            page_buf->misses[0]++;
                    } 
                }     

                
                H5MM_memcpy((uint8_t *)new_page_buf + offset, (const uint8_t *)buf + buf_offset, access_size);

                
                page_entry->is_dirty = true;

                
                if (H5PB__insert_entry(page_buf, page_entry) < 0)
                    HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTSET, FAIL, "error inserting new page in page buffer");
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PB_enabled(H5F_shared_t *f_sh, H5FD_mem_t type, bool *enabled)
{
    H5PB_t *page_buf;            
    bool    bypass_pb = false;   
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(f_sh);

    
    page_buf = f_sh->page_buf;

#ifdef H5_HAVE_PARALLEL
    if (H5F_SHARED_HAS_FEATURE(f_sh, H5FD_FEAT_HAS_MPI)) {
#if 1
        bypass_pb = true;
#else
        
        int mpi_size;

        if ((mpi_size = H5F_shared_mpi_get_size(f_sh)) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "can't retrieve MPI communicator size");
        if (1 != mpi_size)
            bypass_pb = true;
#endif
    } 
#endif

    
    if (NULL == page_buf || (bypass_pb && H5FD_MEM_DRAW == type)) {
        
        if (page_buf) {
            assert(type == H5FD_MEM_DRAW);
            page_buf->bypasses[1]++;
        } 

        
        *enabled = false;
    } 
    else
        
        *enabled = true;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PB__insert_entry(H5PB_t *page_buf, H5PB_entry_t *page_entry)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5SL_insert(page_buf->slist_ptr, page_entry, &(page_entry->addr)) < 0)
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTINSERT, FAIL, "can't insert entry in skip list");
    assert(H5SL_count(page_buf->slist_ptr) * page_buf->page_size <= page_buf->max_size);

    
    if (H5F_MEM_PAGE_DRAW == page_entry->type || H5F_MEM_PAGE_GHEAP == page_entry->type)
        page_buf->raw_count++;
    else
        page_buf->meta_count++;

    
    H5PB__INSERT_LRU(page_buf, page_entry)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5PB__make_space(H5F_shared_t *f_sh, H5PB_t *page_buf, H5FD_mem_t inserted_type)
{
    H5PB_entry_t *page_entry;       
    htri_t        ret_value = true; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);
    assert(page_buf);

    
    page_entry = page_buf->LRU_tail_ptr;

    if (H5FD_MEM_DRAW == inserted_type) {
        
        if (0 == page_buf->raw_count && page_buf->min_meta_count == page_buf->meta_count) {
            assert(page_buf->meta_count * page_buf->page_size == page_buf->max_size);
            HGOTO_DONE(false);
        } 

        
        while (1) {
            if (page_entry->prev && H5F_MEM_PAGE_META == page_entry->type &&
                page_buf->min_meta_count >= page_buf->meta_count)
                page_entry = page_entry->prev;
            else
                break;
        } 
    }     
    else {
        
        if (0 == page_buf->meta_count && page_buf->min_raw_count == page_buf->raw_count) {
            assert(page_buf->raw_count * page_buf->page_size == page_buf->max_size);
            HGOTO_DONE(false);
        } 

        
        while (1) {
            if (page_entry->prev &&
                (H5F_MEM_PAGE_DRAW == page_entry->type || H5F_MEM_PAGE_GHEAP == page_entry->type) &&
                page_buf->min_raw_count >= page_buf->raw_count)
                page_entry = page_entry->prev;
            else
                break;
        } 
    }     

    
    if (NULL == H5SL_remove(page_buf->slist_ptr, &(page_entry->addr)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_BADVALUE, FAIL, "Tail Page Entry is not in skip list");

    
    H5PB__REMOVE_LRU(page_buf, page_entry)
    assert(H5SL_count(page_buf->slist_ptr) == page_buf->LRU_list_len);

    
    if (H5F_MEM_PAGE_DRAW == page_entry->type || H5F_MEM_PAGE_GHEAP == page_entry->type)
        page_buf->raw_count--;
    else
        page_buf->meta_count--;

    
    if (page_entry->is_dirty)
        if (H5PB__write_entry(f_sh, page_entry) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_WRITEERROR, FAIL, "file write failed");

    
    if (page_entry->type == H5F_MEM_PAGE_DRAW || H5F_MEM_PAGE_GHEAP == page_entry->type)
        page_buf->evictions[1]++;
    else
        page_buf->evictions[0]++;

    
    page_entry->page_buf_ptr = H5FL_FAC_FREE(page_buf->page_fac, page_entry->page_buf_ptr);
    page_entry               = H5FL_FREE(H5PB_entry_t, page_entry);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PB__write_entry(H5F_shared_t *f_sh, H5PB_entry_t *page_entry)
{
    haddr_t eoa;                 
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f_sh);
    assert(page_entry);

    
    if (HADDR_UNDEF == (eoa = H5F_shared_get_eoa(f_sh, (H5FD_mem_t)page_entry->type)))
        HGOTO_ERROR(H5E_PAGEBUF, H5E_CANTGET, FAIL, "driver get_eoa request failed");

    
    if (page_entry->addr <= eoa) {
        H5FD_t *file; 
        size_t  page_size = f_sh->page_buf->page_size;

        
        if ((page_entry->addr + page_size) > eoa)
            page_size = (size_t)(eoa - page_entry->addr);

        
        file = f_sh->lf;

        if (H5FD_write(file, (H5FD_mem_t)page_entry->type, page_entry->addr, page_size,
                       page_entry->page_buf_ptr) < 0)
            HGOTO_ERROR(H5E_PAGEBUF, H5E_WRITEERROR, FAIL, "file write failed");
    } 

    page_entry->is_dirty = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
