% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/invwavtrans.R
\name{InvWavTransf1D}
\alias{InvWavTransf1D}
\title{Inverse AI wavelet transform for curve of HPD matrices}
\usage{
InvWavTransf1D(D, M0, order = 5, jmax, periodic = FALSE,
  metric = "Riemannian", ...)
}
\arguments{
\item{D}{a list of arrays containing the pyramid of wavelet coefficients, where each array contains the
(\eqn{d,d})-dimensional wavelet coefficients from the coarsest wavelet scale \code{j = 0} up to the finest
wavelet scale \code{j = jmax}. This is the same format as the \code{$D} component given as output by
 \code{\link{WavTransf1D}}.}

\item{M0}{a numeric array containing the midpoint(s) at the coarsest scale \code{j = 0} in the midpoint pyramid.
This is the same format as the \code{$M0} component given as output by \code{\link{WavTransf1D}}.}

\item{order}{an odd integer larger or equal to 1 corresponding to the order of the intrinsic AI refinement scheme,
defaults to \code{order = 5}. Note that if \code{order > 9}, the computational cost
significantly increases as the wavelet transform no longer uses a fast wavelet refinement scheme based
on pre-determined weights.}

\item{jmax}{the maximum scale (resolution) up to which the HPD midpoints (i.e. scaling coefficients) are reconstructed.
If \code{jmax} is not specified it is set equal to the resolution in the finest wavelet scale \code{jmax = length(D)}.}

\item{periodic}{a logical value determining whether the curve of HPD matrices can be reflected at the boundary for
improved wavelet refinement schemes near the boundaries of the domain. This is useful for spectral matrix estimation,
where the spectral matrix is a symmetric and periodic curve in the frequency domain. Defaults to \code{periodic = FALSE}.}

\item{metric}{the metric that the space of HPD matrices is equipped with. The default choice is \code{"Riemannian"},
but this can also be one of: \code{"logEuclidean"}, \code{"Cholesky"}, \code{"rootEuclidean"},
\code{"Euclidean"} or \code{"Riemannian-Rahman"}. See also the Details section below.}

\item{...}{additional arguments for internal use.}
}
\value{
Returns a (\eqn{d, d, m})-dimensional array corresponding to a length \eqn{m} curve of
(\eqn{d,d})-dimensional HPD matrices.
}
\description{
\code{InvWavTransf1D} computes an inverse intrinsic average-interpolation (AI) wavelet
transform mapping an array of coarsest-scale HPD midpoints combined with a pyramid of Hermitian
wavelet coefficients to a curve in the manifold of HPD matrices equipped with a metric specified by the user,
as described in \insertCite{CvS17}{pdSpecEst} and Chapter 3 of \insertCite{C18}{pdSpecEst}. This is
the inverse operation of the function \code{\link{WavTransf1D}}.
}
\details{
The input list of arrays \code{D} and array \code{M0} correspond to a pyramid of wavelet coefficients and
the coarsest-scale HPD midpoints respectively, both are structured in the same way as in the output of
\code{WavTransf1D}. As in the forward AI wavelet transform, if the refinement order is an odd integer smaller or
equal to 9, the function computes the inverse wavelet transform using a fast wavelet refinement scheme based on
weighted intrinsic averages with pre-determined weights as explained in \insertCite{CvS17}{pdSpecEst} and Chapter 3 of
\insertCite{C18}{pdSpecEst}. If the refinement order is an odd integer larger than 9, the wavelet refinement
scheme uses intrinsic polynomial prediction based on Neville's algorithm in the Riemannian manifold (via \code{\link{pdNeville}}).\cr
The function computes the inverse intrinsic AI wavelet transform in the space of HPD matrices equipped with
one of the following metrics: (i) the affine-invariant Riemannian metric (default) as detailed in e.g., \insertCite{B09}{pdSpecEst}[Chapter 6]
or \insertCite{PFA05}{pdSpecEst}; (ii) the log-Euclidean metric, the Euclidean inner product between matrix logarithms;
(iii) the Cholesky metric, the Euclidean inner product between Cholesky decompositions; (iv) the Euclidean metric; or
(v) the root-Euclidean metric. The default choice of metric (affine-invariant Riemannian) satisfies several useful properties
not shared by the other metrics, see \insertCite{CvS17}{pdSpecEst} or \insertCite{C18}{pdSpecEst} for more details. Note that this comes
at the cost of increased computation time in comparison to one of the other metrics.
}
\examples{
P <- rExamples1D(2^8, example = "bumps")
P.wt <- WavTransf1D(P$f) ## forward transform
P.f <- InvWavTransf1D(P.wt$D, P.wt$M0) ## backward transform
all.equal(P.f, P$f)

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{WavTransf1D}}, \code{\link{pdSpecEst1D}}, \code{\link{pdNeville}}
}
