% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rboot.R
\name{rboot.normality}
\alias{rboot.normality}
\title{Bootstrap for JB normality test}
\usage{
rboot.normality(x, n.boot = 1000, n.cores = 1, fix_beta = FALSE)
}
\arguments{
\item{x}{VAR object of class '\code{varx}' or any other 
that will be \link[=as.varx]{coerced} to '\code{varx}'.}

\item{n.boot}{Integer. Number of bootstrap iterations.}

\item{n.cores}{Integer. Number of allocated processor cores.}

\item{fix_beta}{Logical. If \code{TRUE}, the cointegrating vectors \eqn{\beta} 
are fixed over all bootstrap iterations. Kilian and Demiroglu (2000:43) 
suggest this for VECM with known \eqn{\beta}. 
Ignored in case of rank-unrestricted VAR.}
}
\value{
A list of class '\code{rboot}' with elements:
\item{sim}{Array of dimension \eqn{(3 \times (1+K) \times} \code{n.boot}) 
  containing the \code{n.boot} iteration results for 
  \emph{(i)} the JB, skewness and kurtosis test and for
  \emph{(ii)} the multivariate and each univariate test 
  for the \eqn{K} residual time series.}
\item{stats}{Matrix of dimension \eqn{(3 \times (1+K))} containing the empirical test statistics.}
\item{pvals}{Matrix of dimension \eqn{(3 \times (1+K))} containing the \eqn{p}-values.}
\item{varx}{Input VAR object of class '\code{varx}'.}
\item{args_rboot}{List of characters and integers 
  indicating the test and specifications that have been used.}
}
\description{
Bootstraps the distribution of the Jarque-Bera test 
  for individual VAR and VECM as described by Kilian, Demiroglu (2000).
}
\examples{
\donttest{
# select minimal or full example #
is_min = TRUE
n.boot = ifelse(is_min, 50, 5000)

# prepare the data, estimate and test the VAR model #
set.seed(23211)
library("vars")
data("Canada")
exogen = cbind(qtrend=(1:nrow(Canada))^2)  # quadratic trend
R.vars = VAR(Canada, p=2, type="both", exogen=exogen)
R.norm = rboot.normality(x=R.vars, n.boot=n.boot, n.cores=1)

# density plot #
library("ggplot2")
R.data = data.frame(t(R.norm$sim[ , "MULTI", ]))
R.args = list(df=2*R.vars$K)
F.density = ggplot() +
  stat_density(data=R.data, aes(x=JB, color="bootstrap"), geom="line") +
  stat_function(fun=dchisq, args=R.args, n=500, aes(color="theoretical")) +
  labs(x="JB statistic", y="Density", color="Distribution", title=NULL) +
  theme_bw()
plot(F.density)
}

}
\references{
Jarque, C. M. and Bera, A. K. (1987): 
  "A Test for Normality of Observations and Regression Residuals", 
  \emph{International Statistical Review}, 55, pp. 163-172.

Kilian, L. and Demiroglu, U. (2000): 
  "Residual-Based Tests for Normality in Autoregressions: 
  Asymptotic Theory and Simulation Evidence", 
  \emph{Journal of Business and Economic Statistics}, 32, pp. 40-50.
}
\seealso{
\ldots the \code{\link[vars]{normality.test}} by Pfaff (2008) in \strong{vars},
  which relies on theoretical distributions. Just as this asymptotic version,
  the bootstrapped version is computed by using the residuals that are 
  standardized by a Cholesky decomposition of the residual covariance matrix. 
  Therefore, the results of the multivariate test depend on the ordering 
  of the variables in the VAR model.
}
