#' @title Get Valid Modalities
#' @description Returns a vector of possible output modalities for the supported model.
#' These modalities represent different types of gene expression data that can be
#' generated by the Synthesize Bio API.
#'
#' @return A character vector containing the valid modality strings.
#' @examples
#' # Get all supported modalities
#' modalities <- get_valid_modalities()
#' print(modalities)
#'
#' # Check if a specific modality is supported
#' "bulk" %in% get_valid_modalities()
#' @export
get_valid_modalities <- function() {
  unlist(MODEL_MODALITIES[[RELEASE_NUMBER]])
}

#' @title Get Valid Modes
#' @description Returns a vector of possible modes for the supported model.
#' These modes represent different types of gene expression data that can be
#' generated by the Synthesize Bio API.
#' @param modality Character string specifying the modality. Either "bulk" or
#' "single-cell".
#' @return A character vector containing the valid mode strings.
#' @examples
#' # Get all supported modes
#' modes <- get_valid_modes(modality = "bulk")
#' print(modes)
#'
#' # Check if a specific mode is supported
#' "sample generation" %in% get_valid_modes(modality = "single-cell")
#' @export
get_valid_modes <- function(modality) {
  if (is.null(modality)) stop("modality is required input")
  unlist(MODEL_MODES[[RELEASE_NUMBER]][[modality]])
}


#' @title Get Valid Query Example
#' @description Generates a sample query for prediction and validation for the model.
#' This function provides an example query structure that can be modified for specific needs.
#' The sample query contains two example inputs: one for a cell line with CRISPR perturbation
#' and another for a primary tissue sample with disease information.
#'
#' The returned query includes:
#' \itemize{
#'   \item \code{modality}: The data type ("bulk" or "single-cell")
#'   \item \code{mode}: The prediction mode ("sample generation" for bulk, "mean estimation" for single-cell)
#'   \item \code{inputs}: A list of biological conditions with metadata and num_samples
#' }
#'
#' Optional fields can be added to control model behavior:
#' \itemize{
#'   \item \code{total_count}: Library size
#'   \item \code{deterministic_latents}: If TRUE, produces deterministic outputs
#'   \item \code{seed}: Random seed for reproducibility
#' }
#'
#' @param modality Character string specifying the modality. Either "bulk" or "single-cell".
#'        Default is "bulk".
#' @return A list representing a valid query structure.
#' @examples
#' # Get a sample query for bulk RNA-seq
#' query <- get_valid_query()
#'
#' # Get a sample query for single-cell RNA-seq
#' query_sc <- get_valid_query(modality = "single-cell")
#'
#' # Modify the query structure
#' query$inputs[[1]]$num_samples <- 10
#'
#' # Add optional parameters
#' query$deterministic_latents <- TRUE
#' query$total_count <- 5000000
#' @export
get_valid_query <- function(modality = "bulk") {
  # Build metadata based on modality
  if (modality == "single-cell") {
    # Single-cell metadata (no sample_type field)
    metadata1 <- list(
      cell_type_ontology_id = "CL:0000786",
      tissue_ontology_id = "UBERON:0001155",
      sex = "male"
    )
    metadata2 <- list(
      cell_type_ontology_id = "CL:0000763",
      tissue_ontology_id = "UBERON:0001155",
      sex = "male"
    )
  } else {
    # Bulk metadata (includes sample_type)
    metadata1 <- list(
      cell_type_ontology_id = "CL:0000786",
      tissue_ontology_id = "UBERON:0001155",
      sex = "male",
      sample_type = "primary tissue"
    )
    metadata2 <- list(
      cell_type_ontology_id = "CL:0000763",
      tissue_ontology_id = "UBERON:0001155",
      sex = "male",
      sample_type = "primary tissue"
    )
  }

  list(
    modality = modality,
    mode = "mean estimation",
    seed = 11,
    inputs = list(
      list(
        metadata = metadata1,
        num_samples = 5
      ),
      list(
        metadata = metadata2,
        num_samples = 5
      )
    )
  )
}


#' @title Validate Query Structure
#' @description Validates the structure and contents of the query based on the model.
#' This function checks that the query is a list and contains all required keys.
#'
#' @param query A list containing the query data.
#' @return Invisibly returns TRUE if validation passes.
#' Throws an error If the query structure is invalid or missing required keys.
#' @examples
#' # Create a valid query
#' query <- get_valid_query()
#' validate_query(query) # Passes validation
#'
#' # Example with invalid query (missing required key)
#' \dontrun{
#' invalid_query <- list(inputs = list(), mode = "mean estimation")
#' validate_query(invalid_query) # Throws error for missing modality
#' }
#' @export
validate_query <- function(query) {
  if (!is.list(query)) {
    stop(paste0("Expected `query` to be a list, but got ", class(query)))
  }

  required_keys <- c("inputs", "mode", "modality")

  missing_keys <- setdiff(required_keys, names(query))
  if (length(missing_keys) > 0) {
    stop(paste0(
      "Missing required keys in query: ", paste(missing_keys, collapse = ", "), ". ",
      "Use `get_valid_query()` to get an example."
    ))
  }

  invisible(TRUE)
}

#' @title Validate Query Modality
#' @description Validates that the modality specified in the query is allowed for the model.
#' This function checks that the `modality` value is one of the supported modalities.
#'
#' @param query A list containing the query data.
#' @return Invisibly returns TRUE if validation passes.
#' Throws an error If the modality key is missing or if the selected modality is not allowed.
#' @examples
#' # Create a valid query
#' query <- get_valid_query()
#' validate_modality(query) # Passes validation
#'
#' # Example with invalid modality
#' \dontrun{
#' invalid_query <- get_valid_query()
#' invalid_query$modality <- "unsupported_modality"
#' validate_modality(invalid_query) # Throws error for invalid modality
#' }
#' @export
validate_modality <- function(query) {
  allowed_modalities <- get_valid_modalities()

  if (!("modality" %in% names(query))) {
    stop(paste0("Query requires `modality` key."))
  }

  selected_modality <- query[["modality"]]

  if (!(selected_modality %in% allowed_modalities)) {
    stop(paste0(
      "Invalid modality '", selected_modality, "'. ",
      "Allowed modalities: ", paste(allowed_modalities, collapse = ", ")
    ))
  }

  invisible(TRUE)
}

validate_mode <- function(query) {
  allowed_modes <- get_valid_modes(query[["modality"]])

  if (!("mode" %in% names(query))) {
    stop(paste0("Query requires `mode` key."))
  }

  selected_mode <- query[["mode"]]

  if (!(selected_mode %in% allowed_modes)) {
    stop(paste0(
      "Invalid mode '", selected_mode, "'. ",
      "Allowed modes: ", paste(allowed_modes, collapse = ", ")
    ))
  }

  invisible(TRUE)
}

#' @title Resolve API Slug
#' @description Internal function to resolve the API slug based on modality
#' @param modality The modality string ("bulk" or "single-cell")
#' @return The API slug string
#' @keywords internal
resolve_api_slug <- function(modality) {
  if (modality == "single-cell") {
    return("gem-1-sc")
  }
  if (modality == "bulk") {
    return("gem-1-bulk")
  }
  return("")
}

#' @title Start Model Query
#' @description Internal function to start an async model query
#' @param api_base_url The base URL for the API
#' @param api_slug The API slug for the specific model
#' @param query The query list
#' @return The model query ID
#' @importFrom httr POST add_headers content status_code timeout
#' @importFrom jsonlite toJSON fromJSON
#' @keywords internal
start_model_query <- function(api_base_url, api_slug, query) {
  url <- paste0(api_base_url, "/api/models/", api_slug, "/predict")
  query_json <- toJSON(query, auto_unbox = TRUE)

  response <- tryCatch(
    {
      POST(
        url = url,
        add_headers(
          Accept = "application/json",
          Authorization = paste("Bearer", Sys.getenv("SYNTHESIZE_API_KEY")),
          `Content-Type` = "application/json"
        ),
        body = query_json,
        encode = "json",
        timeout(DEFAULT_TIMEOUT)
      )
    },
    error = function(e) {
      stop(paste0("Predict request failed due to a network issue: ", e$message))
    }
  )

  if (status_code(response) >= 400) {
    stop(paste0(
      "Predict request failed with status ",
      status_code(response), ": ", content(response, "text")
    ))
  }

  parsed_content <- tryCatch(
    {
      fromJSON(content(response, "text"), simplifyDataFrame = TRUE)
    },
    error = function(e) {
      stop(paste0("Failed to decode JSON from predict response: ", e$message))
    }
  )

  if (!is.list(parsed_content) || is.null(parsed_content$modelQueryId)) {
    stop(paste0(
      "Unexpected response from predict endpoint: ",
      paste(names(parsed_content), collapse = ", ")
    ))
  }

  return(as.character(parsed_content$modelQueryId))
}

#' @title Poll Model Query
#' @description Internal function to poll the status endpoint until ready/failed or timeout
#' @param api_base_url The base URL for the API
#' @param model_query_id The model query ID to poll
#' @param poll_interval Seconds between polling attempts
#' @param timeout_seconds Maximum total seconds to wait
#' @return A list with status and payload
#' @importFrom httr GET add_headers content status_code timeout
#' @importFrom jsonlite fromJSON
#' @keywords internal
poll_model_query <- function(api_base_url, model_query_id, poll_interval, timeout_seconds) {
  start_time <- Sys.time()
  status_url <- paste0(api_base_url, "/api/model-queries/", model_query_id, "/status")
  last_payload <- list()

  while (TRUE) {
    response <- tryCatch(
      {
        GET(
          url = status_url,
          add_headers(
            Accept = "application/json",
            Authorization = paste("Bearer", Sys.getenv("SYNTHESIZE_API_KEY"))
          ),
          timeout(DEFAULT_TIMEOUT)
        )
      },
      error = function(e) {
        stop(paste0("Status request failed due to a network issue: ", e$message))
      }
    )

    if (status_code(response) >= 400) {
      stop(paste0(
        "Status request failed with status ",
        status_code(response), ": ", content(response, "text")
      ))
    }

    payload <- tryCatch(
      {
        fromJSON(content(response, "text"), simplifyDataFrame = TRUE)
      },
      error = function(e) {
        stop(paste0("Failed to decode JSON from status response: ", e$message))
      }
    )

    if (!is.list(payload) || is.null(payload$status)) {
      stop(paste0(
        "Unexpected status response: ",
        paste(names(payload), collapse = ", ")
      ))
    }

    status <- as.character(payload$status)
    last_payload <- payload

    if (status %in% c("ready", "failed")) {
      return(list(status = status, payload = payload))
    }

    elapsed <- as.numeric(difftime(Sys.time(), start_time, units = "secs"))
    if (elapsed > timeout_seconds) {
      return(list(status = status, payload = last_payload))
    }

    Sys.sleep(max(1, as.integer(poll_interval)))
  }
}

#' @title Get JSON from URL
#' @description Internal function to fetch JSON from a URL
#' @param url The URL to fetch from
#' @return The parsed JSON content
#' @importFrom httr GET content status_code timeout
#' @importFrom jsonlite fromJSON
#' @keywords internal
get_json <- function(url) {
  response <- tryCatch(
    {
      GET(url, timeout(DEFAULT_TIMEOUT))
    },
    error = function(e) {
      stop(paste0("Failed to fetch download URL due to a network issue: ", e$message))
    }
  )

  if (status_code(response) >= 400) {
    stop(paste0(
      "Download URL fetch failed with status ",
      status_code(response), ": ", content(response, "text")
    ))
  }

  parsed_content <- tryCatch(
    {
      fromJSON(
        content(response, "text", encoding = "UTF-8"),
        simplifyDataFrame = TRUE
      )
    },
    error = function(e) {
      stop(paste0("Failed to decode JSON from download URL response: ", e$message))
    }
  )

  return(parsed_content)
}


#' @title Predict Gene Expression
#' @description Sends a query to the Synthesize Bio API for prediction
#' and retrieves gene expression samples. This function validates the query, sends it
#' to the API, and processes the response into usable data frames.
#'
#' @param query A list representing the query data to send to the API.
#'        Use `get_valid_query()` to generate an example. The query supports additional
#'        optional fields:
#'        \itemize{
#'          \item `total_count` (integer): Library size used when converting predicted log CPM
#'                back to raw counts. Higher values scale counts up proportionally.
#'          \item `deterministic_latents` (logical): If TRUE, the model uses the mean of each
#'                latent distribution instead of sampling, producing deterministic outputs for
#'                the same inputs. Useful for reproducibility.
#'          \item `seed` (integer): Random seed for reproducibility.
#'        }
#' @param as_counts Logical, if FALSE, transforms the predicted expression counts
#'        into logCPM (default is TRUE, returning raw counts).
#' @param api_base_url The base URL for the API server. Default is API_BASE_URL.
#' @param poll_interval_seconds Seconds between polling attempts of the status endpoint.
#'        Default is DEFAULT_POLL_INTERVAL_SECONDS (2).
#' @param poll_timeout_seconds Maximum total seconds to wait before timing out.
#'        Default is DEFAULT_POLL_TIMEOUT_SECONDS (900 = 15 minutes).
#' @param return_download_url Logical, if TRUE, returns a list containing the signed
#'        download URL instead of parsing into data frames. Default is FALSE.
#' @return A list. If `return_download_url` is `FALSE` (default), the list contains
#'         two data frames: `metadata` and `expression`. If `TRUE`, the list
#'         contains `download_url` and empty `metadata` and `expression` data frames.
#' @importFrom httr POST GET add_headers content http_status status_code timeout
#' @importFrom jsonlite toJSON fromJSON
#' @examples
#' # Set your API key (in practice, use a more secure method)
#' \dontrun{
#'
#' # To start using rsynthbio, first you need to have an account with synthesize.bio.
#' # Go here to create one: https://app.synthesize.bio/
#'
#' set_synthesize_token()
#'
#' # Create a query
#' query <- get_valid_query()
#'
#' # Request raw counts
#' result <- predict_query(query, as_counts = TRUE)
#'
#' # Access the results
#' metadata <- result$metadata
#' expression <- result$expression
#'
#' # Request log CPM transformed data
#' log_result <- predict_query(query, as_counts = FALSE)
#' log_expression <- log_result$expression
#'
#' # Explore the top expressed genes in the first sample
#' head(sort(expression[1, ], decreasing = TRUE))
#'
#' # Use deterministic latents for reproducible results
#' query$deterministic_latents <- TRUE
#' result_det <- predict_query(query)
#'
#' # Specify a custom total count (library size)
#' query$total_count <- 5000000
#' result_custom <- predict_query(query)
#' }
#' @export
predict_query <- function(query,
                          as_counts = TRUE,
                          api_base_url = API_BASE_URL,
                          poll_interval_seconds = DEFAULT_POLL_INTERVAL_SECONDS,
                          poll_timeout_seconds = DEFAULT_POLL_TIMEOUT_SECONDS,
                          return_download_url = FALSE) {
  if (!has_synthesize_token()) {
    stop("Please set your API key for Synthesize Bio using set_synthesize_token()")
  }

  # Validate base URL
  if (!grepl("^https?://", api_base_url)) {
    stop(
      paste0(
        "Invalid api_base_url: ",
        api_base_url,
        ". Must start with http:// or https://"
      )
    )
  }

  # Validate the query
  validate_query(query)
  validate_modality(query)
  validate_mode(query)

  modality <- query$modality

  # Add source field for reporting
  query$source <- "rsynthbio"

  # Resolve internal API slug based on modality
  api_slug <- resolve_api_slug(modality)

  # Start async query
  model_query_id <- start_model_query(
    api_base_url = api_base_url,
    api_slug = api_slug,
    query = query
  )

  # Poll for completion
  poll_result <- poll_model_query(
    api_base_url = api_base_url,
    model_query_id = model_query_id,
    poll_interval = poll_interval_seconds,
    timeout_seconds = poll_timeout_seconds
  )

  status <- poll_result$status
  payload <- poll_result$payload

  if (status == "failed") {
    # payload contains message if available
    err_msg <- if (is.list(payload) && !is.null(payload$message)) {
      payload$message
    } else {
      NULL
    }
    stop(paste0(
      "Model query failed. ",
      if (!is.null(err_msg)) err_msg else "No error message provided."
    ))
  }

  if (status != "ready") {
    stop(paste0(
      "Model query did not complete in time (status=", status, "). ",
      "Consider increasing poll_timeout_seconds."
    ))
  }

  # When ready, payload should contain a signed downloadUrl to the final JSON
  download_url <- if (is.list(payload) && !is.null(payload$downloadUrl)) {
    payload$downloadUrl
  } else {
    NULL
  }
  if (is.null(download_url)) {
    stop("Response missing downloadUrl when status=ready")
  }

  if (return_download_url) {
    # Caller wants the URL only; return in a structured payload
    return(list(
      metadata = data.frame(),
      expression = data.frame(),
      download_url = download_url
    ))
  }

  # Fetch the final results JSON and transform to data frames
  final_json <- get_json(download_url)

  result <- extract_expression_data(final_json, as_counts = as_counts)

  return(result)
}
