% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/approximate_algorithm.R
\name{approx_horseshoe}
\alias{approx_horseshoe}
\title{Run approximate MCMC algorithm for horseshoe prior}
\usage{
approx_horseshoe(
  y,
  X,
  burn = 1000,
  iter = 5000,
  auto.threshold = TRUE,
  tau = 1,
  s = 0.8,
  sigma2 = 1,
  alpha = 0.05,
  ...
)
}
\arguments{
\item{y}{Response vector, \eqn{y \in \mathbb{R}^{N}}.}

\item{X}{Design matrix, \eqn{X \in \mathbb{R}^{N \times p}}.}

\item{burn}{Number of burn-in samples. The default is 1000.}

\item{iter}{Number of samples to be drawn from the posterior. The default is
5000.}

\item{auto.threshold}{Argument to set whether to use the the adaptive
threshold selection method.}

\item{tau}{Initial value of the global shrinkage parameter \eqn{\tau} when
starting the algorithm. The default is 1.}

\item{s}{\eqn{s^{2}} is the variance of tau's MH proposal distribution.
0.8 is a good default. If set to 0, the algorithm proceeds by fixing the
global shrinkage parameter \eqn{\tau} to the initial setting value.}

\item{sigma2}{Initial value of error variance \eqn{\sigma^{2}}. The default
is 1.}

\item{alpha}{\eqn{100(1-\alpha)\%} credible interval setting argument.}

\item{...}{There are additional arguments \emph{threshold}, \emph{a}, \emph{b}, \emph{w}, \emph{t},
\emph{p0}, and \emph{p1}.
\emph{threshold} is used when auto.threshold=FALSE is selected and threshold is
set directly. The default value is \eqn{threshold = 1/p}. \emph{a} and \emph{b}
are arguments of the internal rejection sampler function,
and the default values
are \eqn{a = 1/5,\ b = 10}. \emph{w} is the argument of the prior for
\eqn{\sigma^{2}}, and the default value is \eqn{w = 1}. \emph{t}, \emph{p0}, and \emph{p1}
are arguments of the adaptive threshold selection method, and the default
values are \eqn{t = 10,\ p0 = 0,\ p1 = -4.6 \times 10^{-4}}.}
}
\value{
\item{BetaHat}{Posterior mean of \eqn{\beta}.}
\item{LeftCI}{Lower bound of \eqn{100(1-\alpha)\%} credible interval for
\eqn{\beta}.}
\item{RightCI}{Upper bound of \eqn{100(1-\alpha)\%} credible interval for
\eqn{\beta}.}
\item{Sigma2Hat}{Posterior mean of \eqn{\sigma^{2}}.}
\item{TauHat}{Posterior mean of \eqn{\tau}.}
\item{LambdaHat}{Posterior mean of \eqn{\lambda_{j},\ j=1,2,...p.}.}
\item{ActiveMean}{Average number of elements in the active set per iteration
in this algorithm.}
\item{BetaSamples}{Posterior samples of \eqn{\beta}.}
\item{LambdaSamples}{Posterior samples of local shrinkage parameters.}
\item{TauSamples}{Posterior samples of global shrinkage parameter.}
\item{Sigma2Samples}{Posterior samples of \eqn{sigma^{2}}.}
\item{ActiveSet}{\eqn{\mathbb{R}^{iter \times p}} Matrix indicating active
elements as 1 and non-active elements as 0 per iteration of the MCMC
algorithm.}
}
\description{
The approximate MCMC algorithm for the horseshoe prior
}
\details{
This function implements the approximate algorithm introduced in Section
2.2 of Johndrow et al. (2020) and the method proposed in this package, which
improves computation speed when p >> N. The approximate algorithm introduces
a threshold and uses only a portion of the total \eqn{p} columns for matrix
multiplication, reducing the computational cost compared to the existing
MCMC algorithms for the horseshoe prior. The "auto.threshold" argument
determines whether the threshold used in the algorithm will be selected by
the adaptive method proposed in this package. For more information,
browseVignettes("Mhorseshoe").
}
\examples{
# Making simulation data.
set.seed(123)
N <- 200
p <- 100
true_beta <- c(rep(1, 10), rep(0, 90))

X <- matrix(1, nrow = N, ncol = p) # Design matrix X.
for (i in 1:p) {
  X[, i] <- stats::rnorm(N, mean = 0, sd = 1)
}

y <- vector(mode = "numeric", length = N) # Response variable y.
e <- rnorm(N, mean = 0, sd = 2) # error term e.
for (i in 1:10) {
  y <- y + true_beta[i] * X[, i]
}
y <- y + e

# Run with auto.threshold set to TRUE
result1 <- approx_horseshoe(y, X, burn = 0, iter = 100,
                            auto.threshold = TRUE)

# Run with fixed custom threshold
result2 <- approx_horseshoe(y, X, burn = 0, iter = 100,
                            auto.threshold = FALSE, threshold = 1/(5 * p))

# posterior mean
betahat <- result1$BetaHat

# Lower bound of the 95\% credible interval
leftCI <- result1$LeftCI

# Upper bound of the 95\% credible interval
RightCI <- result1$RightCI

}
\references{
Johndrow, J., Orenstein, P., & Bhattacharya, A. (2020).
Scalable Approximate MCMC Algorithms for the Horseshoe Prior. In Journal
of Machine Learning Research, 21, 1-61.
}
