\name{Reshaping to a crosstable}
\alias{db2cross}
\title{
  Converts a dataset from database-format to a cross table
}
\description{
  Reshapes data arranged in 3 columns to a ``crosstable'' matrix.
}

\usage{
db2cross (input, row = 1, col = 2, value = 3, subset = NULL,
  df.row = NA, df.col = NA, out.row = NA, out.col = NA, 
  full.out = FALSE) 
}

\arguments{
  \item{input }{A \code{matrix} in \emph{database} format, (x,y,z) .
  }
  \item{row }{Number or name of the column in \code{input}, to be used as 
    rows in the result.
  }
  \item{col }{Number or name of the column in \code{input}, to be used as 
    columns in the result.
  }
  \item{value }{Number or name of the column in \code{input}, to be used as 
    values in the result.
  }
  \item{subset }{Logical expression indicating elements or rows to keep; 
    missing values are taken as \code{FALSE}
  }
  \item{df.row, df.col }{Maximal distance in row and column values that should 
    be considered the same. The default is to use each unique row or column 
    value in \code{input} as a row or column value in the crosstable.
    Overrruled when \code{out.row} or \code{out.col} are defined.
  }
  \item{out.row, out.col }{Values of rows and columns to be used in the cross table. 
    The default is to use each unique row or column value in \code{input} as a 
    row or column value in the crosstable. Each value in \code{input} is mapped to
    \code{out.row} and \code{out.col} to which it is closest.
    Overrrules \code{df.row} or \code{df.col}.
  }
  \item{full.out }{If \code{TRUE}, will also output how the input values were
    mapped to the output values. This is only relevant if either of \code{df.row}, 
    \code{df.col}, \code{out.row} or \code{out.col} is not \code{NULL}.
  }
}
\value{
  a list containing:
  \item{x }{ The values of the \emph{rows}.    }
  \item{y }{ The values of the \emph{columns}. }
  \item{z }{ The crosstable, a matrix.} 
  
  and if \code{full.out = TRUE} also 
  \item{map }{ The mapping of the \emph{x}  and \emph{y} values, consisting of 
    \code{var.input}, \code{factor}, \code{var.output}, with the original values, 
    how they are mapped, and the resulting values respectively.}
}
\author{
  Karline Soetaert <karline.soetaert@nioz.nl>
}

\examples{
## =======================================================================
## test the function on a small data set
## =======================================================================

 df3 <- data.frame(school = rep(c("a","b","c"), each = 4), 
                   class = rep(9:10, 6),
                   time = rep(c(1,1,2,2), 3),  
                   score = rnorm(12))
 head(df3)
 db2cross(df3, val = 4)

## =======================================================================
## Defining the output rows
## =======================================================================
Samples <- data.frame(time = c(1, 1.1, 1.2, 2, 2.1, 2.2, 4, 4.1, 4.2),
                      var = rep(c("O2", "NO3", "NH3"), 3), 
                      val = 1:9)
Samples

db2cross(Samples)
db2cross(Samples, df.row = 0.5)
db2cross(Samples, out.row = c(1, 2, 4))
db2cross(Samples, out.row = 1:4)

## =======================================================================
## A larger dataset; requires OceanView.Data
## =======================================================================
\dontrun{
 data (pp.aug2009.db)
 crosstab <- db2cross(pp.aug2009.db)
 crosstab$z[crosstab$z>1000] <- 1000
 crosstab$z[crosstab$z<0]    <- NA

 image2D(z = crosstab$z, x = crosstab$x, y = crosstab$y,
       main = "primary production august 2009 mgC/m2/d", 
       NAcol = "black")
 }
}
\details{
 Uses a simple fortran function.
 
 rows and columns are generated by the \code{unique values} in each 
 x- and y-column.
} 
\seealso{
  \link{reshape}, the official (slow) R-function

  \link{remap} to remap a matrix or array to higher or lower resolution
}
\keyword{utility}

