#' Apply Guidelines for Reporting About Network Data (GRAND) to an igraph object
#'
#' The `grand` function stores characteristics about the graph as graph attributes that can be summarized in a narrative
#'    using the [grand.text()] or a table using [grand.table()].
#'
#' @param G An \code{\link{igraph}} object, with weights/signs (if present) stored in `E(G)$weight`
#' @param interactive boolean: Should GRAND run interactively?
#' @param name string: Name of the network
#' @param doi string: DOI associated with the data
#' @param url string: Link to data
#' @param mode string: Mode of data collection
#' @param year numeric: Year in which data was collected
#' @param vertex1 string: Entity represented by vertices
#' @param vertex2 string: Entity represented by vertices
#' @param vertex1.total numeric: Number of entities in the network's boundary
#' @param vertex2.total numeric: Number of entities in the network's boundary
#' @param edge.pos string: Relationship represented by (positive) edges
#' @param edge.neg string: Relationship represented by negative edges
#' @param weight string: What the edge weights represent
#' @param measure string: Scale on which edge weights are measured
#' @param topology string: Vector of topological metrics to be computed in GRAND summaries
#'
#' @details
#' The interactive mode (default) asks the user a series of questions based on the igraph object,
#'    while non-interactive mode allows the user to directly supply the relevant attributes.
#'
#' **Data**
#'
#' The first set of interactive questions ask about the data as a whole:
#' * _name_ - This should usually be specified ending with the word "network" or "data"
#'     (e.g. "Florentine Families Network" or "Airline Traffic Data").
#' * _doi_ - DOI for a manuscript describing the data.
#' * _url_ - Link to a copy of the data.
#' * Data collection _mode_ - This describes how the data was collected or generated. Chose
#'     one of the available options (Survey, Interview, Sensor, Observation, Archival, or
#'    Simulation) or choose `Other` to enter something else.
#' * _year_ - In what year were the data collected?
#'
#' **Nodes**
#'
#' The second set of interactive questions ask about the nodes/vertices:
#' * _vertex1_ (and in bipartite graphs, _vertex2_) - What type of entity do the nodes/vertices represent?
#'     This should be specified as a plural noun (e.g., "People").
#' * _vertex1.total_ (and in bipartite graphs, _vertex2.total_) - Networks often have an
#'     externally-defined boundary that determines which nodes/vertices should be included,
#'     even if some are missing from the network. These ask about the total number of nodes/vertices
#'     inside the boundary (if one exists) and are used to compute rates of missingness.
#'
#' **Edges**
#'
#' The third set of interactive questions ask about the edges:
#' * _edge.pos_ (and in signed graphs, _edge.neg_) - What type of relationship do the edges represent? This
#'     should be specified as a plural noun (e.g., "Friendships").
#' * _weight_ - What do the edge weights represent? Choose one of the available options (Frequency, Intensity,
#'     Multiplexity, or Valence) or choose `Other` to enter something else.
#' * _measure_ - How are the edge weights measured? Choose one of the available options (Continuous, Count,
#'     Ordinal, or Categorical) or choose `Other` to enter something else.
#'
#' **Topology**
#'
#' The final set of interactive questions ask about relevant topological characteristics. You may choose to
#'    (1) use the defaults for this network type, (2) choose characteristics from a list, (3) compute all
#'    available characteristics, or (4) compute no characteristics. For comparability and to ensure they are
#'    well-defined, all characteristics are computed on an undirected and unweighted version of `G` using
#'    existing `igraph` functions. Available topological characteristics include:
#' * _clustering coefficient_ - Computed using `transitivity(G, type = "localaverage")`
#' * _degree centralization_ - Computed using `centr_degree(G)$centralization`
#' * _degree distribution_ - Computed using `fit_power_law(degree(G), implementation = "plfit")`
#' * _density_ - Computed using `edge_density(G)`
#' * _diameter_ - Computed using `diameter(G)`
#' * _efficiency_ - Computed using `global_efficiency(G)`
#' * _mean degree_ - Computed using `mean(degree(G))`
#' * _modularity_ - Computed from a partition generated by `cluster_leiden(G, objective_function = "modularity")`
#' * _number of communities_ - Computed from a partition generated by `cluster_leiden(G, objective_function = "modularity")`
#' * _number of components_ - Computed using `count_components(G)`
#' * _transitivity_ - Computed using `transitivity(G, type = "global")`
#' * _structural balance_ - Computed using the triangle index
#'
#' @return An \code{\link{igraph}} object
#' @export
#'
#' @examples
#' data(airport)  #Load example data
#' airport <- grand(airport)  #Apply GRAND interactively
#' airport <- grand(airport, interactive = FALSE, #Apply GRAND non-interactively
#'                  vertex1 = "Airports",
#'                  vertex1.total = 382,
#'                  edge.pos = "Routes",
#'                  weight = "Passengers",
#'                  measure = "Count",
#'                  mode = "Archival",
#'                  year = "2019",
#'                  topology = c("clustering coefficient", "mean path length", "degree distribution"))
grand <- function(G, interactive = TRUE,
                  name = NA,
                  doi = NA,
                  url = NA,
                  vertex1 = NULL,
                  vertex2 = NULL,
                  vertex1.total = 0,
                  vertex2.total = 0,
                  edge.pos = NULL,
                  edge.neg = NULL,
                  weight = NULL,
                  measure = NULL,
                  mode = NULL,
                  year = NULL,
                  topology = character()) {

  #### Check inputs ####
  if (!methods::is(G, "igraph")) {stop("The input must be an igraph object")}
  if (!igraph::is_simple(G)) {stop("The graph must be simple (i.e., no loops or multi-edges)")}

  #### Check if interactive session ####
  if (!interactive() & interactive == TRUE) {
    warning("When `interactive = TRUE`, grand() requires that R is running interactively")
    return(G)
  }

  #### Begin interactive mode ####
  if (interactive) {
  if ("grand" %in% names(igraph::get.graph.attribute(G))) {
    overwrite <- scan2("This graph already contains a GRAND attribute. Do you want to overwrite (Y/N)?", type = c("Y", "N", "y", "n"))
    if (overwrite == "N" | overwrite == "n") {return(G)}
  }

  #### Identify graph characteristics ####
  directed <- igraph::is_directed(G)
  bipartite <- igraph::is_bipartite(G)
  weighted <- "weight" %in% names(igraph::get.edge.attribute(G))
  if (weighted) {signed <- all(igraph::E(G)$weight %in% c(-1,1))} else {signed <- FALSE}

  #### Data ####
  G$grand$name <- scan2("What is the name of this network (enter NA if unnamed)?", type = "character")

  G$grand$doi <- scan2("What is the DOI for the manuscript describing this network (enter NA if none)?", type = "character")

  G$grand$url <- scan2("What is the URL where these data are available for download (enter NA if none)?", type = "character")

  G$grand$mode <- menu2(c("Survey", "Interview", "Sensor", "Observation", "Archival", "Simulation", "Other"), title = "How were these data collected or generated?")
  if (G$grand$mode == "Other") {G$grand$mode <- scan2("Please describe how these data were collected or generated", type = "character")
  }

  G$grand$year <- scan2("In what year were these data collected?", type = "integer")

  #### Nodes ####
  #Definition
  if (!bipartite) {
    G$grand$vertex1 <- scan2(cat("This network contains", igraph::gorder(G), "nodes. What type of entity do these represent (e.g., people)?"), type = "character")
  }

  if (bipartite) {
    G$grand$vertex1 <- scan2(cat("This network contains", sum(igraph::V(G)$type==FALSE), "top nodes. What type of entity do these represent (e.g., people)?"), type = "character")
    G$grand$vertex2 <- scan2(cat("This network contains", sum(igraph::V(G)$type==TRUE), "bottom nodes. What type of entity do these represent (e.g., people)?"), type = "character")
  }

  #### Boundary ####
  boundary <- scan2("Does this network have a theoretically- or naturalistically-defined boundary (Y/N)", type = c("Y", "N", "y", "n"))
  if (boundary %in% c("N", "n")) {
    G$grand$vertex1.total <- 0
    G$grand$vertex2.total <- 0
  }
  if (boundary %in% c("Y", "y") & !bipartite) {
    G$grand$vertex1.total <- scan2(cat("How many", G$grand$vertex1, "are included in the network's boundary?"), type = "integer")
    if (G$grand$vertex1.total<igraph::gorder(G) & G$grand$vertex1.total!=0) {warning("You reported that there are fewer nodes in the network's boundary than in the network itself. This is likely a mistake.")}
  }
  if (boundary %in% c("Y", "y") & bipartite) {
    G$grand$vertex1.total <- scan2(cat("How many", G$grand$vertex1, "are included in the network's boundary?"), type = "integer")
    if (G$grand$vertex1.total<sum(igraph::V(G)$type==FALSE) & G$grand$vertex1.total!=0) {warning("You reported that there are fewer nodes in the network's boundary than in the network itself. This is likely a mistake.")}
    G$grand$vertex2.total <- scan2(cat("How many", G$grand$vertex2, "are included in network's boundary?"), type = "integer")
    if (G$grand$vertex2.total<sum(igraph::V(G)$type==TRUE) & G$grand$vertex2.total!=0) {warning("You reported that there are fewer nodes in the network's boundary than in the network itself. This is likely a mistake.")}
  }

  #### Edges ####
  #Definition
  if (!signed) {
    G$grand$edge.pos <- scan2(cat("This network contains", igraph::gsize(G), "edges. What type of relationship do these represent (e.g., friendships)?"), type = "character")
  }

  if (signed) {
    G$grand$edge.pos <- scan2(cat("This network contains", sum(igraph::E(G)$weight==1), "positive edges. What type of relationship do these represent (e.g., friendships)?"), type = "character")
    G$grand$edge.neg <- scan2(cat("This network contains", sum(igraph::E(G)$weight==-1), "negative edges. What type of relationship do these represent (e.g., enemyships)?"), type = "character")
  }

  #Weight definition & measurement
  if (weighted & !signed) {
    G$grand$weight <- menu2(c("Frequency", "Intensity", "Multiplexity", "Valence", "Other"), "What do the edge weights represent?")
    if (G$grand$weight=="Other") {G$grand$weight <- scan2("Please describe what the edge weights represent.", type = "character")}

    G$grand$measure <- menu2(c("Continuous", "Count", "Ordinal", "Categorical", "Other"), title = "How are the edge weights measured?")
    if (G$grand$measure == "Other") {G$grand$measure <- scan2("Please describe how the edge weights are measured.", type = "character")}
    }

  #### Topology ####
  if (!bipartite) {topology <- c("clustering coefficient", "mean path length", "modularity")}
  if (!bipartite & signed) {topology <- c(topology, "structural balance")}
  if (bipartite) {topology <- c("density")}

  cat("The default topological characteristics to report for this network include: ")
  cat(topology, sep=", ")
  moretopo <- scan2(". Do you want to report others? (Y/N)", type = c("Y", "N", "y", "n"))

  if (moretopo %in% c("Y", "y")) {
    options <- c("clustering coefficient", "degree centralization", "degree distribution", "density", "diameter", "efficiency", "mean degree", "modularity", "number of communities", "number of components", "transitivity", "structural balance")
    options <- options[options!=topology]  #Remove defaults from the list of options
    moretopo <- menu2(options, "Which additional topological characteristic(s) should be reported (choose one at a time, 0 when done)?", loop = TRUE)
    topology <- c(topology, moretopo)
  }

  G$grand$topology <- topology
  }

  #### Non-interactive mode ####
  if (!interactive) {
    G$grand$name <- name
    G$grand$doi <- doi
    G$grand$url <- url
    G$grand$vertex1 <- vertex1
    G$grand$vertex2 <- vertex2
    G$grand$vertex1.total <- vertex1.total
    G$grand$vertex2.total <- vertex2.total
    G$grand$edge.pos <- edge.pos
    G$grand$edge.neg <- edge.neg
    G$grand$weight <- weight
    G$grand$measure <- measure
    G$grand$mode <- mode
    G$grand$year <- year
    G$grand$topology <- topology
  }

  #### Return object ####
  return(G)
}
