% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_land.R
\name{spwb_land}
\alias{spwb_land}
\alias{growth_land}
\alias{fordyn_land}
\alias{summary.spwb_land}
\alias{summary.growth_land}
\title{Watershed simulations}
\usage{
spwb_land(
  r,
  sf,
  SpParams,
  meteo = NULL,
  dates = NULL,
  CO2ByYear = numeric(0),
  summary_blocks = NULL,
  summary_frequency = "years",
  local_control = defaultControl(soilDomains = "single"),
  watershed_control = default_watershed_control(),
  parallelize = FALSE,
  num_cores = parallel::detectCores() - 1,
  progress = TRUE
)

growth_land(
  r,
  sf,
  SpParams,
  meteo = NULL,
  dates = NULL,
  CO2ByYear = numeric(0),
  summary_blocks = NULL,
  summary_frequency = "years",
  local_control = medfate::defaultControl(soilDomains = "single"),
  watershed_control = default_watershed_control(),
  parallelize = FALSE,
  num_cores = parallel::detectCores() - 1,
  progress = TRUE
)

fordyn_land(
  r,
  sf,
  SpParams,
  meteo = NULL,
  dates = NULL,
  CO2ByYear = numeric(0),
  summary_blocks = NULL,
  summary_frequency = "years",
  local_control = medfate::defaultControl(soilDomains = "single"),
  watershed_control = default_watershed_control(),
  dispersal_control = default_dispersal_control(),
  management_function = NULL,
  parallelize = FALSE,
  num_cores = parallel::detectCores() - 1,
  progress = TRUE
)

\method{summary}{spwb_land}(object, ...)

\method{summary}{growth_land}(object, ...)
}
\arguments{
\item{r}{An object of class \code{\link[terra]{SpatRaster}}, defining the raster topology.}

\item{sf}{An object of class \code{\link[sf]{sf}} with the following columns:
\itemize{
\item{\code{geometry}: Spatial point geometry corresponding to cell centers.}
\item{\code{elevation}: Elevation above sea level (in m).}
\item{\code{slope}: Slope (in degrees).}
\item{\code{aspect}: Aspect (in degrees).}
\item{\code{land_cover_type}: Land cover type of each grid cell (values should be 'wildland', 'agriculture', 'rock', 'artificial' or 'water').}
\item{\code{forest}: Objects of class \code{\link[medfate]{forest}}.}
\item{\code{soil}: Objects of class \code{\link[medfate]{soil}} or data frames of physical properties.}
\item{\code{state}: Objects of class \code{\link[medfate]{spwbInput}} or \code{\link[medfate]{growthInput}} (optional).}
\item{\code{meteo}: Data frames with weather data (required if parameter \code{meteo = NULL}).}
\item{\code{crop_factor}: Crop evapo-transpiration factor. Only required for 'agriculture' land cover type.}
\item{\code{local_control}: A list of control parameters (optional). Used to override function parameter \code{local_control} for specific cells (values can be \code{NULL} for the remaining ones).}
\item{\code{snowpack}: An optional numeric vector with the snow water equivalent content of the snowpack in each cell (in mm). If missing it will be initialized to zero.}
\item{\code{management_arguments}: Lists with management arguments (optional, relevant for \code{fordyn_land} only).}
\item{\code{result_cell}: A logical indicating that local model results are desired (optional, relevant for \code{spwb_land} and  \code{growth_land} only). Model results are only produced for wildland and agriculture cells. }
}
When using TETIS watershed model, the following columns are also REQUIRED:
\itemize{
\item{\code{depth_to_bedrock}: Depth to bedrock (mm).}
\item{\code{bedrock_conductivity}: Bedrock (saturated) conductivity (in m·day-1).}
\item{\code{bedrock_porosity}: Bedrock porosity (the proportion of pore space in the rock).}
}
When using TETIS watershed model, the following columns are OPTIONAL:
\itemize{
\item{\code{aquifer}: A numeric vector with the water content of the aquifer in each cell (in mm). If missing, it will be initialized to zero.}
\item{\code{deep_aquifer_loss}: A numeric vector with the maximum daily loss to a deeper aquifer (in mm·day-1). If missing all cells take their value from \code{deep_aquifer_loss} in \code{\link{default_watershed_control}}}
\item{\code{channel}: A logical (or binary) vector indicating overland channel routing.}
\item{\code{outlet_backlog}: A list vector indicating channel backlog of outlet cells from a previous simulation.}
}}

\item{SpParams}{A data frame with species parameters (see \code{\link[medfate]{SpParamsMED}}). IMPORTANT: If \code{sf} has been already initialized, this parameter has no effect.}

\item{meteo}{Input meteorological data (see \code{\link{spwb_spatial}} and details).}

\item{dates}{A \code{\link{Date}} object describing the days of the period to be modeled.}

\item{CO2ByYear}{A named numeric vector with years as names and atmospheric CO2 concentration (in ppm) as values. Used to specify annual changes in CO2 concentration along the simulation (as an alternative to specifying daily values in \code{meteo}).}

\item{summary_blocks}{A character vector with variable blocks for cell summaries (or \code{NULL} to retain only basic summaries). Accepted summary blocks for \code{spwb_land} are "WaterBalance" and "Stand". For \code{growth_land} and \code{fordyn_land}, "CarbonBalance" and "BiomassBalance" are also accepted.}

\item{summary_frequency}{Frequency in which cell summary will be produced (e.g. "years", "months", "weeks", ...) (see \code{\link{cut.Date}}).
In \code{fordyn_land} summary frequency can only be "months" or "years".}

\item{local_control}{A list of control parameters (see \code{\link[medfate]{defaultControl}}) for function \code{\link[medfate]{spwb_day}} or \code{\link[medfate]{growth_day}}. By default,
parameter \code{soilDomains} is set to \code{"single"}, meaning a single-domain Richards model. IMPORTANT: If \code{sf} has been already initialized, this parameter has no effect.}

\item{watershed_control}{A list of watershed control parameters (see \code{\link{default_watershed_control}}). Importantly, the sub-model used
for lateral water flows - either \enc{Francés}{Frances} et al. (2007) or \enc{Caviedes-Voullième}{Caviedes-Voullieme} et al. (2023) - is specified there.}

\item{parallelize}{Boolean flag to try parallelization (only works with subwatersheds, see details).}

\item{num_cores}{Integer with the number of cores to be used for parallel computation (by default it will use all clusters minus one).}

\item{progress}{Boolean flag to display progress information for simulations.}

\item{dispersal_control}{A list of dispersal control parameters (see \code{\link{default_dispersal_control}}). If NULL, then dispersal is not simulated.}

\item{management_function}{A function that implements forest management actions (see \code{\link[medfate]{fordyn}}).
of such lists, one per spatial unit.}

\item{object}{An object of class \code{spwb_land} or \code{groth_land}}

\item{...}{Additional parameters for summary functions}
}
\value{
Functions \code{spwb_land}, \code{growth_land} and \code{fordyn_land} return a list of class of the same name as the function with the following elements:
\itemize{
\item{\code{watershed_control}: A list with input control parameters.}
\item{\code{sf}: An object of class \code{\link[sf]{sf}}, similar to the output of \code{\link{spwb_spatial}},
with the following columns:
\itemize{
\item{\code{geometry}: Spatial geometry.}
\item{\code{state}: A list of model input objects for each simulated stand.}
\item{\code{aquifer}: A numeric vector with the water volume in the aquifer of each cell.}
\item{\code{snowpack}: A numeric vector with the snowpack water equivalent volume of each cell.}
\item{\code{summary}: A list of cell summaries containing at least the following variables (additional variables are summarized using \code{summary_blocks}):
\itemize{
\item{\code{MinTemperature}: Minimum temperature (degrees Celsius).}
\item{\code{MaxTemperature}: Maximum temperature (degrees Celsius).}
\item{\code{PET}: Potential evapotranspiration (in mm).}
\item{\code{Rain}: Rainfall (in mm).}
\item{\code{Snow}: Snowfall (in mm).}
\item{\code{SWE}: Snow water equivalent (in mm) of the snowpack.}
\item{\code{RWC}: Soil relative water content with respect to field capacity (in percent).}
\item{\code{SoilVol}: Soil water volume integrated across vertical layers (in mm).}
\item{\code{WTD}: Saturated soil water table depth (in mm from surface).}
\item{\code{DTA}: Depth to aquifer (in m from surface).}
}
}
\item{\code{result}: A list of cell detailed results (only for those indicated in the input), with contents depending on the local model.}
\item{\code{outlet}: A logical vector indicating outlet cells.}
\item{\code{outlet_backlog}: A list vector indicating channel backlog of outlet cells (for subsequent simulations).}
}
In function \code{fordyn_land} the \code{\link[sf]{sf}} object contains additional columns:
\itemize{
\item{\code{forest}: A list of \code{\link[medfate]{forest}} objects for each simulated stand, to be used in subsequent simulations (see \code{\link{update_landscape}}).}
\item{\code{management_arguments}: A list of management arguments for each simulated stand, to be used in subsequent simulations (see \code{\link{update_landscape}}).}
\item{\code{tree_table}: A list of data frames for each simulated stand, containing the living trees at each time step.}
\item{\code{shrub_table}: A list of data frames for each simulated stand, containing the living shrub at each time step.}
\item{\code{dead_tree_table}: A list of data frames for each simulated stand, containing the dead trees at each time step.}
\item{\code{dead_shrub_table}: A list of data frames for each simulated stand, containing the dead shrub at each time step.}
\item{\code{cut_tree_table}: A list of data frames for each simulated stand, containing the cut trees at each time step.}
\item{\code{cut_shrub_table}: A list of data frames for each simulated stand, containing the cut shrub at each time step.}
}
}
\item{\code{watershed_balance}: A data frame with as many rows as days and where columns are (spatially-averaged) components of the water balance at the watershed level (i.e., rain, snow, interception, infiltration, soil evaporation, plant transpiration, ...).}
\item{\code{watershed_soil_balance}: A data frame with as many rows as days and where columns are (spatially-averaged) components of the water balance at the watershed level restricted to those cells with a soil definition.}
\item{\code{channel_export_m3s}: A matrix with daily values of runoff (in m3/s) reaching each of the channel cells of the landscape (useful for channel processing with an external model).}
\item{\code{outlet_export_m3s}: A matrix with daily values of runoff (in m3/s) reaching each of the outlet cells of the landscape. Each outlet drains its own subset of cells (sometimes including channel routing), so the
daily overall watershed export corresponds to the sum of row values.}
}
}
\description{
Functions to perform simulations on a watershed described by a set of connected grid cells.
\itemize{
\item{Function \code{spwb_land} implements a distributed hydrological model that simulates daily local water balance, from \code{\link[medfate]{spwb_day}},
on grid cells of a watershed while accounting for overland runoff, subsurface flow and groundwater flow between cells.}
\item{Function \code{growth_land} is similar to \code{spwb_land}, but includes daily local carbon balance, growth and mortality processes in grid cells,
provided by \code{\link[medfate]{growth_day}}.}
\item{Function \code{fordyn_land} extends the previous two functions with the simulation of management, seed dispersal, recruitment
and resprouting.}
}
}
\details{
IMPORTANT: Simulation function will normally call the initialization of state variables via an internal call to \code{\link{initialize_landscape}}, using parameters \code{local_control} and \code{SpParams} in this call. The default \code{soilDomains = "single"} means that vertical bulk soil flows are simulated using a single permeability domain with Richards equation.
However, if object \code{sf} has been previously initialized, then the control parameters of this previous initialization will remain the same. In other words, parameters \code{local_control} and \code{SpParams} will have no effect in the call to the simulation routines if the \code{sf} has been previously initialized.

Two sub-models are available for lateral water transfer processes (overland flow, sub-surface flow, etc.), either "TETIS"
(similar to \enc{Francés}{Frances} et al. 2007) or "SERGHEI" (\enc{Caviedes-Voullième}{Caviedes-Voullieme} et al. 2023).

IMPORTANT: medfateland needs to be compiled along with SERGHEI model in order to launch simulations with using this
distributed hydrological model.

When running \code{fordyn_land}, the input 'sf' object has to be in a Universal Transverse Mercator (UTM) coordinate system (or any other projection using meters as length unit)
for appropriate behavior of dispersal sub-model.

Due to the large communication overload, parallel computation is only allowed for TETIS in combination with definition of subwatersheds (see flag of TETIS parameters in \code{\link{default_watershed_control}}).

When dealing with large data sets, weather data included in the 'sf' object will likely be very data hungry. In those cases, it is
recommended to resort on weather interpolation (see \code{\link{spwb_spatial}}). Weather interpolation can be done using a coarser resolution
than that of raster 'r', by changing the watershed control parameter called 'weather_aggregation_factor' (see \code{\link{default_watershed_control}}).
}
\examples{
\donttest{
# Load example watershed data
data("example_watershed")

# Set crop factor 
example_watershed$crop_factor <- NA
example_watershed$crop_factor[example_watershed$land_cover_type=="agriculture"] <- 0.75

# Set request for daily model results in cells number 3, 6 (outlet) and 9
example_watershed$result_cell <- FALSE
example_watershed$result_cell[c(3,6,9)] <- TRUE

# Get bounding box to determine limits
b <- sf::st_bbox(example_watershed)
b

# Define a raster topology, using terra package, 
# with the same CRS as the watershed. In this example cells have 100 m side.
# Coordinates in the 'sf' object are assumed to be cell centers
r <-terra::rast(xmin = 401380, ymin = 4671820, xmax = 402880, ymax = 4672620, 
                nrow = 8, ncol = 15, crs = "epsg:32631")

# Load example meteo data frame from package meteoland
data("examplemeteo")
  
# Load default medfate parameters
data("SpParamsMED")
  
# Set simulation period
dates <- seq(as.Date("2001-01-01"), as.Date("2001-03-31"), by="day")

# Watershed control parameters (TETIS model; Frances et al. 2007)
ws_control <- default_watershed_control("tetis")

# Launch simulations 
res <- spwb_land(r, example_watershed, SpParamsMED, examplemeteo, 
                 dates = dates, summary_frequency = "month",
                 watershed_control = ws_control)
                 
# Print a summary of water balance components
summary(res)

# Option 'simplify = TRUE' in initialization, may be useful to speed up calculations
example_simplified <- initialize_landscape(example_watershed, SpParams = SpParamsMED,
                                           local_control = defaultControl(soilDomains = "single"), 
                                           simplify = TRUE)

# Launch simulations over simplified landscape (should be considerably faster)
res_simplified <- spwb_land(r, example_simplified, SpParamsMED, examplemeteo, 
                            dates = dates, summary_frequency = "month",
                            watershed_control = ws_control)
}

}
\references{
\enc{Francés}{Frances}, F., \enc{Vélez}{Velez}, J.I. & \enc{Vélez}{Velez}, J.J. (2007). Split-parameter structure for the automatic calibration of distributed hydrological models. Journal of Hydrology, 332, 226–240.

\enc{Caviedes-Voullième}{Caviedes-Voullieme}, D., \enc{Morales-Hernández}{Morales-Hernandez}, M., Norman, M.R. & Ogzen-Xian, I. (2023). SERGHEI (SERGHEI-SWE) v1.0: a performance-portable high-performance parallel-computing shallow-water solver for hydrology and environmental hydraulics. Geoscientific Model Development, 16, 977-1008.
}
\seealso{
\code{\link{default_watershed_control}}, \code{\link{initialize_landscape}}, \code{\link{overland_routing}},
\code{\link{spwb_land_day}}, \code{\link[medfate]{spwb_day}},  \code{\link[medfate]{growth_day}},
\code{\link{spwb_spatial}}, \code{\link{fordyn_spatial}}, \code{\link{dispersal}}
}
\author{
Miquel De \enc{Cáceres}{Caceres} Ainsa, CREAF.

Maria \enc{González-Sanchís}{Gonzalez-Sanchis}, Universitat Politecnica de Valencia.

Daniel \enc{Caviedes-Voullième}{Caviedes-Voullieme}, Forschungszentrum Julich.

Mario \enc{Morales-Hernández}{Morales-Hernandez}, Universidad de Zaragoza.
}
