#' Set ALT Design Information
#'
#' Configures the settings for an accelerated life test.
#'
#' @param k_levels Integer. Number of stress levels.
#' @param j_factor Integer. Number of stress factors.
#' @param n_unit Integer. Total number of test units.
#' @param censor_time Numeric. Test duration or censoring time.
#' @param p Numeric. 0 < p < 1. Lifetime percentile to be estimated at the use condition, i.e. stress levels are 0.
#' @param use_cond Vector. Stress levels at the use condition.
#' @param sigma Numeric. Scale parameter of the lifetime distribution.
#' @param x_l Numeric. Lower bound of stress range. Default is 0.
#' @param x_h Numeric. Upper bound of stress range. Default is 1.
#' @param opt_type Character. Optimality criterion, currently only C-optimality is supported. Default is "C".
#' @param reparam Logical. Whether reparameterization is applied to model parameters. Reparameterization is supported for all design types, while non-reparameterization is only available for locally optimal design \code{design_type = 1}. Default is TRUE.
#' @return A list of design specifications
#' @examples
#' design_info <- set_design_info(k_levels=3, j_factor=1, n_unit=300, 
#'                            censor_time=183, p=0.1, use_cond=c(0), sigma=0.6)
#' @name set_design_info
#' @rdname set_design_info
#' @export
set_design_info <- function(k_levels, j_factor, n_unit, censor_time, 
                            p, use_cond, sigma, 
                            x_l = 0, x_h = 1,
                            opt_type = "C", reparam = TRUE) {
  
  design_info_list <- list()

  design_info_list$n_support = k_levels
  design_info_list$n_factor = j_factor
  design_info_list$n_unit = n_unit
  design_info_list$censor_time = censor_time
  design_info_list$sigma = sigma
  design_info_list$p = p
  design_info_list$use_cond = use_cond
  design_info_list$opt_type = opt_type
  design_info_list$reparam = reparam
  design_info_list$x_l = x_l
  design_info_list$x_h = x_h
  
  return(design_info_list)
}



#' Set PSO Optimization Settings
#'
#' Define hyperparameters for particle swarm optimization (PSO).
#'
#' @param n_swarm Integer. Number of particles in the swarm.
#' @param max_iter Integer. Maximum number of iterations.
#' @param early_stopping Integer. The frequency, i.e. number of iterations, of validating the design optimality using equivalence theorem. The optimization process stops once maximum directional derivative is approximately 1.
#' @param tol Numeric. Convergence tolerance. The algorithm stops if \code{abs(max_directional_derivative - 1) < tol}.
#' @param c1 Numeric. Cognitive acceleration coefficient. Default value is 2.05. 
#' @param c2 Numeric. Social acceleration coefficient. Default value is 2.05.
#' @param w0 Numeric. Starting inertia weight. Default value is 1.2.
#' @param w1 Numeric. Ending inertia weight. Default value is 0.2.
#' @param w_var Numeric. A number between \eqn{[0, 1]} that controls the percentage of iterations during which PSO linearly decrease inertia weight from \code{w0} to \code{w1}. Default value is 0.8.
#' @param vk Numeric. Velocity clamping factor. Default value is 4.
#' @return A list of PSO hyperparameters.
#' @examples
#' pso_info <- pso_setting(n_swarm=32, max_iter=128, early_stopping=10, tol=0.01)
#' @name pso_setting
#' @rdname pso_setting
#' @export
pso_setting <- function(n_swarm = 32, max_iter = 128,
                        early_stopping = 10, tol = 0.01, c1 = 2.05, c2 = 2.05,
                        w0 = 1.2, w1 = 0.2, w_var = 0.8, vk = 4) {
  list(
    n_swarm = n_swarm, max_iter = max_iter, 
    early_stopping = early_stopping, tol = tol, c1 = c1, c2 = c2, w0 = w0, w1 = w1, w_var = w_var,
    vk = vk
  )
}


#' Initialize Particle Swarm Optimization and Nelder-Mead Algorithm Values
#'
#' Sets initial particles for PSO, initial locally optimal design, and initial parameters for Nelder-Mead algorithm.
#'
#' @param init_swarm Optional matrix of initial particle positions. If not defined, particle positions are randomly generated using \code{runif} with pre-determined number of particles \code{n_swarm} and particle size.
#' @param init_local Optional vector of initial locally optimal design. If not defined, the initial vector representing locally optimal design is \code{c(1, 0.6, 0.3)}. 
#' @param init_coef_mat Optional matrix of initial parameters to implement multi-start Nelder-Mead algorithm. The number of rows is the number of starts, and each row is the corresponding initial parameters. If not defined, the initial matrix of parameters is generated by sigmoid transformation of \code{10 * as.matrix(expand.grid(rep(list(c(1, -1)), j_factor + 1)))}.
#' @return A list of initialized values.
#' @examples
#' init_local <- c(1, 0.6, 0.3)
#' 
#' init_coef_mat <- rbind(
#'   c(1e-6, 0.99),
#'   c(1e-2, 1),
#'   c(1.01e-6, 0.9999))
#'   
#' j_factor <- 1
#' k_levels <- 3
#' n_swarm <- 32
#' d_swarm <- (j_factor + 1) * k_levels - 1
#' init_swarm <- matrix(runif(n_swarm*d_swarm), nrow=n_swarm, byrow=TRUE)
#' 
#' init_values <- initialize_values(init_swarm=init_swarm, 
#'                                   init_local=init_local, 
#'                                   init_coef_mat=init_coef_mat)
#' 
#' @export
initialize_values <- function(init_swarm = NULL,
                              init_local = NULL, init_coef_mat = NULL) {
  list(
    init_swarm = init_swarm, 
    init_local = init_local, init_coef_mat = init_coef_mat
  )
}