#' @title
#' Computing the minimal elements of a poset.
#'
#' @description
#' Computes the minimal elements of the input poset, i.e. those elements strictly dominating no other elements.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @return
#' A vector of character strings (the names of the minimal elements).
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' mnms <-POSetMinimals(pos)
#'
#' @name POSetMinimals
#' @export POSetMinimals
POSetMinimals <- function(poset) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    result <- .Call("_Minimal", poset@ptr)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
