% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{create_model}
\alias{create_model}
\title{Train an univariate time series forecasting model}
\usage{
create_model(
  timeS,
  lags = NULL,
  method = c("knn", "lm", "rt", "mt", "bagging", "rf"),
  param = NULL,
  preProcess = NULL
)
}
\arguments{
\item{timeS}{A time series of class \code{ts} or a numeric vector.}

\item{lags}{An integer vector, in increasing order, expressing the lags used
as autoregressive variables. If the default value (\code{NULL}) is provided, a
suitable vector is chosen.}

\item{method}{A string indicating the method used for training and
forecasting. Allowed values are:
\itemize{
\item \code{"knn"}: k-nearest neighbors (the default)
\item \code{"lm"}: linear regression
\item \code{"rt"}: regression trees
\item \code{"mt"}:  model trees
\item \code{"bagging"}
\item \code{"rf"}: random forests.
}

See details for a brief explanation of the models. It is also possible to
use your own regression model, in that case a function explaining how to
build your model must be provided, see the vignette for further details.}

\item{param}{A list with parameters for the underlying function that builds
the model. If the default value (\code{NULL}) is provided, the model is fitted
with its default parameters. See details for the functions used to train the
models.}

\item{preProcess}{A list indicating the preprocessings or transformations.
Currently, the length of the list must be 1 (only one preprocessing). If
\code{NULL} the additive transformation is applied to the series. The element of
the list is created with the \code{\link[=trend]{trend()}} function.}
}
\value{
An S3 object of class \code{utsf}, basically a list with, at least, the
following components: \item{\code{ts}}{The time series being forecast.}
\item{\code{features}}{A data frame with the features of the training set. The
column names of the data frame indicate the autoregressive lags.}
\item{\code{targets}}{A vector with the targets of the training set.}
\item{\code{lags}}{An integer vector with the autoregressive lags.}
\item{\code{model}}{The regression model used recursively to make the forecast.}
}
\description{
This function trains a model from the historical values of a time series using
an autoregressive approach: the targets are the historical values and the
features of the targets their lagged values.
}
\details{
The functions used to build and train the model are:
\itemize{
\item KNN: In this case no model is built and the function \code{\link[FNN:knn.reg]{FNN::knn.reg()}} is
used to predict the future values of the time series.
\item Linear models: Function \code{\link[stats:lm]{stats::lm()}} to build the model and the method
\code{\link[stats:predict.lm]{stats::predict.lm()}} associated with the trained model to forecast the future
values of the time series.
\item Regression trees: Function \code{\link[rpart:rpart]{rpart::rpart()}} to build the model and the
method \code{\link[rpart:predict.rpart]{rpart::predict.rpart()}} associated with the trained model to forecast
the future values of the time series.
\item Model trees: Function \code{\link[Cubist:cubist.default]{Cubist::cubist()}} to build the model and the
method \code{\link[Cubist:predict.cubist]{Cubist::predict.cubist()}} associated with the trained model to
forecast the future values of the time series.
\item Bagging: Function \code{\link[ipred:bagging]{ipred::bagging()}} to build the model and the
method \code{\link[ipred:predict.bagging]{ipred::predict.regbagg()}} associated with the trained model to
forecast the future values of the time series.
\item Random forest: Function \code{\link[ranger:ranger]{ranger::ranger()}} to build the model and the
method \code{\link[ranger:predict.ranger]{ranger::predict.ranger()}} associated with the trained model to
forecast the future values of the time series.
}
}
\examples{
## Build model using k-nearest neighbors
create_model(AirPassengers, method = "knn")

## Using k-nearest neighbors changing the default k value
create_model(AirPassengers, method = "knn", param = list(k = 5))

## Using your own regression model

# Function to build the regression model
my_knn_model <- function(X, y) {
  structure(list(X = X, y = y), class = "my_knn")
}
# Function to predict a new example
predict.my_knn <- function(object, new_value) {
  FNN::knn.reg(train = object$X, test = new_value, y = object$y)$pred
}
create_model(AirPassengers, method = my_knn_model)

}
