% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood.R
\name{compute_L_from_P}
\alias{compute_L_from_P}
\title{Compute likelihood vector from a prior matrix via SVD (center-only, robust)}
\usage{
compute_L_from_P(P, renormalize_rows = TRUE, tol = 1e-12)
}
\arguments{
\item{P}{Numeric matrix \eqn{T \times K}; prior weights per period.}

\item{renormalize_rows}{Logical; if \code{TRUE} (default), rows of \code{P} that
are within tolerance of summing to 1 are renormalized. Otherwise an error is thrown.}

\item{tol}{Numeric tolerance for simplex checks (default \code{1e-12}).}
}
\value{
Numeric vector \eqn{L} of length \eqn{K} (non-negative, sums to 1). Attributes:
\itemize{
  \item \code{"pc1_loadings"}: signed PC1 loadings \eqn{v}.
  \item \code{"explained_var"}: fraction of variance explained by PC1.
  \item \code{"fallback"}: \code{TRUE} if column-mean fallback was used.
}
}
\description{
Builds a sectoral likelihood \eqn{L} (length \eqn{K}) from the prior weights
matrix \eqn{P \in \mathbb{R}^{T \times K}} by taking the absolute value of the
first right singular vector of the centered matrix (no scaling), then
normalizing to the unit simplex. Includes input validation, optional row
renormalization, and a safe fallback when PC1 is degenerate.
}
\details{
\strong{Validation:} \code{P} must be a finite, non-negative numeric matrix.
Each row must either (i) already sum to 1 within \code{tol} or (ii) be renormalizable
within \code{tol}. Rows with (near-)zero sums are not renormalizable and raise an error.
Missing values are not allowed.

\strong{Algorithm (exactly as implemented):}
\enumerate{
  \item Center columns over time: \code{X <- scale(P, center = TRUE, scale = FALSE)}.
  \item Compute SVD: \code{sv <- svd(X)}.
  \item Take the first right singular vector (first column of V matrix); set \eqn{l = |v|}.
  \item If \eqn{\sum l \leq tol} or PC1 is degenerate, fall back to column means of \code{P} (over time) and renormalize.
  \item Otherwise, \eqn{L = l / \sum l}.
}
}
\examples{
set.seed(123)
T <- 10; K <- 4
P <- matrix(rexp(T*K), nrow = T); P <- P / rowSums(P)
L <- compute_L_from_P(P)
stopifnot(length(L) == K, all(L >= 0), abs(sum(L) - 1) < 1e-12)

}
\seealso{
\code{\link{spread_likelihood}}, \code{\link{bayesian_disaggregate}}
}
