#' checkCompOverlap
#'
#' This function is used to check for overlapping components and remove them if
#' they overlap. The mixed modeling can sometimes run into an error in which it
#' produces multiple essentially identical components. This function attempts to 
#' find and remove these duplicates.
#'
#' @param comps Component parameters 
#' @param pois Whether or not the components are poisson or normal distributions
#' @return Returns a components with no overlaps
checkCompOverlap = function(comps, pois = FALSE){
  
  totalSamples = 5000
  remainingComps = ifelse(pois,length(comps),ncol(comps))
  
  finalComps = comps
  i = 1
  match = 0
  while(i < remainingComps){
    
    # Samples the component distribution
    if (pois){
      refYVals = stats::rpois(totalSamples,finalComps[i])
    }
    else{
      refYVals = stats::rnorm(totalSamples,finalComps[1,i],finalComps[2,i])
    }
    remainingComps = ifelse(pois,length(finalComps),ncol(finalComps))
    
    # Goes through the remaining components and calculates ks statistic to 
    # evaluate similarity
    for (other in (i+1):remainingComps){
      if (pois){
        valYVals = stats::rpois(totalSamples,finalComps[other])
      }
      else{
        valYVals = stats::rnorm(totalSamples,finalComps[1,other],finalComps[2,other])
      }
      if (stats::ks.test(refYVals,valYVals)$statistic < 0.1){
        match = other
        break()
      }
    }
    
    # If a match was found it combines the two
    if (match != 0){
      if (pois){
        newComp = mean(c(finalComps[i],finalComps[match]))
        finalComps[i] = newComp
        finalComps = finalComps[-match]
      }
      else{
        newComp = apply(rbind(finalComps[,i],finalComps[,match]),2,mean)
        finalComps[,i] = newComp
        finalComps = finalComps[,-match]
      }
      match = 0
      remainingComps = remainingComps - 1
    }
    else
      i = i + 1
  }

  # Returns the components with duplicates removed.
  return(finalComps)
}
