#' FSK Metadata Classes
#' 
#' S3 classes for FSK metadata based on the FSKX JSON schema specification.
#' These classes ensure proper JSON serialization with correct array/scalar types.
#' 
#' @importFrom jsonlite toJSON

# =============================================================================
# CORE METADATA CLASSES
# =============================================================================

#' Create FSK Metadata object
#' 
#' Main metadata container following FSKX specification
#' 
#' @param modelType Model type, default "genericModel"
#' @param generalInformation FSKGeneralInformation object
#' @param scope FSKScope object  
#' @param dataBackground FSKDataBackground object
#' @param modelMath FSKModelMath object
#' @export
FSKMetadata <- function(modelType = "genericModel", generalInformation = NULL, 
                       scope = NULL, dataBackground = NULL, modelMath = NULL) {
  
  obj <- list(
    modelType = as.character(modelType),
    generalInformation = generalInformation,
    scope = scope,
    dataBackground = dataBackground,
    modelMath = modelMath
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKMetadata", "list")
  return(obj)
}

#' General Information Section
#' 
#' @param name Model name (string)
#' @param source Source of model/data (string)
#' @param identifier Unique identifier (string)
#' @param author List of FSKAuthor objects (array)
#' @param creator List of FSKCreator objects (array) 
#' @param creationDate Creation date (string)
#' @param modificationDate Modification dates (array of strings)
#' @param rights Rights information (string)
#' @param availability Availability (string)
#' @param url URL (string)
#' @param format Format (string) 
#' @param reference List of FSKReference objects (array)
#' @param language Language (string)
#' @param software Software (string)
#' @param languageWrittenIn Language written in (string)
#' @param modelCategory FSKModelCategory object
#' @param status Status (string)
#' @param objective Objective (string)
#' @param description Description (string)
#' @export
FSKGeneralInformation <- function(name = NULL, source = NULL, identifier = NULL,
                                 author = NULL, creator = NULL, creationDate = NULL,
                                 modificationDate = NULL, rights = NULL, availability = NULL,
                                 url = NULL, format = NULL, reference = NULL,
                                 language = NULL, software = NULL, languageWrittenIn = NULL,
                                 modelCategory = NULL, status = NULL, objective = NULL,
                                 description = NULL) {
  
  obj <- list(
    name = as.character(name),
    source = as.character(source), 
    identifier = as.character(identifier),
    author = author,  # array
    creator = creator,  # array
    creationDate = as.character(creationDate),
    modificationDate = modificationDate,  # array
    rights = as.character(rights),
    availability = as.character(availability),
    url = as.character(url),
    format = as.character(format),
    reference = reference,  # array
    language = as.character(language),
    software = as.character(software),
    languageWrittenIn = as.character(languageWrittenIn),
    modelCategory = modelCategory,  # object
    status = as.character(status),
    objective = as.character(objective),
    description = as.character(description)
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKGeneralInformation", "list")
  return(obj)
}

#' Scope Section
#' 
#' @param product List of FSKProduct objects (array)
#' @param hazard List of FSKHazard objects (array)
#' @param populationGroup List of FSKPopulationGroup objects (array)
#' @param generalComment General comment (string)
#' @param temporalInformation Temporal information (string)  
#' @param spatialInformation Spatial information (array of strings)
#' @export
FSKScope <- function(product = NULL, hazard = NULL, populationGroup = NULL,
                    generalComment = NULL, temporalInformation = NULL, 
                    spatialInformation = NULL) {
  
  obj <- list(
    product = product,  # array
    hazard = hazard,  # array
    populationGroup = populationGroup,  # array
    generalComment = if(!is.null(generalComment)) as.character(generalComment) else NULL,
    temporalInformation = if(!is.null(temporalInformation)) as.character(temporalInformation) else NULL,
    spatialInformation = spatialInformation  # array
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKScope", "list")
  return(obj)
}

#' Data Background Section
#' 
#' @param study FSKStudy object
#' @param studySample List of FSKStudySample objects (array)
#' @param dietaryAssessmentMethod List of FSKDietaryAssessmentMethod objects (array)
#' @param laboratory List of FSKLaboratory objects (array)
#' @param assay List of FSKAssay objects (array)
#' @export
FSKDataBackground <- function(study = NULL, studySample = NULL, 
                             dietaryAssessmentMethod = NULL, laboratory = NULL,
                             assay = NULL) {
  
  obj <- list(
    study = study,  # object
    studySample = studySample,  # array
    dietaryAssessmentMethod = dietaryAssessmentMethod,  # array
    laboratory = laboratory,  # array
    assay = assay  # array
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKDataBackground", "list")
  return(obj)
}

#' Model Math Section
#' 
#' @param parameter List of FSKParameter objects (array)
#' @param qualityMeasures List of FSKQualityMeasures objects (array)
#' @param modelEquation List of FSKModelEquation objects (array)
#' @param fittingProcedure Fitting procedure (string)
#' @param exposure List of FSKExposure objects (array)
#' @param event Event information (array of strings)
#' @export
FSKModelMath <- function(parameter = NULL, qualityMeasures = NULL, 
                        modelEquation = NULL, fittingProcedure = NULL,
                        exposure = NULL, event = NULL) {
  
  obj <- list(
    parameter = parameter,  # array
    qualityMeasures = qualityMeasures,  # array
    modelEquation = modelEquation,  # array
    fittingProcedure = as.character(fittingProcedure),  # string
    exposure = exposure,  # array
    event = event  # array
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKModelMath", "list")
  return(obj)
}

# =============================================================================
# NESTED OBJECT CLASSES
# =============================================================================

#' Author Information
#' 
#' @param title Title (string)
#' @param familyName Family name (string)
#' @param givenName Given name (string)
#' @param email Email address (string, required)
#' @param telephone Telephone (string)
#' @param streetAddress Street address (string)
#' @param country Country (string)
#' @param zipCode Zip code (string)
#' @param region Region (string)
#' @param timeZone Time zone (string)
#' @param gender Gender (string)
#' @param note Note (string)
#' @param organization Organization (string)
#' @export
FSKAuthor <- function(title = NULL, familyName = NULL, givenName = NULL,
                     email = NULL, telephone = NULL, streetAddress = NULL,
                     country = NULL, zipCode = NULL, region = NULL,
                     timeZone = NULL, gender = NULL, note = NULL,
                     organization = NULL) {
  
  if (is.null(email)) {
    stop("Email is required for FSKAuthor")
  }
  
  obj <- list(
    title = as.character(title),
    familyName = as.character(familyName),
    givenName = as.character(givenName),
    email = as.character(email),
    telephone = as.character(telephone),
    streetAddress = as.character(streetAddress),
    country = as.character(country),
    zipCode = as.character(zipCode),
    region = as.character(region),
    timeZone = as.character(timeZone),
    gender = as.character(gender),
    note = as.character(note),
    organization = as.character(organization)
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKAuthor", "list")
  return(obj)
}

# FSKCreator is same structure as FSKAuthor
#' @rdname FSKAuthor
#' @export
FSKCreator <- function(title = NULL, familyName = NULL, givenName = NULL,
                      email = NULL, telephone = NULL, streetAddress = NULL,
                      country = NULL, zipCode = NULL, region = NULL,
                      timeZone = NULL, gender = NULL, note = NULL,
                      organization = NULL) {
  
  if (is.null(email)) {
    stop("Email is required for FSKCreator")
  }
  
  obj <- FSKAuthor(title, familyName, givenName, email, telephone, 
                   streetAddress, country, zipCode, region, timeZone,
                   gender, note, organization)
  
  class(obj) <- c("FSKCreator", "FSKAuthor", "list")
  return(obj)
}

#' Reference Information
#' 
#' @param isReferenceDescription Is reference description (boolean)
#' @param title Title (string, required)
#' @param doi DOI (string, required)
#' @param publicationType Publication type (string)
#' @param date Date (string)
#' @param pmid PubMed ID (string)
#' @param authorList Author list (string)
#' @param abstract Abstract (string)
#' @param journal Journal (string)
#' @param volume Volume (string)
#' @param issue Issue (string)
#' @param status Status (string)
#' @param website Website (string)
#' @param comment Comment (string)
#' @export
FSKReference <- function(isReferenceDescription = NULL, title = NULL, doi = NULL,
                        publicationType = NULL, date = NULL, pmid = NULL,
                        authorList = NULL, abstract = NULL, journal = NULL,
                        volume = NULL, issue = NULL, status = NULL,
                        website = NULL, comment = NULL) {
  
  if (is.null(isReferenceDescription) || is.null(title) || is.null(doi)) {
    stop("isReferenceDescription, title, and doi are required for FSKReference")
  }
  
  obj <- list(
    isReferenceDescription = as.logical(isReferenceDescription),
    title = as.character(title),
    doi = as.character(doi),
    publicationType = as.character(publicationType),
    date = as.character(date),
    pmid = as.character(pmid),
    authorList = as.character(authorList),
    abstract = as.character(abstract),
    journal = as.character(journal),
    volume = as.character(volume),
    issue = as.character(issue),
    status = as.character(status),
    website = as.character(website),
    comment = as.character(comment)
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKReference", "list")
  return(obj)
}

#' Model Category Information
#' 
#' @param modelClass Model class (string, required)
#' @param modelSubClass Model subclass (array of strings)
#' @param modelClassComment Model class comment (string)
#' @param basicProcess Basic process (array of strings)
#' @export
FSKModelCategory <- function(modelClass = NULL, modelSubClass = NULL,
                            modelClassComment = NULL, basicProcess = NULL) {
  
  if (is.null(modelClass)) {
    stop("modelClass is required for FSKModelCategory")
  }
  
  obj <- list(
    modelClass = as.character(modelClass),
    modelSubClass = modelSubClass,  # array
    modelClassComment = as.character(modelClassComment),
    basicProcess = basicProcess  # array
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKModelCategory", "list")
  return(obj)
}

#' Parameter Information
#' 
#' @param id Parameter ID (string, required)
#' @param classification Classification (string, required)
#' @param name Parameter name (string, required)
#' @param description Description (string)
#' @param unit Unit (string, required)
#' @param unitCategory Unit category (string)
#' @param dataType Data type (string, required)
#' @param source Source (string)
#' @param subject Subject (string)
#' @param distribution Distribution (string)
#' @param value Value (string)
#' @param reference FSKReference object
#' @param variabilitySubject Variability subject (string)
#' @param minValue Minimum value (string)
#' @param maxValue Maximum value (string)
#' @param error Error (string)
#' @export
FSKParameter <- function(id = NULL, classification = NULL, name = NULL,
                        description = NULL, unit = NULL, unitCategory = NULL,
                        dataType = NULL, source = NULL, subject = NULL,
                        distribution = NULL, value = NULL, reference = NULL,
                        variabilitySubject = NULL, minValue = NULL,
                        maxValue = NULL, error = NULL) {
  
  required_fields <- c("id", "classification", "name", "unit", "dataType")
  missing_fields <- c()
  
  if (is.null(id)) missing_fields <- c(missing_fields, "id")
  if (is.null(classification)) missing_fields <- c(missing_fields, "classification")
  if (is.null(name)) missing_fields <- c(missing_fields, "name")
  if (is.null(unit)) missing_fields <- c(missing_fields, "unit")
  if (is.null(dataType)) missing_fields <- c(missing_fields, "dataType")
  
  if (length(missing_fields) > 0) {
    stop(paste("Required fields missing for FSKParameter:", paste(missing_fields, collapse = ", ")))
  }
  
  obj <- list(
    id = as.character(id),
    classification = as.character(classification),
    name = as.character(name),
    description = as.character(description),
    unit = as.character(unit),
    unitCategory = as.character(unitCategory),
    dataType = as.character(dataType),
    source = as.character(source),
    subject = as.character(subject),
    distribution = as.character(distribution),
    value = as.character(value),
    reference = reference,  # object
    variabilitySubject = as.character(variabilitySubject),
    minValue = as.character(minValue),
    maxValue = as.character(maxValue),
    error = as.character(error)
  )
  
  # Remove NULL elements
  obj <- obj[!sapply(obj, is.null)]
  
  class(obj) <- c("FSKParameter", "list")
  return(obj)
}

# =============================================================================
# HELPER FUNCTIONS
# =============================================================================

#' Recursively clean object by removing empty values
#' 
#' @param obj Object to clean (any R object - list, vector, etc.)
#' @return Cleaned object with empty values removed
#' @export
clean_empty_values <- function(obj) {
  # Base cases - values that should be omitted entirely
  if (is.null(obj)) return(NULL)
  if (length(obj) == 0) return(NULL)
  if (is.character(obj) && length(obj) == 1 && (obj == "" || obj == "NULL" || is.na(obj))) return(NULL)
  
  # Handle lists (including data.frames, which are lists)
  if (is.list(obj)) {
    # Recursively clean each element
    cleaned <- lapply(obj, clean_empty_values)
    
    # Remove NULL elements (those that were omitted)
    cleaned <- cleaned[!sapply(cleaned, is.null)]
    
    # If the list is now empty, return NULL
    if (length(cleaned) == 0) return(NULL)
    
    return(cleaned)
  }
  
  # Handle vectors (including character, numeric, logical)
  if (is.vector(obj)) {
    # For vectors, filter out empty/null elements
    if (is.character(obj)) {
      # Remove empty strings and "NULL" strings
      filtered <- obj[!is.na(obj) & obj != "" & obj != "NULL"]
      if (length(filtered) == 0) return(NULL)
      return(filtered)
    } else {
      # For numeric/logical vectors, just remove NAs
      filtered <- obj[!is.na(obj)]
      if (length(filtered) == 0) return(NULL)
      return(filtered)
    }
  }
  
  # For all other types, return as-is
  return(obj)
}

# =============================================================================
# JSON SERIALIZATION METHODS
# =============================================================================

#' @method toJSON FSKMetadata
#' @export
toJSON.FSKMetadata <- function(x, ...) {
  # Build result with proper structure
  result <- list()
  
  # modelType as scalar (comparing with TypeScript: modelType?: string)
  if (!is.null(x$modelType)) {
    result$modelType = as.character(x$modelType[1])
  }
  
  # Other sections use their own toJSON methods
  if (!is.null(x$generalInformation)) {
    result$generalInformation = x$generalInformation
  }
  if (!is.null(x$scope)) {
    result$scope = x$scope
  }
  if (!is.null(x$dataBackground)) {
    result$dataBackground = x$dataBackground
  }
  if (!is.null(x$modelMath)) {
    result$modelMath = x$modelMath
  }
  
  jsonlite::toJSON(result, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKGeneralInformation
#' @export
toJSON.FSKGeneralInformation <- function(x, ...) {
  # Build result matching TypeScript GeneralInformation type exactly
  result <- list()
  
  # Scalar fields (TypeScript: name?: string, etc.)
  scalar_fields <- c("name", "source", "identifier", "rights", "availability", 
                    "url", "format", "language", "software", "languageWrittenIn", 
                    "status", "objective", "description")
  
  for (field in scalar_fields) {
    if (!is.null(x[[field]])) {
      # Extract scalar value - no arrays!
      if (is.character(x[[field]]) && length(x[[field]]) > 0) {
        result[[field]] <- as.character(x[[field]][1])
      }
    }
  }
  
  # creationDate can be number[] or string (TypeScript: creationDate?: number[] | string | null)
  if (!is.null(x$creationDate)) {
    result$creationDate <- x$creationDate  # Keep as is
  }
  
  # Array fields (TypeScript: author?: Person[], creator?: Person[], etc.)
  if (!is.null(x$author)) result$author <- x$author
  if (!is.null(x$creator)) result$creator <- x$creator
  if (!is.null(x$reference)) result$reference <- x$reference
  if (!is.null(x$modificationDate)) result$modificationDate <- x$modificationDate
  
  # Object field (TypeScript: modelCategory?: ModelCategory | null)
  if (!is.null(x$modelCategory)) {
    result$modelCategory <- x$modelCategory
  }
  
  jsonlite::toJSON(result, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKScope
#' @export
toJSON.FSKScope <- function(x, ...) {
  result <- list()
  
  # Scalar fields - only include if they have actual values
  scalar_fields <- c("generalComment", "temporalInformation")
  
  for (field in scalar_fields) {
    if (!is.null(x[[field]])) {
      # Handle different input types
      field_value <- NULL
      
      if (is.vector(x[[field]]) && length(x[[field]]) > 0) {
        field_value <- as.character(x[[field]][1])
      } else if (is.character(x[[field]]) && length(x[[field]]) == 1) {
        field_value <- x[[field]]
      }
      
      # Only include if not empty or null
      if (!is.null(field_value) && field_value != "" && field_value != "NULL") {
        result[[field]] <- field_value
      }
      # If empty, skip entirely (don't add to result)
    }
  }
  
  # Array fields
  array_fields <- c("product", "hazard", "populationGroup", "spatialInformation")
  
  for (field in array_fields) {
    if (!is.null(x[[field]])) {
      result[[field]] <- x[[field]]
    }
  }
  
  jsonlite::toJSON(result, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKDataBackground
#' @export
toJSON.FSKDataBackground <- function(x, ...) {
  result <- list()
  
  # Object fields
  if (!is.null(x$study)) {
    result$study <- x$study
  }
  
  # Array fields
  array_fields <- c("studySample", "dietaryAssessmentMethod", "laboratory", "assay")
  
  for (field in array_fields) {
    if (!is.null(x[[field]])) {
      result[[field]] <- x[[field]]
    }
  }
  
  jsonlite::toJSON(result, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKModelMath
#' @export
toJSON.FSKModelMath <- function(x, ...) {
  result <- list()
  
  # Array fields
  array_fields <- c("parameter", "qualityMeasures", "modelEquation", "exposure", "event")
  
  for (field in array_fields) {
    if (!is.null(x[[field]])) {
      result[[field]] <- x[[field]]
    }
  }
  
  # Scalar fields
  if (!is.null(x$fittingProcedure)) {
    result$fittingProcedure <- x$fittingProcedure
  }
  
  jsonlite::toJSON(result, auto_unbox = TRUE, pretty = TRUE, ...)
}

# Default toJSON for other classes - with empty value filtering
#' @method toJSON FSKAuthor
#' @export
toJSON.FSKAuthor <- function(x, ...) {
  # Clean empty values before serialization
  cleaned <- clean_empty_values(unclass(x))
  jsonlite::toJSON(cleaned, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKCreator
#' @export
toJSON.FSKCreator <- function(x, ...) {
  # Clean empty values before serialization
  cleaned <- clean_empty_values(unclass(x))
  jsonlite::toJSON(cleaned, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKReference
#' @export
toJSON.FSKReference <- function(x, ...) {
  # Clean empty values before serialization
  cleaned <- clean_empty_values(unclass(x))
  jsonlite::toJSON(cleaned, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKModelCategory
#' @export
toJSON.FSKModelCategory <- function(x, ...) {
  # Clean empty values before serialization  
  cleaned <- clean_empty_values(unclass(x))
  
  # Ensure basicProcess is always an array, even if it has one element
  if (!is.null(cleaned$basicProcess)) {
    # Convert to character vector if it isn't already
    cleaned$basicProcess <- as.character(cleaned$basicProcess)
    # Ensure it's always treated as array by wrapping in list if needed
    if (length(cleaned$basicProcess) == 1) {
      cleaned$basicProcess <- list(cleaned$basicProcess)
    }
  }
  
  jsonlite::toJSON(cleaned, auto_unbox = TRUE, pretty = TRUE, ...)
}

#' @method toJSON FSKParameter
#' @export
toJSON.FSKParameter <- function(x, ...) {
  # Clean empty values before serialization
  cleaned <- clean_empty_values(unclass(x))
  jsonlite::toJSON(cleaned, auto_unbox = TRUE, pretty = TRUE, ...)
}
