\name{gen.fuzzy}
\alias{gen.fuzzy}

\title{Generate 2D synthetic datasets with known fuzzy memberships}

\description{
Simulates several 2D datasets together with fuzzy cluster memberships \eqn{U}. The memberships are defined by analytic density/curve proximity rules (detailed below) so they can be used as "ground truth" for fuzzy clustering and visualization (see \code{\link{plot.fuzzy}}).
}

\usage{
gen.fuzzy(n = 500,
          dataset = c("gaussian", "hyperbolas", "spirals",
                      "wedges", "rings", "worms", "random"),
          k = NULL,
          noise = 0.1,
          covType = c("spherical", "diagonal", "rotated", "correlated"),
          seed = NULL)
}

\arguments{
  \item{n}{Total number of observations.}
  \item{dataset}{Which data generator to use, with options \code{"gaussian"}, \code{"hyperbolas"}, \code{"spirals"}, \code{"wedges"}, \code{"rings"}, \code{"worms"}, or \code{"random"}.}
  \item{k}{Number of clusters for \code{dataset="random"}, ignored otherwise; if \code{NULL}, defaults to \code{k = 20}.}
  \item{noise}{Additive noise or curve-thickness parameter for applicable generators (see Details).}
  \item{covType}{Covariance structure for \code{dataset="random"}; one of \code{"spherical"}, \code{"diagonal"}, \code{"rotated"}, \code{"correlated"}.}
  \item{seed}{Optional seed for reproducibility.}
}

\details{
Let \eqn{X \in \mathbb{R}^{n \times 2}} be the simulated observations and \eqn{U \in \mathbb{R}^{n \times k}} the fuzzy memberships. For each dataset, memberships are defined below and row-normalized to sum to 1.

\strong{gaussian (k = 3).}
Three Gaussian components with means \eqn{(-2,0)}, \eqn{(2,0)}, \eqn{(0,3)} and covariances \eqn{([1,0.3];[0.3,1])}, \eqn{([1, -0.3], [-0.3,1])} and \eqn{([0.8,0];[0,0.8])}.
If component sizes are \eqn{\pi_j}, then \eqn{U_{ij} \propto \pi_j  \phi_2(x_i | \mu_j, \Sigma_j)}.


\strong{hyperbolas (k = 5).}
One Gaussian near \eqn{(0,0)} and four hyperbola branches
\eqn{\{(x,y): (x\pm a)^2/b^2 - (y)^2/a^2=1\}} and its rotated or flipped analogues, sampled along
\eqn{t \in [-2,2]} with noise. For observation \eqn{x_i}, \eqn{w_{\text{ball}} = 50 \cdot \phi_2(x_i | (0,0), 0.2I_2)}, and
\eqn{w_{\text{hyp},\ell} = \exp \big(-d^2(x_i,\mathcal{C}_\ell)/(\sigma^2)\big)}, where \eqn{d(\cdot,\mathcal{C}_\ell)}
is minimum distance to branch \eqn{\ell} for curve \eqn{\mathcal{C}}. We set \eqn{U_{i\cdot} \propto w}.

\strong{spirals (k = 3).}
Three spirals generated by \eqn{(r,\theta) \mapsto (x,y) = ((0.5+0.8t)\cos(\theta_s+t),(0.5+0.8t)\sin(\theta_s+t))}
with shifts \eqn{\theta_s \in \{0,2\pi/3,4\pi/3\}}, with additive noise. For each spiral \eqn{s},
\eqn{d_s = \min_{t \in [0,\pi]} \|x_i - \gamma_s(t)\|}, where \eqn{\gamma_s(t)} is the parameterized spiral curve described above, and  \eqn{U_{is} \propto \exp \big(-d_s^2/\sigma^2)\big)}.
Note, if \eqn{\|x_i\| < 1}, set \eqn{U_{i\cdot} \leftarrow (1-\alpha)U_{i\cdot} + \alpha(1,1,1)/3} with \eqn{\alpha = 0.5e^{-\|x_i\|}} and
normalize after.


\strong{wedges (k = 8).}
Eight angular wedges with inner/outer radii \eqn{1} and \eqn{4}, respectively, with small gaps between wedges. For observation \eqn{x_i} with radius \eqn{r} and angle \eqn{\theta},
membership to wedge \eqn{j} is \eqn{U_{ij} \propto \exp \big(-\delta(\theta,\theta_j)^2/\sigma^2\big)},
where \eqn{\delta} is a wrapped angular distance to the wedge centre angle \eqn{\theta_j}.

\strong{rings (k = 3).}
For \eqn{x_i \in \mathbb{R}^2} and \eqn{r_i = \|x_i\|_2}, there are three concentric rings with radii
\eqn{R_j \in \{1, 2.5, 4\}} with widths \eqn{W_j \in \{0.3, 0.4, 0.5\}} for \eqn{j=1,2,3}. Let
\eqn{w_{ij} = \exp\left(-(r_i - R_j)^2/W_j^2\right)}, then \eqn{
U_{ij} = w_{ij}/\sum_{\ell=1}^{3} w_{i\ell}}.

\strong{worms (k = 4).}
Each worm \eqn{j} is a sinusoidal curve parameterized on \eqn{t \in [0,2\pi]} by \eqn{
\gamma_j(t) = \big(x(t), y_j(t)\big)} with \eqn{x(t)=2(t-\pi)}, \eqn{
y_j(t) = A_j \sin(f_j t + \phi_j) + y^{\mathrm{off}}_j}, with amplitudes \eqn{A_j},
frequencies \eqn{f_j}, phases \eqn{\phi_j}, and vertical offsets \eqn{y^{\mathrm{off}}_j}.
For observation \eqn{x_i \in \mathbb{R}^2}, the distance to worm \eqn{j} is
\eqn{d_j(x_i) = \min_{t \in [0,2\pi]} \|x_i - \gamma_j(t)\|_2,}. Then
\eqn{w_{ij} = \exp\left( -d_j(x_i)^2/\sigma^2\right)}, and \eqn{
U_{ij} = w_{ij}/\sum_{\ell=1}^{4} w_{i\ell}}.


\strong{random (k is user-specified).}
Mixture of \eqn{k} Gaussians with common covariance determined by \code{covType} with random centres in \eqn{[0,30]^2} and random cluster sizes. With mixture weights \eqn{\pi_j},
\eqn{U_{ij} \propto \pi_j  \phi_2(x_i | \mu_j, \Sigma).}
}

\section{Notes}{
The \code{noise} argument is used by \code{"gaussian"}, \code{"hyperbolas"}, \code{"spirals"}, \code{"rings"}, and \code{"worms"}; it is ignored by \code{"wedges"}.
}


\value{
A list with components:
\item{X}{An \eqn{n \times 2} numeric matrix of observations.}
\item{U}{An \eqn{n \times k} matrix of probabilistic/fuzzy cluster memberships.}
\item{y}{A vector length \eqn{n} of integers corresponding to hard cluster labels.}
\item{k}{Number of clusters.}
\item{centres, clusSz, covMatrix}{Returned only for \code{dataset="random"}: the centres, cluster sizes, and common covariance used.}
}

\seealso{
\code{\link{plot.fuzzy}}
}

\examples{
set.seed(1)

g <- gen.fuzzy(n = 600, dataset = "gaussian", seed = 1)
plot.fuzzy(g, plotFuzzy = TRUE, colorCluster = TRUE)

s <- gen.fuzzy(n = 450, dataset = "spirals", noise = 0.2, seed = 1)
plot.fuzzy(s, plotFuzzy = TRUE, colorCluster = FALSE)

r <- gen.fuzzy(n = 800, dataset = "random", k = 15, covType = "rotated", seed = 1)
plot.fuzzy(r, plotFuzzy = TRUE, colorCluster = TRUE)
}

\keyword{data generation}
\keyword{clustering}
\keyword{fuzzy}
\keyword{simulation}
