% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.entropy.R
\name{get.entropy}
\alias{get.entropy}
\title{Calculate Classification Entropy}
\usage{
get.entropy(object)
}
\arguments{
\item{object}{An object of class \code{"LCA"} or \code{"LPA"} returned by
\code{\link[LCPA]{LCA}} or \code{\link[LCPA]{LPA}} functions, or any other object containing:
\itemize{
\item \code{P.Z.Xn}: \eqn{N \times L} matrix of posterior class probabilities for each observation.
\item \code{params$P.Z}: Vector of length \eqn{L} with latent class prior probabilities.
}}
}
\value{
A numeric value between 0 and 1 representing the relative entropy (Nylund-Gibson et al., 2018; Clark et al., 2013):
\itemize{
\item 1.0: Perfect classification (each case belongs exclusively to one class)
\item 0.8-1.0: Good classification quality
\item 0.6-0.8: Moderate classification quality
\item < 0.6: Poor classification quality (consider model simplification)
}

Calculated using the formula:
\deqn{1 - \frac{\sum_{n=1}^N \sum_{l=1}^L -p_{nl} \ln(p_{nl})}{N \ln(L)}}
where:
\itemize{
\item \eqn{N} = Sample size
\item \eqn{L} = Number of latent classes
\item \eqn{p_{nl}} = Posterior probability of observation \eqn{n} belonging to class \eqn{l}
}
}
\description{
Computes the relative entropy statistic to evaluate classification quality in
Latent Class Analysis (LCA) or Latent Profile Analysis (LPA) models. Entropy
measures how accurately cases are assigned to latent classes based on posterior
probabilities, with values closer to 1 indicating better separation between classes.
}
\note{
The calculation includes a small constant (\code{1e-10}) to avoid \code{log(0)}
instability when posterior probabilities are exactly zero.

Values should be interpreted alongside other diagnostics (BIC, bootstrapped LRT)
as high entropy alone doesn't guarantee model validity. Low entropy may indicate:
\itemize{
\item Overly complex model (too many classes)
\item Poorly measured latent constructs
\item Violation of local independence assumption
}
}
\examples{
# Example with simulated data
set.seed(123)
data.obj <- sim.LCA(N = 500, I = 4, L = 2, IQ=0.9)
response <- data.obj$response

# Fit 2-class model with EM algorithm
\donttest{
fit.em <- LCA(response, L = 2, method = "EM", nrep = 10)

entropy_value <- get.entropy(fit.em)
cat("Classification entropy:", round(entropy_value, 3), "\n")

}

}
\references{
Nylund-Gibson, K., & Choi, A. Y. (2018). Ten frequently asked questions about latent class analysis. Translational Issues in Psychological Science, 4(4), 440-461. https://doi.org/10.1037/tps0000176

Clark, S. L., Muthén, B., Kaprio, J., D'Onofrio, B. M., Viken, R., & Rose, R. J. (2013). Models and Strategies for Factor Mixture Analysis: An Example Concerning the Structure Underlying Psychological Disorders. Structural Equation Modeling: A Multidisciplinary Journal, 20(4), 681-703. https://doi.org/10.1080/10705511.2013.824786
}
