#' Print Method for Logic Forest Models
#'
#' Prints the most important predictors and interactions from a fitted logic forest model,
#' along with their importance scores and frequency of occurrence.
#'
#' @param x An object of class \code{"logforest"}.
#' @param sortby Character string specifying whether to sort the output by \code{"importance"}
#' (default) or \code{"frequency"}.
#' @param ... Additional arguments (currently ignored).
#'
#' @details
#' This method displays a matrix of the top predictors and interactions from a logic forest model.
#' If \code{x$norm = TRUE}, the variable importance scores are normalized such that the largest
#' score is 1 and all other scores are scaled accordingly.
#'
#' @return No return value. This function is called for its side effect of printing.
#'
#' @author Bethany Wolf \email{wolfb@@musc.edu}
#' @seealso \code{\link{logforest}}
#' @export
print.logforest<-function(x, sortby = 'importance', ...)
{
  if (is(x)!= "logforest")
    stop("n not of class logforest")
  num<-x$numout
  OOBerr<-x$OOBmiss

  #MN added if statements to work with sortby option
  if (sortby == 'frequency') {
    pscore<-x$Predictor.importance
    pred.freq<-sort(x$Predictor.frequency[names(pscore)[1:num]], decreasing=TRUE, na.last = TRUE)
    piscore<-x$PI.importance
    vimp.freq<-sort(x$PI.frequency[names(piscore)[1:num]], decreasing=TRUE, na.last = TRUE)
  }
  else if (sortby == 'importance'){
    pscore<-sort(x$Predictor.importance, decreasing=TRUE, na.last = TRUE)
    pred.freq<-x$Predictor.frequency[names(pscore)[1:num]]
    piscore<-sort(x$PI.importance, decreasing=TRUE, na.last = TRUE)
    vimp.freq<-x$PI.frequency[names(piscore)[1:num]]
  }
  #MN edits end

  p.1st<-paste("Top ", num, " Predictors", sep="")
  pi.1st<-paste("Top ", num, " Interactions", sep="")
  if (x$norm==TRUE)
  {
    pred.score<-round(pscore[1:num]/pscore[1], digits=4)
    vimp.score<-round(piscore[1:num]/piscore[1], digits=4)
    p.cnames<-c(p.1st, "Normalized Predictor Importance","Frequency")
    pi.cnames<-c(pi.1st, "Normalized Interaction Importance","Frequency")
  }
  else {
    pred.score<-round(pscore[1:num], digits=2)
    vimp.score<-round(piscore[1:num], digits=2)
    p.cnames<-c(p.1st, "Predictor Importance","Frequency")
    pi.cnames<-c(pi.1st, "Interaction Importance","Frequency")
  }
  prds<-cbind(names(pred.score), pred.score, pred.freq)
  pis<-cbind(names(vimp.score), vimp.score, vimp.freq)
  colnames(prds)<-p.cnames
  colnames(pis)<-pi.cnames
  rownames(pis)<-rownames(prds)<-c(1:num)
  cat("Number of logic regression trees =", length(x$AllFits), sep=" ")
  cat("\n")
  if (x$model.type=="Classification") cat("Out of Bag Misclassification =", OOBerr, sep=" ")
  if (x$model.type=="Linear Regression") cat("Out of Bag MSE =", OOBerr, sep=" ")
  cat("\n")
  cat("\n")
  cat(num, " most important predictors \n", sep="")
  cat("\n")
  print.default(prds, quote=FALSE, print.gap=3)
  cat("\n")
  cat(num, " most important interactions \n", sep="")
  cat("\n")
  print.default(pis, quote=FALSE, print.gap=3)
}
