# LogisExpExt.R
#' Logistic-Exponential Extension Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic-Exponential Extension distribution.
#'
#' The Logistic-Exponential Extension distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic-Exponential Extension distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left[\exp \left\{-\lambda x e^{-\beta / x}\right\}
#' -1\right]^\alpha}  \, ; \, x > 0.
#' }
#'  
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dlogis.exp.ext()} — Density function
#'   \item \code{plogis.exp.ext()} — Distribution function
#'   \item \code{qlogis.exp.ext()} — Quantile function
#'   \item \code{rlogis.exp.ext()} — Random generation
#'   \item \code{hlogis.exp.ext()} — Hazard function
#' }
#'
#' @usage
#' dlogis.exp.ext(x, alpha, beta, lambda, log = FALSE)
#' plogis.exp.ext(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.exp.ext(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.exp.ext(n, alpha, beta, lambda)
#' hlogis.exp.ext(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.exp.ext}: numeric vector of (log-)densities
#'   \item \code{plogis.exp.ext}: numeric vector of probabilities
#'   \item \code{qlogis.exp.ext}: numeric vector of quantiles
#'   \item \code{rlogis.exp.ext}: numeric vector of random variates
#'   \item \code{hlogis.exp.ext}: numeric vector of hazard values
#' }
#'
#' @references   
#' Chaudhary,A.K., & Kumar, V.(2020). 
#' A Study on Properties and Real Data Applications of the Logistic 
#' Exponential Extension Distribution with Properties. \emph{International 
#' Journal of Latest Trends In Engineering and Technology (IJLTET)},
#' \bold{18(2)}, 20-30. 
#'  
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dlogis.exp.ext(x, 2.0, 5.0, 0.1)
#' plogis.exp.ext(x, 2.0, 5.0, 0.1)
#' qlogis.exp.ext(0.5, 2.0, 5.0, 0.1)
#' rlogis.exp.ext(10, 2.0, 5.0, 0.1)
#' hlogis.exp.ext(x, 2.0, 5.0, 0.1)
#' 
#' # Data
#' x <- stress31
#' # ML estimates
#' params = list(alpha=1.7919, beta=418.0473, lambda=0.1211)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.exp.ext, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.exp.ext, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params,
#'              dfun = dlogis.exp.ext, pfun=plogis.exp.ext, plot=TRUE)
#' print.gofic(res)
#' 
#' @name LogisExpExt
#' @aliases dlogis.exp.ext plogis.exp.ext qlogis.exp.ext rlogis.exp.ext hlogis.exp.ext
#' @export
NULL

#' @rdname LogisExpExt
#' @usage NULL
#' @export   
dlogis.exp.ext <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- exp(lambda * x[valid] * exp(-beta/x[valid]))
    num <- alpha * lambda *(1+(beta/x[valid])) *exp(-beta/x[valid])* u *((u - 1)^(alpha - 1))
    deno <- (1 + (u - 1) ^ alpha) ^ 2
    pdf[valid] <- num/deno
	}
    if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisExpExt
#' @usage NULL
#' @export  
plogis.exp.ext <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
    u <- exp(lambda * q[valid] * exp(-beta/q[valid]))
	tmp <- 1 + ((u - 1) ^ alpha)
    cdf[valid] <- 1 - (1/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisExpExt
#' @usage NULL
#' @export 
qlogis.exp.ext <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	-log(x)+(beta/x)+log((1/lambda) * log(1 + ((wi/(1 - wi)) ^ (1/alpha))))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 
#' @rdname LogisExpExt
#' @usage NULL
#' @export
rlogis.exp.ext <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.exp.ext(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisExpExt
#' @usage NULL
#' @export
hlogis.exp.ext <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
    
    nume <- dlogis.exp.ext(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.exp.ext(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 



