#' @title GPQ-Mix
#'
#' @description
#' Computes a confidence interval for the OVL between two populations under Gaussian and two-component Gaussian mixture models, or both
#' populations modeled as two-component Gaussian mixtures, using generalized inference.
#' @param x Numeric vector. Data from the first group. When \code{all_mix = FALSE}, this group
#'   is modeled as Gaussian.
#' @param y Numeric vector. Data from the second group, modeled as a two-component Gaussian mixture.
#' @param alpha confidence level.
#' @param interv Numeric vector of length 2. Search interval for intersection points between the
#'   corresponding densities.
#' @param k Number of simulated generalized pivotal quantities.
#' @param all_mix Logical. If \code{TRUE}, both groups are modeled as two-component Gaussian mixtures.
#'   If \code{FALSE}, only \code{y} is modeled as a mixture and \code{x} is Gaussian.
#'
#' @return confidence interval.
#'
#' @examples
#' set.seed(1)
#' x <- ifelse(runif(100) < 0.5,
#'             rnorm(100, mean = 0, sd = 1),
#'             rnorm(100, mean = 2, sd = 1))
#' y <- ifelse(runif(100) < 0.5,
#'             rnorm(100, mean = 2.5, sd = 1),
#'             rnorm(100, mean = 2, sd = 1))
#'res <- OVL.GPQ.mix(x, y, all_mix = TRUE, interv = c(-10, 10))
#'res$IC1
#'res$IC2
#' @export OVL.GPQ.mix
#' @importFrom stats dnorm pnorm sd quantile rmultinom rt rchisq runif rbeta
#' @importFrom mixtools normalmixEM

OVL.GPQ.mix <- function(x, y, alpha = 0.05, interv = c(0, 20), k = 1000, all_mix = FALSE) {
  n1 <- length(x)
  n2 <- length(y)

  if (all_mix == FALSE) {
    #Suponemos que x es la v.a que sigue una distribucón normal
    mu1_hat <- mean(x)
    sigma1_hat <- sd(x)

    #Para y suponemos que sigue una mixtura de normal de dos componentes
    EM_result_y <- try(normalmixEM(y), silent = TRUE)
    if (inherits(EM_result_y, "try-error")) {
      EM_result_y <- EM(y)
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$pi
      N_y <- try(colSums(EM_result_y$posterior), silent = TRUE)
    } else {
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$lambda
      N_y <- try(colSums(EM_result_y$posterior), silent = TRUE)
    }

    if (!inherits(N_y, "try-error") && all(N_y > 1)) {
      sigma2_hat <- sqrt(N_y / (N_y - 1)) * sigma2_hat
    }

    if (any(is.na(c(pi2_hat, mu2_hat, sigma2_hat)))) {
      return(list(IC1 = NA, IC2 = NA))
    }

    R_OVL <- numeric(k)

    for (i in seq_len(k)) {
      con2 <- 0
      repeat {
        ny <- rmultinom(1, n2, pi2_hat)
        if (all(ny >= 2)) break
        con2 <- con2 + 1
        if (con2 > 10) return(list(IC1 = NA, IC2 = NA))
      }

      t_x <- rt(1, df = n1 - 1)
      t_y <- rt(2, df = ny[,1] - 1)
      ji_x <- rchisq(1, df = n1 - 1)
      ji_y <- rchisq(2, df = ny[,1] - 1)

      if (any(!is.finite(c(t_x, t_y, ji_x, ji_y)))) next

      R_mu_x <- mu1_hat - t_x * sigma1_hat / sqrt(n1)
      R_mu_y <- mu2_hat - t_y * sigma2_hat / sqrt(ny[,1])

      R_var_x <- (1 / ji_x) * (n1 - 1) * sigma1_hat^2
      R_var_y <- (1 / ji_y) * (ny[,1] - 1) * sigma2_hat^2

      R_sigma_x <- sqrt(R_var_x)
      R_sigma_y <- sqrt(R_var_y)

      R_pi2_1 <- ifelse(runif(1) < 0.5,
                        rbeta(1, ny[1,1], n2 - ny[1,1] + 1),
                        rbeta(1, ny[1,1] + 1, n2 - ny[1,1]))
      R_pi_2 <- c(R_pi2_1, 1 - R_pi2_1)

      intersection_function <- function(x) {
        dnorm(x, R_mu_x, R_sigma_x) - dnorm_mixture(x, R_mu_y, R_sigma_y, R_pi_2)
      }

      intersec <- encontrar_raices(intersection_function, interv)
      if (length(intersec) == 0) next

      R_OVL[i] <- OVL_mix(R_mu_x, R_mu_y, R_sigma_x, R_sigma_y, 1, R_pi_2, intersec)
    }

    return(list(
      IC1 = quantile(R_OVL, alpha / 2, na.rm = TRUE),
      IC2 = quantile(R_OVL, 1 - alpha / 2, na.rm = TRUE)
    ))
  } else {
    n1 <- length(x)
    n2 <- length(y)

    EM_result_x <- try({normalmixEM(x)}, silent = TRUE)

    if (inherits(EM_result_x, "try-error")) {
      EM_result_x <- EM(x)
      mu1_hat <- EM_result_x$mu
      sigma1_hat <- EM_result_x$sigma
      pi1_hat <- EM_result_x$pi
    } else {
      mu1_hat <- EM_result_x$mu
      sigma1_hat <- EM_result_x$sigma
      pi1_hat <- EM_result_x$lambda
    }

    if(any(is.na(pi1_hat)) || any(is.na(mu1_hat)) || any(is.na(sigma1_hat))) {
      return(list(IC1 = NA, IC2 = NA))
    }

    EM_result_y <- try({normalmixEM(y)}, silent = TRUE)

    if (inherits(EM_result_y, "try-error")) {
      EM_result_y <- EM(y)
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$pi
    } else {
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$lambda
    }

    if(any(is.na(pi2_hat)) || any(is.na(mu2_hat)) || any(is.na(sigma2_hat))) {
      return(list(IC1 = NA, IC2 = NA))
    }

    R_OVL <- numeric(k)

    for (i in 1:k) {
      con1 <- 0
      repeat {
        nx <- rmultinom(1, n1, pi1_hat)
        if (all(nx >= 2)) {break} else {
          con1 <- con1 + 1
        }

        if(con1 > 10){
          return(list(IC1 = NA, IC2 = NA))
        }
      }
      con2 <- 0
      repeat {
        ny <- rmultinom(1, n2, pi2_hat)
        if (all(ny >= 2)) {break} else {
          con2 <- con2 + 1
        }
        if(con2 > 10){
          return(list(IC1 = NA, IC2 = NA))
        }
      }

      t_x_1 <- rt(1, df = nx[1,1] - 1)
      t_x_2 <- rt(1, df = nx[2,1] - 1)
      ji_x_1 <- rchisq(1, df = nx[1,1] - 1)
      ji_x_2 <- rchisq(1, df = nx[2,1] - 1)

      t_y_1 <- rt(1, df = ny[1,1] - 1)
      t_y_2 <- rt(1, df = ny[2,1] - 1)
      ji_y_1 <- rchisq(1, df = ny[1,1] - 1)
      ji_y_2 <- rchisq(1, df = ny[2,1] - 1)

      R_mu_x_1 <- mu1_hat[1] - t_x_1 * sigma1_hat[1] / sqrt(nx[1,1])
      R_mu_x_2 <- mu1_hat[2] - t_x_2 * sigma1_hat[2] / sqrt(nx[2,1])
      R_mu_x <- c(R_mu_x_1, R_mu_x_2)

      R_var_x_1 <- (1 / ji_x_1) * (nx[1,1] - 1) * (sigma1_hat[1])^2
      R_var_x_2 <- (1 / ji_x_2) * (nx[2,1] - 1) * (sigma1_hat[2])^2
      R_sigma_x_1 <- sqrt(R_var_x_1)
      R_sigma_x_2 <- sqrt(R_var_x_2)
      R_sigma_x <- c(R_sigma_x_1, R_sigma_x_2)

      R_pi1_1 <- ifelse(runif(1) < 1/2,
                        rbeta(1, nx[1,1], n1 - nx[1,1] + 1),
                        rbeta(1, nx[1,1] + 1, n1 - nx[1,1]))
      R_pi1_2 <- 1 - R_pi1_1
      R_pi_1 <- c(R_pi1_1, R_pi1_2)

      R_mu_y_1 <- mu2_hat[1] - t_y_1 * sigma2_hat[1] / sqrt(ny[1,1])
      R_mu_y_2 <- mu2_hat[2] - t_y_2 * sigma2_hat[2] / sqrt(ny[2,1])
      R_mu_y <- c(R_mu_y_1, R_mu_y_2)

      R_var_y_1 <- (1 / ji_y_1) * (ny[1,1] - 1) * (sigma2_hat[1])^2
      R_var_y_2 <- (1 / ji_y_2) * (ny[2,1] - 1) * (sigma2_hat[2])^2
      R_sigma_y_1 <- sqrt(R_var_y_1)
      R_sigma_y_2 <- sqrt(R_var_y_2)
      R_sigma_y <- c(R_sigma_y_1, R_sigma_y_2)

      R_pi2_1 <- ifelse(runif(1) < 1/2,
                        rbeta(1, ny[1,1], n2 - ny[1,1] + 1),
                        rbeta(1, ny[1,1] + 1, n2 - ny[1,1]))
      R_pi2_2 <- 1 - R_pi2_1
      R_pi_2 <- c(R_pi2_1, R_pi2_2)

      intersection_function <- function(x) {
        dnorm_mixture(x, R_mu_x, R_sigma_x, R_pi_1) - dnorm_mixture(x, R_mu_y, R_sigma_y, R_pi_2)
      }

      intersec <- encontrar_raices(intersection_function, interv)
      if(length(intersec) == 0){next}
      R_OVL[i] <- OVL_mix(R_mu_x, R_mu_y, R_sigma_x, R_sigma_y, R_pi_1 ,R_pi_2, intersec)
    }

    IC1 <- quantile(R_OVL, alpha / 2, na.rm = TRUE)
    IC2 <- quantile(R_OVL, 1 - alpha / 2, na.rm = TRUE)

    return(list(IC1 = IC1, IC2 = IC2))
  }
}

