
#' List the available configured `OmopViewer` themes
#'
#' @return A character vector with the available `OmopViewer` themes.
#' @export
#'
#' @examples
#' library(OmopViewer)
#'
#' omopViewerThemes()
#'
omopViewerThemes <- function() {
  unlist(availableThemes(), use.names = FALSE)
}

availableThemes <- function() {
  list(
    # visOmopResults themes
    visOmopResults = system.file("brand", package = "visOmopResults") |>
      list.files() |>
      purrr::keep(\(x) stringr::str_ends(string = x, pattern = "\\.yml")) |>
      stringr::str_replace_all(pattern = "\\.yml$", replacement = ""),
    # OmopViewer themes
    OmopViewer = system.file("brand", package = "OmopViewer") |>
      list.files() |>
      purrr::keep(\(x) stringr::str_ends(string = x, pattern = "\\.yml")) |>
      stringr::str_replace_all(pattern = "\\.yml$", replacement = "")
  )
}
validateTheme <- function(theme, call = parent.frame()) {
  if (is.null(theme)) theme <-  "default"

  omopgenerics::assertCharacter(x = theme, length = 1, call = call)

  if (!endsWith(x = theme, suffix = ".yml")) {
    at <- availableThemes()
    choices <- unlist(at, use.names = FALSE)
    omopgenerics::assertChoice(x = theme, choices = choices, length = 1, call = call)
    pkg <- ifelse(theme %in% at$visOmopResults, "visOmopResults", "OmopViewer")
    file <- system.file("brand", paste0(theme, ".yml"), package = pkg)
  } else {
    if (!file.exists(theme)) {
      cli::cli_abort(c(x = "File {.path {theme}} does not exist."))
    }
    file <- theme
    pkg <- "none"
  }

  # read theme
  content <- brand.yml::read_brand_yml(path = file)

  # correct font units
  if (isTRUE(!is.na(as.numeric(content$typography$base$size)))) {
    content$typography$base$size <- paste0(content$typography$base$size, "pt")
  }

  return(content)
}
getThemes <- function() {
  omopViewerThemes() |>
    rlang::set_names() |>
    purrr::map(\(x) bslib::bs_theme(brand = validateTheme(x)))
}

validateLogo <- function(logo, theme, call = parent.frame()) {
  if (is.null(logo) & "logo" %in% names(theme)) {
    logo <- theme$logo$path
  }
  omopgenerics::assertCharacter(logo, length = 1, null = TRUE, call = call)
  return(logo)
}
