#' Import and Merge PCR Data
#'
#' Reads the raw machine export and merges it with a user-defined layout map.
#'
#' @param raw_file Path to the machine excel output (.xls or .xlsx).
#' @param map_file Path to the user-defined CSV map (Cols: Well, Sample, Gene, Group).
#' @param skip_rows Number of rows of metadata to skip in the raw file (default 0).
#' @return A clean, merged tibble ready for analysis.
#' @export
#' @examples
#' # Locate the sample data bundled with the package
#' my_raw <- system.file("extdata", "experiment_data.xlsx", package = "PlateVision")
#' my_map <- system.file("extdata", "plate_map.csv", package = "PlateVision")
#'
#' # Run import
#' if(file.exists(my_raw) && file.exists(my_map)) {
#'   df <- import_plate(raw_file = my_raw, map_file = my_map)
#'   head(df)
#' }
import_plate <- function(raw_file, map_file, skip_rows = 0) {

  # 1. Read Raw Machine Output
  raw_data <- readxl::read_excel(raw_file, skip = skip_rows)

  # Normalize Column Names
  colnames(raw_data)[1] <- "Well"
  ct_col_index <- grep("Ct|Cq|CT", colnames(raw_data), ignore.case = TRUE)[1]

  if(is.na(ct_col_index)) {
    stop("Could not find a column named 'Ct' or 'Cq' in the raw file.")
  }

  # Select and Clean
  clean_raw <- raw_data %>%
    dplyr::select(Well, Ct = dplyr::all_of(ct_col_index)) %>%
    dplyr::mutate(Ct = suppressWarnings(as.numeric(as.character(Ct))))

  # 2. Read Map and Merge
  plate_map <- read.csv(map_file, stringsAsFactors = FALSE)

  merged_df <- dplyr::left_join(clean_raw, plate_map, by = "Well") %>%
    dplyr::filter(!is.na(Sample))

  return(merged_df)
}
