#' Plot Fold Change (Bar Chart)
#'
#' Aggregates biological replicates and plots the Group Mean +/- SEM.
#'
#' @param results Output from calculate_ddct().
#' @return A ggplot object.
#' @export
#' @examples
#' # Mock results data
#' results <- data.frame(
#'   Gene = c("GeneA", "GeneA", "GeneB", "GeneB"),
#'   Group = c("WT", "Treated", "WT", "Treated"),
#'   fold_change = c(1, 5, 1, 0.5)
#' )
#'
#' plot_bars(results)
plot_bars <- function(results) {

  plot_data <- results %>%
    dplyr::group_by(Gene, Group) %>%
    dplyr::summarise(
      mean_fc = mean(fold_change, na.rm = TRUE),
      se_fc = sd(fold_change, na.rm = TRUE) / sqrt(dplyr::n()),
      .groups = "drop"
    ) %>%
    dplyr::mutate(
      ymin = mean_fc - se_fc,
      ymax = mean_fc + se_fc
    )

  ggplot2::ggplot(plot_data, ggplot2::aes(x = Gene, y = mean_fc, fill = Group)) +
    ggplot2::geom_bar(stat = "identity", position = ggplot2::position_dodge(0.9), color = "black") +
    ggplot2::geom_errorbar(ggplot2::aes(ymin = ymin, ymax = ymax), width = 0.2, position = ggplot2::position_dodge(0.9)) +
    ggplot2::scale_fill_brewer(palette = "Paired") +
    ggplot2::labs(title = "Relative Gene Expression", subtitle = "Mean Fold Change +/- SEM", y = "Fold Change (2^-ddCt)", x = "Target Gene") +
    ggplot2::theme_classic() +
    ggplot2::geom_hline(yintercept = 1, linetype = "dashed", color = "grey50")
}

#' Plot Volcano (Log2FC vs P-value)
#'
#' Creates a volcano plot to identify significantly regulated genes.
#'
#' @param results Output from calculate_ddct().
#' @param p_cutoff Significance threshold (default 0.05).
#' @param fc_cutoff Log2 Fold Change threshold (default 1).
#' @return A ggplot object.
#' @export
#' @examples
#' # Mock results data
#' results <- data.frame(
#'   log2_fc = c(2.5, -3.0, 0.1, 1.5),
#'   p_val = c(0.001, 0.0001, 0.8, 0.04)
#' )
#'
#' plot_volcano(results)
plot_volcano <- function(results, p_cutoff = 0.05, fc_cutoff = 1) {

  volcano_data <- results %>%
    dplyr::mutate(
      Significance = dplyr::case_when(
        p_val < p_cutoff & log2_fc > fc_cutoff ~ "Upregulated",
        p_val < p_cutoff & log2_fc < -fc_cutoff ~ "Downregulated",
        TRUE ~ "Not Significant"
      )
    )

  ggplot2::ggplot(volcano_data, ggplot2::aes(x = log2_fc, y = -log10(p_val), color = Significance)) +
    ggplot2::geom_point(alpha = 0.8, size = 3) +
    ggplot2::geom_hline(yintercept = -log10(p_cutoff), linetype = "dashed", color = "grey") +
    ggplot2::geom_vline(xintercept = c(-fc_cutoff, fc_cutoff), linetype = "dashed", color = "grey") +
    ggplot2::scale_color_manual(values = c("Upregulated" = "red", "Downregulated" = "blue", "Not Significant" = "grey")) +
    ggplot2::labs(title = "Volcano Plot", x = "Log2 Fold Change", y = "-Log10 P-value") +
    ggplot2::theme_minimal()
}
