% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Gauss_ST_F.R
\name{Gauss.st.F}
\alias{Gauss.st.F}
\title{Simulate a spatio-temporal Gaussian random field on a regular grid}
\usage{
Gauss.st.F(
  xlim = c(0, 1),
  ylim = c(0, 1),
  tlim = c(0, 1),
  par1 = c(1, 0.05),
  par2 = c(1, 0.06),
  sigmas = c(0.5, 0.5, 1),
  grid = c(15L, 15L, 10L)
)
}
\arguments{
\item{xlim, ylim, tlim}{Numeric vectors of length 2 giving the ranges for the spatial and temporal
axes. Defaults are \code{c(0,1)} for each.}

\item{par1}{Numeric vector of length 2 giving the temporal covariance parameters
\code{c(variance, scale)} for an exponential covariance \eqn{var * exp(-d/scale)}.}

\item{par2}{Numeric vector of length 2 giving the spatial covariance parameters
\code{c(variance, scale)} for an exponential covariance \eqn{var * exp(-d/scale)}.}

\item{sigmas}{Numeric vector of length 3 specifying the weights
\eqn{(\sigma_1,\sigma_2,\sigma_3)} for combining the spatial, temporal, and
spatio-temporal components of the field.}

\item{grid}{Integer vector of length 3 giving the number of grid points in the \eqn{x}, \eqn{y}, and \eqn{t}
directions.}
}
\value{
A list with components:
\describe{
\item{Z}{Numeric array of dimension \code{c(nx, ny, nt)} containing simulated field values.}
\item{xcoord}{Numeric vector of length \code{nx} with x-grid coordinates.}
\item{ycoord}{Numeric vector of length \code{ny} with y-grid coordinates.}
\item{tcoord}{Numeric vector of length \code{nt} with time-grid coordinates.}
}
}
\description{
Simulates a space-time Gaussian random field on a regular \eqn{(x,y,t)} grid.
The field is returned as a 3D array and can be used as a latent field for
log-Gaussian Cox process (LGCP) simulation.
}
\details{
The simulated field is a weighted sum of three independent Gaussian components:
\deqn{Z(x,y,t) = \sigma_1 Z_s(x,y) + \sigma_2 Z_t(t) + \sigma_3 Z_{st}(x,y,t),}
where \eqn{Z_s} is a purely spatial field, \eqn{Z_t} is a purely temporal field,
and \eqn{Z_{st}} is a spatio-temporal field with separable exponential covariance
in space and time.

The function uses \code{\link[MASS]{mvrnorm}} for multivariate normal simulation
and \code{\link[fields]{rdist}} to compute pairwise distances for covariance
matrix construction.

Spatial and temporal covariances are exponential. The spatio-temporal component uses a separable
covariance \eqn{C_{st}((u,t),(u',t')) = C_s(u,u') C_t(t,t')}. Simulation is performed via Cholesky
factors without constructing the full \eqn{(nx*ny*nt) \times (nx*ny*nt)} covariance matrix.
}
\examples{

 if (requireNamespace("MASS", quietly = TRUE) && requireNamespace("fields", quietly = TRUE)) {
  set.seed(1)
  field <- Gauss.st.F(
    xlim = c(0, 1), ylim = c(0, 1), tlim = c(0, 1),
    par1 = c(1, 0.05), par2 = c(1, 0.06),
    sigmas = c(0.5, 0.5, 1),
    grid = c(15, 15, 10)
  )
# Inspect dimensions and visualize one time slice
dim(field$Z)
image(field$xcoord, field$ycoord, field$Z[, , 1],
      main = "Gaussian Random Field (t = 1)",
      col = RColorBrewer::brewer.pal(11, "Spectral"))
  }

}
\references{
Ghorbani M., Vafaei N., Dvořák J., Myllymäki M. (2021).
Testing the first-order separability hypothesis for spatio-temporal point patterns.
\emph{Computational Statistics and Data Analysis}, \bold{161}, 107245.
}
\seealso{
\code{\link[MASS]{mvrnorm}}, \code{\link[fields]{rdist}}
}
\author{
Mohammad Ghorbani \email{mohammad.ghorbani@slu.se}\cr
Nafiseh Vafaei \email{nafiseh.vafaei@slu.se}
}
