% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Simulate_ST_DPP.R
\name{rstDPP}
\alias{rstDPP}
\title{Simulate a spatio-temporal Determinantal Point Process (DPP) based on spectral density}
\usage{
rstDPP(
  mode = c("stationary", "inhom"),
  model = c("S", "NS"),
  spectral = c("exp", "matern"),
  alpha_s,
  alpha_t = NULL,
  lambda_max,
  nu = 2,
  eps = 1,
  lambda_s = NULL,
  lambda_non_s = NULL,
  grid_size = 4
)
}
\arguments{
\item{mode}{Character. Type of dependence model:
\describe{
\item{\code{"Stationary"}}{ A homogeneous spatio-temporal DPP is generated.}
\item{\code{"Inhomogeneous"}}{A non-homogeneous spatio-temporal DPP is generated.}
}}

\item{model}{Character. Type of dependence model:
\describe{
\item{\code{"S"}}{Separable spatio-temporal covarince function model, where space and time components are separable.}
\item{\code{"NS"}}{Non-separable spatio-temporal model, allowing interaction between space and time.}
}}

\item{spectral}{Character. Type of spectral density function to use:
\code{"exp"} for exponential spectral form or \code{"matern"} for a Mat\'ern-type spectral model.}

\item{alpha_s}{Numeric. Spatial decay or range parameter in the spectral density.}

\item{alpha_t}{Numeric. Temporal decay or range parameter in the spectral density.}

\item{lambda_max}{Numeric. The maximum intensity value used for thinning.
Must be specified.}

\item{nu}{Numeric. Smoothness parameter for the Mat\'ern-type spectral density
(only relevant if \code{spectral = "matern"}). Default is \code{2}.}

\item{eps}{Numeric. Degree of separability for the Mat\'ern model:
\code{eps = 0} corresponds to full non-separability,
\code{eps = 1} yields complete separability,
and intermediate values provide partial separability.}

\item{lambda_s}{Optional. Intensity function \code{lambda_s(u, t)} for separable models.
If not provided, a default function is used.}

\item{lambda_non_s}{Optional. Intensity function \code{lambda_non_s(u, t)} for
non-separable models. If not provided, a default function is used.}

\item{grid_size}{Numeric. Half-width of the spatio-temporal frequency grid.
The total grid size is \code{(2 * grid_size + 1)^3}.}
}
\value{
A numeric matrix with three columns (\code{x}, \code{y}, \code{t}) representing
the retained spatio-temporal events after thinning.
}
\description{
Generates a realization of a spatio-temporal determinantal point process (DPP)
using a user-defined spectral density model.
The function supports both separable (\code{model = "S"}) and non-separable
(\code{model = "NS"}) dependence structures and allows for either exponential
or Mat\'ern-type spectral densities.
The simulation is performed on a 3D spatio-temporal frequency grid and can
incorporate user-specified intensity functions for thinning.
}
\details{
This function implements a spectral simulation method for spatio-temporal DPPs,
following the theoretical framework introduced in
Vafaei et al. (2023) and  Ghorbani et al. (2025).

The algorithm proceeds as follows:
\enumerate{
\item Construct a 3D grid of spatial and temporal frequency components
\eqn{(\omega_x, \omega_y, \tau)}.
\item Evaluate the chosen spectral density \eqn{\phi(\omega, \tau)} across the grid.
\item Use the resulting spectral values as eigenvalues to simulate a realization
of a DPP via \eqn{spatstat.model::rdpp()}.
\item Optionally apply thinning using a user-defined intensity function
\eqn{\lambda(u, t)}, scaled by \code{lambda_max}, to induce inhomogeneity.
}

Two spectral families are supported:
\itemize{
\item \strong{Exponential form:}
\deqn{
      \phi(\omega, \tau) \propto
        \exp\left[-(\pi \alpha_s |\omega|)^2\right]
        \left(1 + 4 (\pi \alpha_t \tau)^2\right)^{-1}.
    }
\item \strong{Mat\'ern-type form:}
\deqn{
      \phi_{\epsilon}(\omega, \tau) \propto
        \left(
          \alpha_s^2 \alpha_t^2
          + \alpha_t^2 |\omega|^2
          + \alpha_s^2 \tau^2
          + \epsilon |\omega|^2 \tau^2
        \right)^{-\nu},
    }
where \eqn{\epsilon \in [0, 1]} determines the degree of separability between
space and time.
}

This framework enables simulation of spatio-temporal point patterns that exhibit
varying degrees of spatial–temporal dependence, providing a versatile tool for
evaluating separability tests and modeling non-separable dynamics.
}
\examples{

# Simulate a stationary separable Mat\'ern ST-DPP
 if (requireNamespace("spatstat", quietly = TRUE)) {
sim <- rstDPP(
  mode     = "stationary",
  model    = "S",
  spectral = "matern",
  alpha_s  = 10,
  alpha_t  = 4.7,
  nu       = 2,
  eps      = 1,
  lambda_max = 70,
  grid_size  = 2
)
 plot_stDPP(sim, type = "3D",  alpha_s = 10, alpha_t = 4.7)
# example 2
# Generate realization
sim <- rstDPP(mode = "stationary",
                model = "S",
                spectral = "matern",
                alpha_s = 10, alpha_t = 4.7,
                nu = 2,
               eps = 1,
               lambda_s = 70,
               lambda_non_s = NULL,
               grid_size = 2,
               lambda_max=70)
head(sim)
}

}
\references{
Vafaei, N., Ghorbani, M., Ganji, M., and Myllymäki, M. (2023).
Spatio-temporal determinantal point processes.
\emph{arXiv:2301.02353}.

Ghorbani, M., Vafaei, N., and Myllymäki, M. (2025).
A kernel-based test for the first-order separability of spatio-temporal point processes.
\emph{TEST}, 34, 580-611.
https://doi.org/10.1007/s11749-025-00972-y
}
\seealso{
\code{\link{plot_stpp}} for visualizing spatio-temporal point patterns.
}
