% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filters.R
\name{filter_hpgc}
\alias{filter_hpgc}
\title{Grant-Chan Embedded Hodrick-Prescott Filter}
\usage{
filter_hpgc(
  y,
  prior_config = "weak",
  n_chains = 4,
  iterations = 20000,
  burnin = 5000,
  verbose = FALSE
)
}
\arguments{
\item{y}{Numeric vector of the time series. Will be internally scaled
for numerical stability.}

\item{prior_config}{Character string or list specifying prior configuration.
Options: "weak" (default), "informative", or "empirical". Alternatively,
a named list with prior parameters (see Details).}

\item{n_chains}{Integer number of MCMC chains to run. Default is 4.}

\item{iterations}{Integer total number of MCMC iterations per chain.
Default is 20000.}

\item{burnin}{Integer number of burn-in iterations to discard. Default is
5000.}

\item{verbose}{Logical indicating whether to print progress messages.}
}
\value{
A list of class "signaly_hpgc" containing:
\itemize{
\item \code{trend}: Numeric vector of posterior mean trend
\item \code{trend_lower}: Numeric vector of 2.5 percent posterior quantile
\item \code{trend_upper}: Numeric vector of 97.5 percent posterior quantile
\item \code{cycle}: Numeric vector of posterior mean cycle component
\item \code{cycle_lower}: Numeric vector of 2.5 percent posterior quantile
\item \code{cycle_upper}: Numeric vector of 97.5 percent posterior quantile
\item \code{draws}: List of posterior draws for all parameters
\item \code{diagnostics}: Convergence diagnostics including R-hat and ESS
\item \code{dic}: Deviance Information Criterion
\item \code{settings}: Parameters used in the analysis
}
}
\description{
Implements the Bayesian Hodrick-Prescott filter embedded in an unobserved
components model, as developed by Grant and Chan (2017). This approach
provides principled uncertainty quantification for the extracted trend
through Markov Chain Monte Carlo sampling.
}
\details{
The Grant-Chan model decomposes the observed series \eqn{y_t} as:
\deqn{y_t = \tau_t + c_t}

where \eqn{\tau_t} is the trend component and \eqn{c_t} is the cyclical
component.

\strong{Trend Model (Second-Order Markov Process)}:
\deqn{\Delta^2 \tau_t = u_t^\tau, \quad u_t^\tau \sim N(0, \sigma_\tau^2)}

This implies the trend growth rate follows a random walk, allowing for
time-varying trend growth.

\strong{Cycle Model (Stationary AR(2))}:
\deqn{c_t = \phi_1 c_{t-1} + \phi_2 c_{t-2} + u_t^c, \quad u_t^c \sim N(0, \sigma_c^2)}

with stationarity constraints on \eqn{\phi}.
}
\section{Prior Configurations}{

\describe{
\item{weak}{Diffuse priors allowing data to dominate. Good for initial
exploration.}
\item{informative}{Tighter priors based on typical macroeconomic
dynamics. Suitable when strong smoothness is desired.}
\item{empirical}{Priors calibrated from data moments. Balances
flexibility with data-driven regularization.}
}

Custom priors can be specified as a list with elements:
\itemize{
\item \code{phi_mu}: Mean of phi prior (2-vector)
\item \code{phi_v_i}: Precision matrix for phi prior (2x2)
\item \code{gamma_mu}: Mean of gamma (initial trend growth) prior
\item \code{gamma_v_i}: Precision matrix for gamma prior
\item \code{s_tau}: Upper bound for uniform prior on \eqn{\sigma_\tau^2}
\item \code{s_c_shape}: Shape parameter for inverse-gamma prior on
\eqn{\sigma_c^2}
\item \code{s_c_rate}: Rate parameter for inverse-gamma prior on
\eqn{\sigma_c^2}
}
}

\section{Relationship to Standard HP Filter}{

The standard HP filter solves:
\deqn{\min_\tau \sum_t (y_t - \tau_t)^2 + \lambda \sum_t (\Delta^2 \tau_t)^2}

The Grant-Chan approach embeds this within a probabilistic model where
\eqn{\lambda = \sigma_c^2 / \sigma_\tau^2}, allowing this ratio to be
estimated from data with full uncertainty quantification.
}

\examples{
\donttest{
set.seed(123)
y <- cumsum(rnorm(100)) + sin(seq(0, 4*pi, length.out = 100))
result <- filter_hpgc(y, prior_config = "weak", n_chains = 2,
                      iterations = 5000, burnin = 1000)
plot(y, type = "l", col = "gray")
lines(result$trend, col = "red", lwd = 2)
}

}
\references{
Grant, A. L., & Chan, J. C. C. (2017). Reconciling output gaps: Unobserved
components model and Hodrick-Prescott filter. Journal of Economic Dynamics
and Control, 75, 114-121. \doi{10.1016/j.jedc.2016.12.007}

Chan, J., Koop, G., Poirier, D. J., & Tobias, J. L. (2019). Bayesian
Econometric Methods (2nd ed.). Cambridge University Press.
}
\seealso{
\code{\link{filter_wavelet}}, \code{\link{filter_emd}}
}
