#' File, directory, and path CLI assertions
#'
#' These functions wrap filesystem-related checks from \pkg{checkmate} and
#' either return the underlying check result (`check_*_cli()`) or raise a
#' `cli::cli_abort()` error via [make_assertion()] (`assert_*_cli()`).
#' They are useful for validating input and output locations in user-facing
#' functions, ensuring that required files and directories exist (or can be
#' created) and that paths are safe to write to, while emitting consistent
#' CLI-styled error messages.
#'
#' @param x Object to be checked.
#'   This is the value whose type, length, or other properties are validated.
#'
#' @param ... Additional arguments passed on to the corresponding
#'   \pkg{checkmate} function (e.g. `lower`, `upper`, `any.missing`,
#'   `all.missing`, `min.len`, `null.ok`).
#'
#' @param .var.name Character scalar used in error messages to refer to
#'   the checked object. Defaults to [checkmate::vname()], which tries to
#'   infer the variable name from the calling context.
#'
#' @param add Optional [checkmate::AssertCollection] to which
#'   assertion failures are added instead of triggering an immediate
#'   error. Defaults to `NULL`, which causes a `cli::cli_abort()` on
#'   failure.
#'
#' - Files:
#'   - [check_file_cli()], [assert_file_cli()] validate file paths with
#'     properties such as existence, readability, and writability.
#'   - [check_file_exists_cli()], [assert_file_exists_cli()] specifically check
#'     that a file exists.
#' - Directories:
#'   - [check_directory_cli()], [assert_directory_cli()] wrap
#'     [checkmate::checkDirectory()] for directory-like paths (often used for
#'     input).
#'   - [check_directory_exists_cli()], [assert_directory_exists_cli()] ensure
#'     that a directory already exists.
#' - Output paths:
#'   - [check_path_for_output_cli()], [assert_path_for_output_cli()] ensure that
#'     a path is suitable for writing output (e.g., directory exists, file does
#'     not unexpectedly overwrite unless allowed).
#'
#' In all `assert_*_cli()` functions, `x` is the object being checked and
#' `.var.name` is used only for error message construction; the return value is
#' always `x` (invisibly) on success.
#'
#' @returns
#'
#' - `check_*_cli()` functions return `TRUE` on success or a character vector
#'   describing the failure, exactly like the corresponding
#'   \pkg{checkmate} checks.
#' - `assert_*_cli()` functions return `x` invisibly on success and either:
#'   - raise a `cli::cli_abort()` error with bullet-style messages, or
#'   - push messages into an [checkmate::AssertCollection] if `add` is supplied.
#'
#' @examples
#'
#' # Files:
#'
#' # Create a temporary file for demonstration
#' f <- tempfile(fileext = ".csv")
#' write.csv(data.frame(x = 1:3), f, row.names = FALSE)
#'
#' # Check that f is a readable file
#' check_file_cli(f, access = "r")
#' check_file_exists_cli(f)
#'
#' # Clearly invalid path
#' bad_file <- file.path(tempdir(), "does_not_exist.csv")
#' try(check_file_cli(bad_file, access = "r"))          # failure
#' try(check_file_exists_cli(bad_file))                 # failure
#'
#' assert_file_cli(f, access = "r")
#' try(assert_file_cli(bad_file, access = "r"))
#'
#' assert_file_exists_cli(f)
#' try(assert_file_exists_cli(bad_file))
#'
#' # Directories:
#'
#' d <- tempdir()
#'
#' # Generic directory check (using checkDirectory)
#' check_directory_cli(d)
#'
#' # Directory must exist
#' check_directory_exists_cli(d)
#'
#' bad_dir <- file.path(tempdir(), "no_such_dir_xyz")
#' try(check_directory_cli(bad_dir))                    # typically failure
#' try(check_directory_exists_cli(bad_dir))             # failure
#'
#' assert_directory_cli(d)
#' try(assert_directory_cli(bad_dir))
#'
#' assert_directory_exists_cli(d)
#' try(assert_directory_exists_cli(bad_dir))
#'
#' # Output paths:
#'
#' # Valid output path in an existing directory
#' out_file <- file.path(tempdir(), "results.rds")
#' check_path_for_output_cli(out_file)
#'
#' # Directory part does not exist
#' bad_out <- file.path(tempdir(), "no_such_dir", "results.rds")
#' try(check_path_for_output_cli(bad_out))              # failure
#'
#' assert_path_for_output_cli(out_file)
#' try(assert_path_for_output_cli(bad_out))
#'
#' # Combine with other assertions in a pipeline:
#' out_file2 <- file.path(tempdir(), "summary.csv")
#' assert_directory_exists_cli(dirname(out_file2))
#' assert_path_for_output_cli(out_file2)
#'
#' @seealso
#'   [checkmate::checkFile()], [checkmate::check_file_exists()],
#'   [checkmate::checkDirectory()], [checkmate::check_directory_exists()],
#'   [checkmate::check_path_for_output()], [make_assertion()], [assert_cli()]
#'
#' @name checkcli-files
#' @family checkCLI
#' @import checkmate
#' @import cli
#' @importFrom glue glue
#' @importFrom purrr map2

#' @rdname checkcli-files
#' @export
check_file_cli <- function(...) {
  checkmate::checkFile(...)
}

#' @rdname checkcli-files
#' @export
assert_file_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::checkFile(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-files
#' @export
check_file_exists_cli <- function(...) {
  checkmate::check_file_exists(...)
}

#' @rdname checkcli-files
#' @export
assert_file_exists_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_file_exists(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-files
#' @export
check_directory_cli <- function(...) {
  checkmate::checkDirectory(...)
}

#' @rdname checkcli-files
#' @export
assert_directory_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::checkDirectory(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-files
#' @export
check_directory_exists_cli <- function(...) {
  checkmate::check_directory_exists(...)
}

#' @rdname checkcli-files
#' @export
assert_directory_exists_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_directory_exists(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-files
#' @export
check_path_for_output_cli <- function(...) {
  checkmate::check_path_for_output(...)
}

#' @rdname checkcli-files
#' @export
assert_path_for_output_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_path_for_output(x, ...),
      .var.name,
      add
    )
  )
}


