% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trialr_crm_selector.R
\name{get_trialr_crm}
\alias{get_trialr_crm}
\title{Get an object to fit the CRM model using the trialr package.}
\usage{
get_trialr_crm(
  parent_selector_factory = NULL,
  skeleton,
  target,
  model,
  tite = FALSE,
  ...
)
}
\arguments{
\item{parent_selector_factory}{optional object of type
\code{\link{selector_factory}} that is in charge of dose selection before
this class gets involved. Leave as NULL to just use CRM from the start.}

\item{skeleton}{Dose-toxicity skeleton, a non-decreasing vector of
probabilities.}

\item{target}{We seek a dose with this probability of toxicity.}

\item{model}{character string identifying which model form to use. Options
include empiric, logistic, logistic2. The model form chosen determines which
prior hyperparameters are required. See \code{\link[trialr]{stan_crm}}
for more details.}

\item{tite}{FALSE to use regular CRM; TRUE to use TITE-CRM. See Description.}

\item{...}{Extra args are passed to \code{\link[trialr]{stan_crm}}.}
}
\value{
an object of type \code{\link{selector_factory}} that can fit the
CRM model to outcomes.
}
\description{
This function returns an object that can be used to fit a CRM model using
methods provided by the trialr package.

Dose selectors are designed to be daisy-chained together to achieve different
behaviours. This class is a **resumptive** selector, meaning it carries on
when the previous dose selector, where present, has elected not to continue.
For example, this allows instances of this class to be preceded by a selector
that follows a fixed path in an initial escalation plan, such as that
provided by \code{\link{follow_path}}. In this example, when the observed
trial outcomes deviate from that initial plan, the selector following the
fixed path elects not to continue and responsibility passes to this class.
See Examples.

The time-to-event variant, TITE-CRM, is used when you specify
\code{tite = TRUE}. This weights the observations to allow dose-selections
based on partially observed outcomes.
}
\examples{
skeleton <- c(0.05, 0.1, 0.25, 0.4, 0.6)
target <- 0.25
# The model to use must be specified in trialr:
model1 <- get_trialr_crm(skeleton = skeleton, target = target,
                         model = 'empiric', beta_sd = 1.34)
# Refer to the trialr documentation for more details on model forms.
outcomes <- '1NNN 2NTN'
model1 \%>\% fit(outcomes) \%>\% recommended_dose()

# But we can provide extra args to trialr that are than passed onwards to
# the call to trialr::stan_crm to override the defaults.
# For example, if we want the one-parameter logistic model, we run:
model2 <- get_trialr_crm(skeleton = skeleton, target = target,
                         model = 'logistic', a0 = 3,
                         beta_mean = 0, beta_sd = 1)
model2 \%>\% fit(outcomes) \%>\% recommended_dose()
# And, if we want the two-parameter logistic model, we run:
model3 <- get_trialr_crm(skeleton = skeleton, target = target,
                         model = 'logistic2',
                         alpha_mean = 0, alpha_sd = 2,
                         beta_mean = 0, beta_sd = 1)
model3 \%>\% fit(outcomes) \%>\% recommended_dose()

# We can use an initial dose-escalation plan, a pre-specified path that
# should be followed until trial outcomes deviate, at which point the CRM
# model takes over. For instance, if we want to use two patients at each of
# the first three doses in the absence of toxicity, irrespective the model's
# advice, we would run:
model1 <- follow_path('1NN 2NN 3NN') \%>\%
  get_trialr_crm(skeleton = skeleton, target = target, model = 'empiric',
                 beta_sd = 1.34)

# If outcomes match the desired path, the path is followed further:
model1 \%>\% fit('1NN 2N') \%>\% recommended_dose()

# But when the outcomes diverge:
model1 \%>\% fit('1NN 2T') \%>\% recommended_dose()

# Or the pre-specified path comes to an end:
model1 \%>\% fit('1NN 2NN 3NN') \%>\% recommended_dose()
# ...the CRM model takes over.

}
\references{
Kristian Brock (2020). trialr: Clinical Trial Designs in 'rstan'.
R package version 0.1.5. https://github.com/brockk/trialr

Kristian Brock (2019). trialr: Bayesian Clinical Trial Designs in R and Stan.
arXiv preprint arXiv:1907.00161.
}
