#' Likelihood-based evidence ratio test
#'
#' Computes an effect estimate, a Wald-style uncertainty interval, and a
#' likelihood-based evidence ratio (log10 scale) from a single statistical model.
#'
#' @param ... Arguments defining the data and model.
#'
#' @return An object of class \code{evidence_result}.
#'
#' @examples
#' x <- sleep$extra[sleep$group == 1]
#' evidence_test(x)
#'
#' tbl <- matrix(c(30, 70, 20, 80), nrow = 2)
#' evidence_test(tbl)
#'
#' evidence_test(mpg ~ wt, data = mtcars, coef = "wt")
#'
#' @export
evidence_test <- function(...) {
	UseMethod("evidence_test")
}

#' @export
evidence_test.numeric <- function(x, y = NULL, mu = 0, sigma = NULL,
																	paired = FALSE, level = 0.95, ...) {
	if (!is.null(y)) {
		.evidence_mean_two_sample(x, y, paired = paired, level = level)
	} else {
		.evidence_mean_one_sample(x, mu = mu, sigma = sigma, level = level)
	}
}

#' @export
evidence_test.matrix <- function(x,
																 scale = c("log_or", "risk_diff", "log_rr"),
																 level = 0.95, ...) {
	scale <- match.arg(scale)
	.evidence_table_2x2(x, scale = scale, level = level)
}

#' @export
evidence_test.table <- function(x,
																scale = c("log_or", "risk_diff", "log_rr"),
																level = 0.95, ...) {
	scale <- match.arg(scale)
	.evidence_table_2x2(as.matrix(x), scale = scale, level = level)
}

#' @export
evidence_test.formula <- function(formula, data, family = stats::gaussian(),
																	coef, prior_sd = NULL, level = 0.95, ...) {
	.evidence_regression(formula, data, family, coef, prior_sd, level)
}
