% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VARIMA.R
\name{VARIMA}
\alias{VARIMA}
\alias{report.VARIMA}
\alias{forecast.VARIMA}
\alias{fitted.VARIMA}
\alias{residuals.VARIMA}
\alias{tidy.VARIMA}
\alias{glance.VARIMA}
\alias{generate.VARIMA}
\alias{IRF.VARIMA}
\title{Estimate a VARIMA model}
\usage{
VARIMA(formula, identification = NULL, ...)

\method{forecast}{VARIMA}(
  object,
  new_data = NULL,
  specials = NULL,
  bootstrap = FALSE,
  times = 5000,
  ...
)

\method{fitted}{VARIMA}(object, ...)

\method{residuals}{VARIMA}(object, ...)

\method{tidy}{VARIMA}(x, ...)

\method{glance}{VARIMA}(x, ...)

\method{report}{VARIMA}(object, ...)

\method{generate}{VARIMA}(x, new_data, specials, ...)

\method{IRF}{VARIMA}(x, new_data, specials, impulse = NULL, orthogonal = FALSE, ...)
}
\arguments{
\item{formula}{Model specification (see "Specials" section).}

\item{identification}{The identification technique used to estimate the model. Possible options include NULL (automatic selection), "kronecker_indices" (Kronecker index identification), and "scalar_components" (scalar component identification). More details can be found in the "Identification" section below.}

\item{...}{Further arguments for arima}

\item{object}{A model for which forecasts are required.}

\item{new_data}{A tsibble containing the time points and exogenous regressors to produce forecasts for.}

\item{specials}{(passed by \code{\link[fabletools:forecast]{fabletools::forecast.mdl_df()}}).}

\item{bootstrap}{If \code{TRUE}, then forecast distributions are computed using simulation with resampled errors.}

\item{times}{The number of sample paths to use in estimating the forecast distribution when \code{bootstrap = TRUE}.}

\item{x}{A fitted model.}

\item{impulse}{A character string specifying the name of the variable that is shocked (the impulse variable).}

\item{orthogonal}{If TRUE, orthogonalised impulse responses will be computed.}
}
\value{
A model specification.

A one row tibble summarising the model's fit.
}
\description{
Estimates a VARIMA model of a given order.
}
\details{
Exogenous regressors and \code{\link{common_xregs}} can be specified in the model
formula.
}
\section{Specials}{


\subsection{pdq}{
The \code{pdq} special is used to specify non-seasonal components of the model.
\preformatted{
pdq(p = 0:5, d = 0:2, q = 0:5)
}

\tabular{ll}{
\code{p}      \tab The order of the non-seasonal auto-regressive (AR) terms. If multiple values are provided, the one which minimises \code{ic} will be chosen. \cr
\code{d}      \tab The order of integration for non-seasonal differencing. If multiple values are provided, one of the values will be selected via repeated KPSS tests. \cr
\code{q}      \tab The order of the non-seasonal moving average (MA) terms. If multiple values are provided, the one which minimises \code{ic} will be chosen. \cr
}
}

\subsection{xreg}{
Exogenous regressors can be included in an VARIMA model without explicitly using the \code{xreg()} special. Common exogenous regressor specials as specified in \code{\link{common_xregs}} can also be used. These regressors are handled using \code{\link[stats:model.frame]{stats::model.frame()}}, and so interactions and other functionality behaves similarly to \code{\link[stats:lm]{stats::lm()}}.

The inclusion of a constant in the model follows the similar rules to \code{\link[stats:lm]{stats::lm()}}, where including \code{1} will add a constant and \code{0} or \code{-1} will remove the constant. If left out, the inclusion of a constant will be determined by minimising \code{ic}.

\preformatted{
xreg(...)
}

\tabular{ll}{
\code{...}      \tab Bare expressions for the exogenous regressors (such as \code{log(x)})
}
}
}

\section{Identification}{


\subsection{Kronecker indices (\code{"kronecker_indices"}, the default)}{
Determines the structural complexity and degrees of freedom in a VARIMA model
by analysing the singularities in the polynomial matrices.

Kronecker indices represent the structural properties of the VARIMA system,
focusing on the relationship between system inputs and outputs. These indices
define the minimal realisation of the model, helping to determine the order
and complexity of each equation in the system. They are particularly suited
for capturing dynamic dependencies in multivariate systems with cointegrated
processes. This is particularly useful for understanding system-wide
dependencies and cointegrating relationships, however it is computationally
intensive for models with many variables.
}
\subsection{Scalar components (\code{"scalar_components"})}{
Simplifies VARIMA models by identifying univariate "scalar components" that
combine linear combinations of variables into simpler sub-models. This uses
canonical correlation analysis (CCA) to find linear combinations of variables
with minimal lag orders. These combinations are then modeled as simpler
ARIMA processes reducing the complexity and dimensionality of the full
VARIMA model. This is particularly useful for identifying models with many
variables, however it assumes good separability of the components.
}
\subsection{No identification (\code{"none"})}{
Directly estimates the model as specified by \code{p}, \code{d}, and \code{q}. This allows
all coefficients up to lag \code{p} and \code{q} (for the AR and MA components) to be
freely estimated. This can be problematic as the estimation of parameters
without identification is not unique.

Identification is necessary for VARIMA models to ensure that the model is
parsimonious, unique, and interpretable. Without proper identification, the
model can become overly complex, redundant, or ambiguous, making estimation
and interpretation challenging.
}

For a more detailed comparison of identification methods, refer to
Athanasopoulos et al (2012).
}

\examples{
\dontshow{if (!requireNamespace("MTS", quietly = TRUE)) withAutoprint(\{ # examplesIf}
# The MTS package is required for VARIMA models
# Install it with: install.packages("MTS")
\dontshow{\}) # examplesIf}
\dontshow{if (requireNamespace("tsibbledata", quietly = TRUE)) withAutoprint(\{ # examplesIf}
library(tsibbledata)
library(MTS)

aus_production \%>\% 
  autoplot(vars(Beer, Cement))

fit <- aus_production \%>\%
  model(VARIMA(vars(Beer, Cement) ~ pdq(4,1,1), identification = "none"))

fit


fit \%>\%
  forecast(h = 50) \%>\%
  autoplot(tail(aus_production, 100))

fitted(fit)
residuals(fit)
tidy(fit)
glance(fit)
report(fit)
generate(fit, h = 10)
IRF(fit, h = 10, impulse = "Beer")
\dontshow{\}) # examplesIf}
}
\references{
Athanasopoulos, George, D. S. Poskitt, and Farshid Vahid. "Two Canonical VARMA Forms: Scalar Component Models Vis-à-Vis the Echelon Form." Econometric Reviews 31, no. 1 (January 2012): 60–83. https://doi.org/10.1080/07474938.2011.607088.
}
\seealso{
\code{\link[MTS:VARMA]{MTS::VARMA()}}, \code{\link[MTS:Kronfit]{MTS::Kronfit()}}.
}
