% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visualize_gsea.R
\name{visualize_gsea}
\alias{visualize_gsea}
\title{Visualize GSEA results}
\usage{
visualize_gsea(
  gsea_results,
  plot_type = "enrichment_plot",
  n_pathways = 20,
  sort_by = "p.adjust",
  colors = NULL,
  abundance = NULL,
  metadata = NULL,
  group = NULL,
  network_params = list(),
  heatmap_params = list(),
  pathway_label_column = NULL,
  scale = NULL
)
}
\arguments{
\item{gsea_results}{A data frame containing GSEA results from the pathway_gsea function}

\item{plot_type}{A character string specifying the visualization type: "enrichment_plot", "dotplot", "barplot", "network", or "heatmap"}

\item{n_pathways}{An integer specifying the number of pathways to display}

\item{sort_by}{A character string specifying the sorting criterion: "NES", "pvalue", or "p.adjust"}

\item{colors}{A vector of colors for the visualization}

\item{abundance}{A data frame containing the original abundance data (required for heatmap visualization)}

\item{metadata}{A data frame containing sample metadata (required for heatmap visualization)}

\item{group}{A character string specifying the column name in metadata that contains the grouping variable (required for heatmap visualization)}

\item{network_params}{A list of parameters for network visualization}

\item{heatmap_params}{A list of parameters for heatmap visualization}

\item{pathway_label_column}{A character string specifying which column to use for pathway labels.
If NULL (default), the function will automatically use 'pathway_name' if available, otherwise 'pathway_id'.
This allows for custom labeling when using annotated GSEA results.}

\item{scale}{Optional palette/scale for customizing colors. Accepts: (1) a character vector of colors,
(2) a function that returns colors given an integer (e.g., viridisLite::viridis), or
(3) a ggplot2 scale object (e.g., ggplot2::scale_fill_gradientn(...)).
When NULL, defaults keep current behavior. Applies to: enrichment_plot (fill, continuous),
dotplot (color, continuous), barplot (fill, discrete Positive/Negative), network (color, diverging around 0),
heatmap (main heatmap col; row annotation stays default unless overridden in heatmap_params).}
}
\value{
A ggplot2 object or ComplexHeatmap object
}
\description{
This function creates various visualizations for Gene Set Enrichment Analysis (GSEA) results.
It automatically detects whether pathway names are available (from gsea_pathway_annotation())
and uses them for better readability, falling back to pathway IDs if names are not available.
}
\examples{
\dontrun{
# Load example data
data(ko_abundance)
data(metadata)

# Prepare abundance data
abundance_data <- as.data.frame(ko_abundance)
rownames(abundance_data) <- abundance_data[, "#NAME"]
abundance_data <- abundance_data[, -1]

# Run GSEA analysis (using camera method - recommended)
gsea_results <- pathway_gsea(
  abundance = abundance_data,
  metadata = metadata,
  group = "Environment",
  pathway_type = "KEGG",
  method = "camera"
)

# Create enrichment plot with pathway IDs (default)
visualize_gsea(gsea_results, plot_type = "enrichment_plot", n_pathways = 10)

# Annotate results for better pathway names
annotated_results <- gsea_pathway_annotation(
  gsea_results = gsea_results,
  pathway_type = "KEGG"
)

# Create plots with readable pathway names
visualize_gsea(annotated_results, plot_type = "dotplot", n_pathways = 20)
visualize_gsea(annotated_results, plot_type = "barplot", n_pathways = 15)

# Create network plot with custom labels
visualize_gsea(annotated_results, plot_type = "network", n_pathways = 15)

# Use custom column for labels (if available)
visualize_gsea(annotated_results, plot_type = "barplot",
               pathway_label_column = "pathway_name", n_pathways = 10)

# Create heatmap
visualize_gsea(
  annotated_results,
  plot_type = "heatmap",
  n_pathways = 15,
  abundance = abundance_data,
  metadata = metadata,
  group = "Environment"
)
}
}
