% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write.r
\name{h5_write}
\alias{h5_write}
\title{Write an R Object to HDF5}
\usage{
h5_write(data, file, name, attr = NULL, as = "auto", compress = TRUE)
}
\arguments{
\item{data}{The R object to write. Supported: \code{numeric}, \code{complex},
\code{logical}, \code{character}, \code{factor}, \code{raw}, \code{matrix}, \code{data.frame}, \code{integer64},
\code{POSIXt}, \code{NULL}, and nested \code{list}s.}

\item{file}{The path to the HDF5 file.}

\item{name}{The name of the dataset or group to write (e.g., "/data/matrix").}

\item{attr}{The name of an attribute to write.
\itemize{
\item If \code{NULL} (default), \code{data} is written as a dataset or group at the path \code{name}.
\item If provided (string), \code{data} is written as an attribute named \code{attr} attached to the object \code{name}.
}}

\item{as}{The target HDF5 data type. Defaults to \code{"auto"}.
See the \strong{Data Type Selection} section for a full list of valid options
(including \code{"int64"}, \code{"bfloat16"}, \code{"utf8[n]"}, etc.) and how to map
sub-components of \code{data}.}

\item{compress}{Compression configuration.
\itemize{
\item \code{TRUE} (default): Enables compression (zlib level 5).
\item \code{FALSE} or \code{0}: Disables compression.
\item Integer \code{1-9}: Specifies the zlib compression level.
}}
}
\value{
Invisibly returns \code{file}. This function is called for its side effects.
}
\description{
Writes an R object to an HDF5 file, creating the file if it does not exist.
This function acts as a unified writer for datasets, groups (lists), and attributes.
}
\section{Writing Scalars}{


By default, \code{h5_write} saves single-element vectors as 1-dimensional arrays.
To write a true HDF5 scalar, wrap the value in \code{I()} to treat it "as-is."
\subsection{Examples}{

\if{html}{\out{<div class="sourceCode">}}\preformatted{h5_write(I(5), file, "x") # Creates a scalar dataset
h5_write(5, file, "x")    # Creates a 1D array of length 1
}\if{html}{\out{</div>}}
}
}

\section{Data Type Selection (\code{as} Argument)}{


By default, \code{as = "auto"} will automatically select the most appropriate
data type for the given object. For numeric types, this will be the smallest
type that can represent all values in the vector. For character types,
\code{h5lite} will use a ragged vs rectangular heuristic, favoring small file
size over fast I/O. For R data types not mentioned below, see
\code{vignette("data-types")} for information on their fixed mappings to HDF5
data types.
\subsection{Numeric and Logical Vectors}{

When writing a numeric or logical vector, you can specify one of the
following storage types for it:
\itemize{
\item \strong{Floating Point:} \code{"float16"}, \code{"float32"}, \code{"float64"}, \code{"bfloat16"}
\item \strong{Signed Integer:} \code{"int8"}, \code{"int16"}, \code{"int32"}, \code{"int64"}
\item \strong{Unsigned Integer:} \code{"uint8"}, \code{"uint16"}, \code{"uint32"}, \code{"uint64"}
}

\strong{NOTE:} \code{NA} values must be stored as \code{float64}. \code{NaN}, \code{Inf}, and \code{-Inf}
must be stored as a floating point type.
\subsection{Examples}{

\if{html}{\out{<div class="sourceCode">}}\preformatted{h5_write(1:100, file, "big_ints", as = "int64")
h5_write(TRUE,  file, "my_bool",  as = "float32")
}\if{html}{\out{</div>}}
}

}

\subsection{Character Vectors}{

You can control whether character vectors are stored as variable or fixed
length strings, and whether to use UTF-8 or ASCII encoding.
\itemize{
\item \strong{Variable Length Strings:} \code{"utf8"}, \code{"ascii"}
\item \strong{Fixed Length Strings:}
\itemize{
\item \code{"utf8[]"} or \code{"ascii[]"} (length is set to the longest string)
\item \code{"utf8[n]"} or \code{"ascii[n]"} (where \code{n} is the length in bytes)
}
}

\strong{NOTE:} Variable-length strings allow for \code{NA} values but cannot be
compressed on disk. Fixed-length strings allow for compression but do not
support \code{NA}.
\subsection{Examples}{

\if{html}{\out{<div class="sourceCode">}}\preformatted{h5_write(letters[1:5],    file, "len10_strs", as = "utf8[10]")
h5_write(c('X', 'Y', NA), file, "var_chars",  as = "ascii")
}\if{html}{\out{</div>}}
}

}

\subsection{Lists, Data Frames, and Attributes}{

Provide a named vector to apply type mappings to sub-components of \code{data}.
Set \code{"skip"} as the type to skip a specific component.
\itemize{
\item \strong{Specific Name:} \code{"col_name" = "type"} (e.g., \code{c(score = "float32")})
\item \strong{Specific Attribute:} \code{"@attr_name" = "type"}
\item \strong{Class-based:} \code{".integer" = "type"}, \code{".numeric" = "type"}
\item \strong{Class-based Attribute:} \code{"@.character" = "type"}, \code{"@.logical" = "type"}
\item \strong{Global Fallback:} \code{"." = "type"}
\item \strong{Global Attribute Fallback:} \code{"@." = "type"}
}
\subsection{Examples}{

\if{html}{\out{<div class="sourceCode">}}\preformatted{# To strip attributes when writing:
h5_write(data, file, 'no_attrs_obj', as = c('@.' = "skip"))

# To only save the `hp` and `wt` columns:
h5_write(mtcars, file, 'my_df', as = c('hp' = "auto", 'wt' = "float32", '.' = "skip"))
}\if{html}{\out{</div>}}
}

}
}

\section{Dimension Scales}{

\code{h5lite} automatically writes \code{names}, \code{row.names}, and \code{dimnames} as
HDF5 dimension scales. Named vectors will generate an \verb{<name>_names}
dataset. A data.frame with row names will generate an \verb{<name>_rownames}
dataset (column names are saved internally in the original dataset).
Matrices will generate \verb{<name>_rownames} and \verb{<name>_colnames} datasets.
Arrays will generate \verb{<name>_dimscale_1}, \verb{<name>_dimscale_2}, etc.
Special HDF5 metadata attributes link the dimension scales to the dataset.
The dimension scales can be relocated with \code{h5_move()} without breaking the
link.
}

\examples{
file <- tempfile(fileext = ".h5")

# 1. Writing Basic Datasets
h5_write(1:10, file, "data/integers")
h5_write(rnorm(10), file, "data/floats")
h5_write(letters[1:5], file, "data/chars")

# 2. Writing Attributes
# Write an object first
h5_write(1:10, file, "data/vector")
# Attach an attribute to it using the 'attr' parameter
h5_write(I("My Description"), file, "data/vector", attr = "description")
h5_write(I(100), file, "data/vector", attr = "scale_factor")

# 3. Controlling Data Types
# Store values as 32-bit signed integers
h5_write(1:5, file, "small_ints", as = "int32")

# 4. Writing Complex Structures (Lists/Groups)
my_list <- list(
  meta    = list(id = 1, name = "Experiment A"),
  results = matrix(runif(9), 3, 3),
  valid   = I(TRUE)
)
h5_write(my_list, file, "experiment_1", as = c(id = "uint16"))

# 5. Writing Data Frames (Compound Datasets)
df <- data.frame(
  id    = 1:5,
  score = c(10.5, 9.2, 8.4, 7.1, 6.0),
  grade = factor(c("A", "A", "B", "C", "D"))
)
h5_write(df, file, "records/scores", as = c(grade = "ascii[1]"))

# 6. Fixed-Length Strings
h5_write(c("A", "B"), file, "fixed_str", as = "ascii[10]")

# 7. Review the file structure
h5_str(file)

# 8. Clean up
unlink(file)
}
\seealso{
\code{\link[=h5_read]{h5_read()}}
}
