#-------------------------------------------------------------------------------
#' @encoding UTF-8
#' 
#' @title Hasse diagram of the layout structure
#' @description Returns a Hasse diagram of the layout structure of an experimental design 
#' 
#' @param datadesign A data frame, list or environment (or object coercible by \code{\link[base]{as.data.frame}} to a data frame) containing the factors in the experimental design. The data frame should \strong{only} include the factors/columns that the user wants to include in the Hasse diagram. All factors are treated as categorical. Moreover, the first two letters of factor names are used for interactions between factors so it is advised that these be unique.
#' @param randomfacsid An optional vector specifying whether the factors are defined as fixed (entry = 0) or random (entry = 1). The default choice is NULL and the function automatically sets all entries to 0. The length of the vector should be equal to the number of factors in the design, i.e., the length of the vector should be equal to the number of columns of the argument \code{datadesign}.
#' @param showLS logical. If \code{FALSE} then generation of the Hasse diagram is suppressed. The default is TRUE.
#' @param showpartialLS logical. If \code{FALSE} then the partial crossing between structural objects (using dotted connecting lines) is not illustrated on the Hasse diagram of the layout structure. The default is TRUE.
#' @param showdfLS logical. If \code{FALSE} then the structural object label is not displayed on the Hasse diagram of the layout structure. The default is TRUE.
#' @param check.confound.df logical. If \code{FALSE} then the check for confounded degrees of freedom is not performed. The default is TRUE.
#' @param maxlevels.df logical. If \code{FALSE} then the potential maximum number of levels of a generalised factor is removed from the structural object label on the Hasse diagram of the layout structure. The default is TRUE.
#' @param table.out logical. If \code{TRUE} then a table that shows the relationships between the structural objects in the layout structure is printed. The default is FALSE.
#' @param pdf logical. If \code{TRUE} then a pdf file containing the Hasse diagram of the layout structure is generated. The default is FALSE, i.e., a pdf file is not generated.
#' @param example File name for the pdf output file containing the Hasse diagram. The default is set to "example".
#' @param outdir Location of the pdf output file if \code{pdf=TRUE}. The default is set to \code{NULL} and in this case the pdf output file containing the Hasse diagram output is stored to a temporary file. To specify a permanent location this argument needs be specified.
#' @param hasse.font The name of the font family used for all text included in the Hasse diagram. Standard and safe font families to choose are "sans", "serif", and "mono". If the design's factor labels contain Unicode characters, or for consistency with Hasse diagrams of restricted layout structures using \link[hassediagrams]{hasserls}, a Unicode friendly font family should be selected. For more details on Unicode friendly family options see the Details section in the \link[hassediagrams]{hasserls} documentation. The default is "sans".
#' @param produceBWPlot logical. If \code{TRUE} then the Hasse diagram will be generated in black and white format. The default is set to FALSE, i.e., a coloured version of the plot is produced.
#' @param structural.colour The colour of the structural lines that connect structural objects on the Hasse diagram. The default colour is "grey".
#' @param structural.width The width of the structural lines on the Hasse diagram. The default width is 2.
#' @param partial.colour The colour of the partial crossing dotted lines of the connecting objects on the Hasse diagram. The default colour is "orange".
#' @param partial.width The width of the partial crossing dotted lines on the Hasse diagram. The default width is 1.5.
#' @param objects.colour The colour of the labels of the structural objects on the Hasse diagram. The default colour is "mediumblue".
#' @param df.colour The colour of the degrees of the freedom labels on the Hasse diagram. The default colour is "red".
#' @param larger.fontlabelmultiplier The large font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram where there are four or less objects at that level in the diagram. The default is 1.
#' @param middle.fontlabelmultiplier The medium font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram involving a factor that is equivalent to a generalised factor. The default is 1.
#' @param smaller.fontlabelmultiplier The small font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram where there are five or more objects at that level of the diagram. The default is 1.
#'
#'
#' @details The hasselayout function generates the Hasse diagram of the layout structure of the experimental design, as described in Bate and Chatfield (2016a). The diagram consists of a set of structural objects, corresponding to the factors and generalised factors, and the relationships between the structural objects (either crossed, nested, partially crossed or equivalent), as defined by the structure of the experimental design.
#'
#' The function requires a dataframe containing only the factors corresponding to the experimental factors that define the experimental design (i.e., no response should be included). 
#'
#' In the dataframe the levels of the factors must be uniquely identified and have a physical meaning, otherwise the function will not correctly identify the nesting/crossing of the factors. For example, consider an experiment consisting of Factor A (with k levels) that nests Factor B (with q levels per level of Factor A). The levels of Factor B should be labelled 1 to k x q and not 1 to q (repeated k times). 
#'
#' Where present, two partially crossed factors are illustrated on the diagram with a dotted line connecting them. This feature can be excluded using the \code{showpartialLS} option.
#'
#' The maximum number of possible levels of each generalised factor, along with the actual number present in the design and the "skeleton ANOVA" degrees of freedom, can be included in the structural object label on the Hasse diagram.
#'
#' Using the randomfacsid argument the factors that correspond to random effects can be highlighted by underlining them on the Hasse diagram. The vector should be equal to the number of factors in the design and consist of fixed (entry = 0) or random (entry = 1) values.
#'
#' The \code{\link[hassediagrams]{hasselayout}} function evaluates the design in order to identify if there are any confounded degrees of freedom across the design. It is not recommended to perform this evaluation for large designs due to the potential high computational cost. This can be controlled using the check.confound.df = FALSE option. 
#'
#' @return The function \code{\link[hassediagrams]{hasselayout}} returns:
#'
#' 1. The Hasse diagram of the layout structure (if \code{showLS=TRUE}).
#'
#' 2. The layout structure table shows the relationships between the structural objects in the layout structure (if \code{table.out=TRUE}). 
#' The individual entries in the table consist of blanks on the main diagonal and 0's, (0)'s or 1's elsewhere. 
#' If the factor (or generalised factor) corresponding to the ith row and the factor (or generalised factor) corresponding to the 
#' jth column are fully crossed, then a 0 is entered in the (i,j)th entry in the table. If these factors (or generalised factors) are 
#' partially crossed, or the ith row factor (or generalised factor) only has one level and nests the jth column factor (or generalised factor), 
#' then the (i,j)th entry is (0). If the ith row factor (or generalised factor) is nested within the jth column factor (or generalised factor), 
#' then a 1 is entered in the (i,j)th entry. If two factors (or generalised factors) are equivalent, then they share a single row and column in 
#' the table, where the row and column headers include both factor (or generalised factor) names, separated by an "=" sign.
#'
#' 3. If there are confounded degrees of freedom, a table of the structural objects and a description of the associated degrees of freedom is printed. 
#' Confounded degrees of freedom often indicate that the design objects are not specified appropriately (for example a factor or pseudofactor or supremum may have been missed from the design supplied) 
#' or that the design needs to be changed. However, sometimes once appropriate randomization is performed the layout structure, modified to account for this, no longer contains confounded degrees of freedom.
#' 
#' In addition, the function returns an object of class \code{"ls"},
#' which is a list with the following components:
#'
#' \describe{
#'   \item{\code{str_objects}}{
#'     A character vector containing the names of all structural objects
#'     derived from the design.
#'   }
#'
#'   \item{\code{rand_template}}{
#'     A character vector giving a template for use in
#'     \code{\link[hassediagrams]{hasserls}}.  
#'     Structural objects not present in the restricted layout structure are set
#'     to \code{"NULL"}.
#'   }
#'
#'   \item{\code{str_rand_map}}{
#'     A two-column matrix pairing/mapping all structural objects with their suggested
#'     randomisation entries.  
#'     This matrix assists users when constructing the
#'     \code{rand.objects} argument for \code{\link[hassediagrams]{hasserls}}.
#'   }
#'
#'   \item{\code{notes}}{
#'     A short explanatory message describing how to use the returned objects
#'     when preparing inputs for \code{\link[hassediagrams]{hasserls}}.
#'   }
#' }
#'
#' The returned object allows programmatic access to the structural objects
#' and can be passed to \code{summary()} or \code{print()} methods for compact inspection.
#' 
#' @author
#' Damianos Michaelides, Simon Bate, and Marion Chatfield
#'  
#' @references
#' Bate, S.T. and Chatfield, M.J. (2016a), Identifying the structure of the experimental design. Journal of Quality Technology, 48, 343-364.
#' 
#' Bate, S.T. and Chatfield, M.J. (2016b), Using the structure of the experimental design and the randomization to construct a mixed model. Journal of Quality Technology, 48, 365-387.
#' 
#' Box, G.E.P., Hunter, J.S., and Hunter, W.G., (1978), Statistics for Experimenters. Wiley.
#' 
#' Joshi, D.D. (1987), Linear Estimation and Design of Experiments. Wiley Eastern, New Delhi.
#' 
#' Williams, E.R., Matheson, A.C. and Harwood, C.E. (2002), Experimental design and analysis for tree improvement. 2nd edition. CSIRO, Melbourne, Australia.
#'
#' @export
#' 
#' @import igraph
#' @import methods
#' @import MASS
#' @importFrom grDevices dev.off
#' @importFrom grDevices cairo_pdf
#' @importFrom graphics par
#' 
#' @examples
#' \donttest{
#' ## Examples using the package build-in data concrete, dental, human, analytical.
#' 
#' ## A fractional factorial design for investigating asphalt concrete production
#' hasselayout(datadesign=concrete, larger.fontlabelmultiplier=1.6, 
#'             smaller.fontlabelmultiplier=1.3, table.out=TRUE)
#' 
#' ## A crossover design for a dental study
#' hasselayout(datadesign=dental, randomfacsid = c(0,1,0,0,0), 
#'             larger.fontlabelmultiplier = 1.6)
#' 
#' ## A block design for an experiment assessing human-computer interaction
#' hasselayout(datadesign=human, randomfacsid = c(1,1,0,0,0,0,1), 
#'             larger.fontlabelmultiplier=1.4)
#' 
#' ## A cross-nested design for an analytical method investigation
#' hasselayout(datadesign=analytical, randomfacsid = c(0,0,1,1,1,0,0,0), 
#'             showpartialLS=FALSE, check.confound.df=FALSE, 
#'             larger.fontlabelmultiplier=1, 
#'             smaller.fontlabelmultiplier=1.6)
#' 
#' 
#' ## Examples using data from the dae package (conditionally loaded)
#' 
#' if (requireNamespace("dae", quietly = TRUE)) {
#'   
#'   ## Data for a balanced incomplete block experiment, Joshi (1987)
#'   
#'   data(BIBDWheat.dat, package = "dae")
#'   # remove the response from the dataset
#'   BIBDWheat <- BIBDWheat.dat[, -4]
#'   hasselayout(datadesign=BIBDWheat, example = "BIBDWheat")
#'   
#'   
#'   ## Data for an un-replicated 2^4 factorial experiment to investigate a chemical process
#'   ## from Table 10.6 of Box, Hunter and Hunter (1978)
#'   
#'   data(Fac4Proc.dat, package = "dae")
#'   # remove the response from the dataset
#'   Fac4Proc <- Fac4Proc.dat[, -6]
#'   hasselayout(datadesign=Fac4Proc, example = "Fac4Proc", showpartialLS=FALSE, 
#'               smaller.fontlabelmultiplier=2)
#'   
#'  
#'  ## Data for an experiment with rows and columns from p.144 of 
#'  ## Williams, Matheson and Harwood (2002)
#'
#'  data(Casuarina.dat, package = "dae")
#'  # remove the response from the dataset
#'  Casuarina <- Casuarina.dat[, -7]
#'  # create unique factor level labels
#'  Casuarina$Row <- paste(Casuarina$Reps, Casuarina$Rows)
#'  Casuarina$Col <- paste(Casuarina$Reps, Casuarina$Columns)
#'  Casuarina <- Casuarina[, -c(2,3)]
#'  hasselayout(datadesign=Casuarina, randomfacsid=c(1,0,1,0,0,0), 
#'              example="Casuarina", check.confound.df=FALSE, 
#'              showpartialLS=FALSE)
#'   
#' } else {
#'   message("Examples using data from the 'dae' package 
#'            require 'dae' to be installed.")
#' }
#' }
#' 
#' 
hasselayout <- function(datadesign, 
                        randomfacsid = NULL,
                        showLS = TRUE,
                        showpartialLS = TRUE, 
                        showdfLS = TRUE,
                        check.confound.df = TRUE,
                        maxlevels.df = TRUE,
                        table.out = FALSE,
                        pdf = FALSE,
                        example = "example",
                        outdir = NULL,
                        hasse.font = "sans",
                        produceBWPlot = FALSE,
                        structural.colour = "grey",
                        structural.width = 2,
                        partial.colour = "orange",
                        partial.width = 1.5,
                        objects.colour = "mediumblue",
                        df.colour = "red",
                        larger.fontlabelmultiplier = 1,   
                        middle.fontlabelmultiplier = 1,   
                        smaller.fontlabelmultiplier = 1) {
  
  ls_results <- .build_layout_structure(datadesign = datadesign, randomfacsid = randomfacsid)
  
  logical_args <- list(
    showLS = showLS,
    showpartialLS = showpartialLS,
    showdfLS = showdfLS,
    check.confound.df = check.confound.df,
    maxlevels.df = maxlevels.df,
    table.out = table.out,
    pdf = pdf,
    produceBWPlot = produceBWPlot
  )
  
  for (arg_name in names(logical_args)) {
    arg_value <- logical_args[[arg_name]]
    if (!is.logical(arg_value) || length(arg_value) != 1) {
      stop(sprintf("Argument '%s' must be a single logical value (TRUE or FALSE).", arg_name), call. = FALSE)
    }
  }
  
  if (!is.character(example) || length(example) != 1) {
    stop("Argument 'example' must be a single character string.", call. = FALSE)
  }
  
  if (!is.null(outdir) && (!dir.exists(outdir))) {
    stop("Argument 'outdir' must be NULL or an existing directory path.", call. = FALSE)
  }
  
  if (!is.character(hasse.font) || length(hasse.font) != 1) {
    stop("Argument 'hasse.font' must be a single character string naming a font family.", call. = FALSE)
  }
  if (!(hasse.font %in% c("sans", "serif", "mono"))) {
    warning("hasse.font is safe to be used for 'sans', 'serif', and 'mono'. \nYour selected font is not in that list, which may lead to potential errors.", call.=FALSE)
  }
  
  colour_args <- list(
    structural.colour = structural.colour,
    partial.colour = partial.colour,
    objects.colour = objects.colour,
    df.colour = df.colour
  )
  for (arg_name in names(colour_args)) {
    if (!is.character(colour_args[[arg_name]]) || length(colour_args[[arg_name]]) != 1) {
      stop(sprintf("Argument '%s' must be a single character colour name.", arg_name), call. = FALSE)
    }
  }
  
  numeric_args <- list(
    structural.width = structural.width,
    partial.width = partial.width,
    larger.fontlabelmultiplier = larger.fontlabelmultiplier,
    middle.fontlabelmultiplier = middle.fontlabelmultiplier,
    smaller.fontlabelmultiplier = smaller.fontlabelmultiplier
  )
  for (arg_name in names(numeric_args)) {
    arg_value <- numeric_args[[arg_name]]
    if (!is.numeric(arg_value) || length(arg_value) != 1 || is.na(arg_value) || arg_value <= 0) {
      stop(sprintf("Argument '%s' must be a single positive numeric value.", arg_name), call. = FALSE)
    }
  }
  
  if (showpartialLS==TRUE || showdfLS==TRUE) showLS <- TRUE
  
  if (is.null(outdir)) {
    data.folder.location <- tempdir()
  } else {
    data.folder.location <- outdir
  }
  
  if (pdf == TRUE) {
    edgewidth <- structural.width
    dottedline <- 2
    cairo_pdf(filename = file.path(data.folder.location, paste0(example, "_output.pdf")), width = 11, height = 8)
  } else {
    edgewidth <- structural.width
    dottedline <- 2
  }
  
  Colourblue <- "mediumblue"
  Colourred <- "red"
  Colourpurple <- "purple"
  Colourorange <- "orange"
  
  if (produceBWPlot == TRUE) {	
    Colourblue <- "black"
    Colourred <- "black"
    Colourpurple <- "black"
    Colourorange <- "black"
    structural.colour <- "black"
    partial.colour <- "black"
    objects.colour="black"
    df.colour="black"
  }
  
  
  old_opts <- options(show.error.messages = FALSE)
  on.exit(options(old_opts), add = TRUE)
  
  noall <- length(colnames(ls_results$outputlistip1$designi))

  if (showLS == TRUE) {
    
    names(ls_results$finaleffects) <- ls_results$finaleffectsnames
    
    if (table.out == TRUE) {
      message("\nThe following table shows the relationships between the factors and generalised factors in the Layout Structure\n") 
      message(paste(capture.output(print(ls_results$finalstructure)), collapse = "\n"))
    }
    
    adjm <- matrix(NA, nrow=nrow(ls_results$finalstructure), ncol=ncol(ls_results$finalstructure),
                   dimnames=dimnames(ls_results$finalstructure))
    for (i in 1:nrow(ls_results$finalstructure)) {
      for (j in 1:ncol(ls_results$finalstructure)) {
        if (ls_results$finalstructure[i,j]=="1") adjm[i,j] <- 1
        if (ls_results$finalstructure[i,j]=="0") adjm[i,j] <- 0
        if (ls_results$finalstructure[i,j]=="(0)") adjm[i,j] <- 0
        if (ls_results$finalstructure[i,j]==" ") adjm[i,j] <- 0
      }
    }
    
    adjm.adjust <- adjm
    for (j in 1:ncol(adjm)) {
      for (i in 1:nrow(adjm)) {
        if (adjm.adjust[i,j]==1) {
          for (k in 1:nrow(adjm)) {
            adjm.adjust[k,j] <- max(adjm.adjust[k,j]-adjm[k,i],0)
          }
        }
      }
    }
    
    adjm.adjust <- rbind(aaadum1=c(0), Mean=adjm.adjust[1,], zzzdum2=c(0), adjm.adjust[-1,])
    adjm.adjust <- cbind(aaadum1=c(0), Mean=adjm.adjust[,1], zzzdum2=c(0), adjm.adjust[,-1])
    
    adjm.reverse <- adjm.adjust[nrow(adjm.adjust):1, ncol(adjm.adjust):1]
    
    g1 <- graph_from_adjacency_matrix(adjm.reverse, mode="max")
    
    g <- simplify(g1)
    V(g)$label <- V(g)$name
    
    dscoords <- dscoords.fun("LS", ls_results$finaleffects, ls_results$finalstructure, larger.fontlabelmultiplier, smaller.fontlabelmultiplier, middle.fontlabelmultiplier)  
    g$layout <- dscoords$coords
    
    font_used <- hasse.font  
    vertex.label.font <- rep(2,length(colnames(adjm.reverse)))
    vertex.label.color.objects <- c(rep(objects.colour,length(colnames(adjm.reverse))-3), "transparent", objects.colour, "transparent")
    vertex.label.color.black <- c(rep("black", length(colnames(adjm.reverse))-3), "transparent", "black", "transparent")
    vertex.label.color.df <- c(rep(df.colour, length(colnames(adjm.reverse))-3), "transparent", df.colour, "transparent")
    
    finaleffectrandom.reverse <- c(ls_results$finalrandomeffects[length(ls_results$finalrandomeffects):1], 0, 0)
    
    adjm.reverse.blank <- adjm.reverse
    for (m in 1:length(colnames(adjm.reverse.blank))) {
      if (finaleffectrandom.reverse[m]==1) {
        colnames(adjm.reverse.blank)[m] <- paste("", paste(rep("_",nchar(colnames(adjm.reverse.blank)[m])), collapse=""))
      } else {
        colnames(adjm.reverse.blank)[m] <- ""
      }
    }
    
    g2 <- graph_from_adjacency_matrix(adjm.reverse.blank, mode="max")
    
    g2a <- simplify(g2)
    V(g2a)$label <- V(g2a)$name
    g2a$layout <- dscoords$coords
    vcount(g2a)
    g2a.edges <- get.edges(g2a, 1:ecount(g2a))[ ,1]-1   
    node.dumg <- c(vcount(g2a)-3, vcount(g2a)-1)   
    edge.color<-rep(structural.colour, length(g2a.edges))
    edge.color[g2a.edges %in% node.dumg]<-"transparent"
    
    if (all(par("mfrow") == c(1, 1))) {
      
      default_mar <- c(5.1, 4.1, 4.1, 2.1)
      current_mar <- par("mar")
      
      if (isTRUE(all.equal(current_mar, default_mar))) {
        
        old_mar <- current_mar
        on.exit(par(mar = old_mar), add = TRUE)
        
        par(mar = c(
          (2 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.8,
          (5 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.4,
          0.2,
          (5 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.4
        ))
      }
    }
    
    tryCatch({
      plot(g2a, asp=FALSE, add=F, vertex.label.color=vertex.label.color.black, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=0.6, vertex.size=5, vertex.color="transparent", vertex.shape="circle", vertex.frame.color="white", edge.color=edge.color, edge.width = edgewidth, vertex.label.family=font_used)
    }, error = function(e) {
      message("The font selected in hasse.font is not available in the system's fonts and rendering failed. See the Details section for more information on fonts. The hasse.font is set to 'sans' instead.")
      font_used <<- "sans"
      plot(g2a, asp=FALSE, add=F, vertex.label.color=vertex.label.color.black, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=0.6, vertex.size=5, vertex.color="transparent", vertex.shape="circle", vertex.frame.color="white", edge.color=edge.color, edge.width = edgewidth, vertex.label.family=font_used)
    })
    
    if (showpartialLS==TRUE) {
      adjm3 <- matrix(0,nrow=nrow(adjm),ncol=ncol(adjm),dimnames=dimnames(adjm))
      for (i in 1:nrow(ls_results$finalstructure)) {
        for (j in 1:ncol(ls_results$finalstructure)) {
          if (ls_results$finalstructure[i,j]=="(0)" && ls_results$finalstructure[j,i]=="(0)") adjm3[i,j] <- 1
        }
      }
      
      adjm3.adjust <- adjm3
      for (j in 1:ncol(adjm3)) {
        for (i in 1:nrow(adjm3)) {
          if (adjm3.adjust[i,j]==1) {
            for (k in 1:nrow(adjm3)) {
              adjm3.adjust[k,j] <- max(adjm3.adjust[k,j]-adjm3[k,i],0)
            }
          }
        }
      }
      
      adjm3.adjust <-rbind(aaadum1=c(0), Mean=adjm3.adjust[1, ], zzzdum2=c(0), adjm3.adjust[-1, ])
      adjm3.adjust <-cbind(aaadum1=c(0), Mean=adjm3.adjust[ ,1], zzzdum2=c(0), adjm3.adjust[ ,-1])
      adjm3.reverse <- adjm3.adjust[nrow(adjm3.adjust):1,ncol(adjm3.adjust):1]
      
      g3 <- graph_from_adjacency_matrix(adjm3.reverse,mode="max")                   
      g3 <- simplify(g3)
      V(g3)$label <- V(g3)$name
      
      g3$layout <- dscoords$coords
      
      plot(g3, asp=FALSE, add=TRUE, vertex.label.color="transparent", vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent",  edge.label.color=Colourred, edge.label.font=2, edge.color=partial.colour,edge.lty=dottedline, edge.width = partial.width, vertex.label.family=font_used)
    }
    
    plot(g, asp=FALSE, add=T, vertex.label.color=vertex.label.color.objects, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent", vertex.shape="circle", edge.lty=0, edge.width = edgewidth, vertex.label.family=font_used)
  }
  
  if (showdfLS==TRUE) {
    LS.output <- dfs.fun("LS", noall, ls_results$finaleffects, ls_results$finalstructure, adjm, ls_results$outputlistip1, ls_results$nfactors, maxlevels.df, check.confound.df, ls_results$datadesign)
    g4 <- g
    
    V(g4)$label <- paste(sep="", "[",LS.output$xdfs.reverse[ ,3],LS.output$maxlevelsf.reverse[ ],",",LS.output$xdfs.reverse[ ,4],"]")
    plot(g4, asp=FALSE, add=T, vertex.label.color=vertex.label.color.df, vertex.label.cex=dscoords$textlabel.size.df, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=(1*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*1,vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent", vertex.shape="circle", edge.lty=0, edge.width = edgewidth, vertex.label.family=font_used)
    LS.output$xdfs
  }
  
    leff <- length(ls_results$finaleffectsnames)
    mat.vec2 <- rep("NULL", leff)
    mat.vec2[1] <- ls_results$finaleffectsnames[1]
    mat.finaleffectsnames <- cbind(ls_results$finaleffectsnames, mat.vec2)
    rownames(mat.finaleffectsnames) <- c(1:leff)
    colnames(mat.finaleffectsnames) <- c("All Structural Objects", "Randomisation Objects")
  
  output <- list(
    str_objects   = ls_results[["finaleffectsnames"]],
    rand_template = c("Mean", rep("NULL", leff - 1)),
    str_rand_map  = mat.finaleffectsnames,
    notes = paste(
      "Use rand_template to define rand.objects in hasserls().",
      "Objects not present in the restricted layout must remain 'NULL'."
    )
  )
  
  if (pdf==TRUE) hidedevoff <- dev.off()
  
  
  class(output) <- "ls"
  invisible(output)
  
  
}



