/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5G_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5HLprivate.h" 
#include "H5Opkg.h"      

static void *H5O__stab_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__stab_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                               const void *_mesg);
static void  *H5O__stab_copy(const void *_mesg, void *_dest);
static size_t H5O__stab_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__stab_free(void *_mesg);
static herr_t H5O__stab_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static void  *H5O__stab_copy_file(H5F_t *file_src, void *native_src, H5F_t *file_dst, bool *recompute_size,
                                  unsigned *mesg_flags, H5O_copy_t *cpy_info, void *_udata);
static herr_t H5O__stab_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                                       void *mesg_dst, unsigned *mesg_flags, H5O_copy_t *cpy_info);
static herr_t H5O__stab_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_STAB[1] = {{
    H5O_STAB_ID,              
    "stab",                   
    sizeof(H5O_stab_t),       
    0,                        
    H5O__stab_decode,         
    H5O__stab_encode,         
    H5O__stab_copy,           
    H5O__stab_size,           
    NULL,                     
    H5O__stab_free,           
    H5O__stab_delete,         
    NULL,                     
    NULL,                     
    NULL,                     
    NULL,                     
    H5O__stab_copy_file,      
    H5O__stab_post_copy_file, 
    NULL,                     
    NULL,                     
    H5O__stab_debug           
}};

H5FL_DEFINE_STATIC(H5O_stab_t);

static void *
H5O__stab_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                 unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_stab_t    *stab      = NULL;
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (stab = H5FL_CALLOC(H5O_stab_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(f, &p, &(stab->btree_addr));

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(f, &p, &(stab->heap_addr));

    ret_value = stab;

done:
    if (!ret_value && stab)
        H5FL_FREE(H5O_stab_t, stab);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__stab_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                 const void *_mesg)
{
    const H5O_stab_t *stab = (const H5O_stab_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(stab);

    
    H5F_addr_encode(f, &p, stab->btree_addr);
    H5F_addr_encode(f, &p, stab->heap_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__stab_copy(const void *_mesg, void *_dest)
{
    const H5O_stab_t *stab      = (const H5O_stab_t *)_mesg;
    H5O_stab_t       *dest      = (H5O_stab_t *)_dest;
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(stab);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_stab_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *stab;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__stab_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = (size_t)(2 * H5F_SIZEOF_ADDR(f));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__stab_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_stab_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__stab_delete(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, void *mesg)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(mesg);

    
    if (H5G__stab_delete(f, (const H5O_stab_t *)mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free symbol table");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__stab_copy_file(H5F_t *file_src, void *native_src, H5F_t *file_dst, bool H5_ATTR_UNUSED *recompute_size,
                    unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t H5_ATTR_UNUSED *cpy_info, void *_udata)
{
    H5O_stab_t         *stab_src = (H5O_stab_t *)native_src;
    H5O_stab_t         *stab_dst = NULL;
    H5G_copy_file_ud_t *udata    = (H5G_copy_file_ud_t *)_udata;
    size_t              size_hint;        
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(stab_src);
    assert(file_dst);

    
    if (NULL == (stab_dst = H5FL_MALLOC(H5O_stab_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5HL_get_size(file_src, stab_src->heap_addr, &size_hint) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGETSIZE, NULL, "can't query local heap size");

    
    H5_BEGIN_TAG(H5AC__COPIED_TAG)

    
    if (H5G__stab_create_components(file_dst, stab_dst, size_hint) < 0)
        HGOTO_ERROR_TAG(H5E_SYM, H5E_CANTINIT, NULL, "can't create symbol table components");

    
    H5_END_TAG

    
    udata->cache_type            = H5G_CACHED_STAB;
    udata->cache.stab.btree_addr = stab_dst->btree_addr;
    udata->cache.stab.heap_addr  = stab_dst->heap_addr;

    
    ret_value = stab_dst;

done:
    if (!ret_value)
        if (stab_dst)
            stab_dst = H5FL_FREE(H5O_stab_t, stab_dst);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__stab_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc, void *mesg_dst,
                         unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t *cpy_info)
{
    const H5O_stab_t *stab_src = (const H5O_stab_t *)mesg_src;
    H5O_stab_t       *stab_dst = (H5O_stab_t *)mesg_dst;
    H5G_bt_it_cpy_t   udata;               
    H5HL_t           *heap      = NULL;    
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(stab_src);
    assert(H5_addr_defined(dst_oloc->addr));
    assert(dst_oloc->file);
    assert(stab_dst);
    assert(cpy_info);

    
    if (cpy_info->max_depth >= 0 && cpy_info->curr_depth >= cpy_info->max_depth)
        HGOTO_DONE(SUCCEED);

    
    if (NULL == (heap = H5HL_protect(src_oloc->file, stab_src->heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_PROTECT, FAIL, "unable to protect local heap");

    
    udata.src_oloc       = src_oloc;
    udata.src_heap       = heap;
    udata.src_block_size = H5HL_heap_get_size(heap);
    udata.dst_file       = dst_oloc->file;
    udata.dst_stab       = stab_dst;
    udata.cpy_info       = cpy_info;

    
    if ((H5B_iterate(src_oloc->file, H5B_SNODE, stab_src->btree_addr, H5G__node_copy, &udata)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "iteration operator failed");

done:
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect local heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__stab_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_stab_t *stab = (const H5O_stab_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(stab);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "B-tree address:", stab->btree_addr);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "Name heap address:", stab->heap_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
