/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND      
#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDpkg.h"     
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

static herr_t H5FD__free_cls(void *cls, void **request);
static herr_t H5FD__query(const H5FD_t *f, unsigned long *flags );

bool H5_PKG_INIT_VAR = false;

htri_t H5FD_ignore_disabled_file_locks_p = FAIL;

static unsigned long H5FD_file_serial_no_g;

static const H5I_class_t H5I_VFL_CLS[1] = {{
    H5I_VFL,                   
    0,                         
    0,                         
    (H5I_free_t)H5FD__free_cls 
}};

herr_t
H5FD_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__init_package(void)
{
    char  *lock_env_var = NULL;    
    herr_t ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_register_type(H5I_VFL_CLS) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "unable to initialize interface");

    
    H5FD_file_serial_no_g = 0;

    
    lock_env_var = getenv(HDF5_USE_FILE_LOCKING);
    if (lock_env_var && !strcmp(lock_env_var, "BEST_EFFORT"))
        H5FD_ignore_disabled_file_locks_p = true; 
    else if (lock_env_var && (!strcmp(lock_env_var, "TRUE") || !strcmp(lock_env_var, "1")))
        H5FD_ignore_disabled_file_locks_p = false; 
    else
        H5FD_ignore_disabled_file_locks_p = FAIL; 

    
    if (H5FD__core_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register core VFD");
#ifdef H5_HAVE_DIRECT
    if (H5FD__direct_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register direct VFD");
#endif
    if (H5FD__family_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register family VFD");
#ifdef H5_HAVE_LIBHDFS
    if (H5FD__hdfs_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register hdfs VFD");
#endif
#ifdef H5_HAVE_IOC_VFD
    if (H5FD__ioc_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register ioc VFD");
#endif
    if (H5FD__log_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register log VFD");
#ifdef H5_HAVE_MIRROR_VFD
    if (H5FD__mirror_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register mirror VFD");
#endif
#ifdef H5_HAVE_PARALLEL
    if (H5FD__mpio_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register mpio VFD");
#endif
    if (H5FD__multi_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register multi VFD");
    if (H5FD__onion_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register onion VFD");
#ifdef H5_HAVE_ROS3_VFD
    if (H5FD__ros3_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register ros3 VFD");
#endif
    if (H5FD__sec2_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register sec2 VFD");
    if (H5FD__splitter_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register splitter VFD");
    if (H5FD__stdio_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register stdio VFD");
#ifdef H5_HAVE_SUBFILING_VFD
    if (H5FD__subfiling_register() < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register subfiling VFD");
#endif

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5FD_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        if (H5I_nmembers(H5I_VFL) > 0) {
            (void)H5I_clear_type(H5I_VFL, false, false);

            
            H5FD__core_unregister();
#ifdef H5_HAVE_DIRECT
            H5FD__direct_unregister();
#endif
            H5FD__family_unregister();
#ifdef H5_HAVE_LIBHDFS
            H5FD__hdfs_unregister();
#endif
#ifdef H5_HAVE_IOC_VFD
            H5FD__ioc_unregister();
#endif
            H5FD__log_unregister();
#ifdef H5_HAVE_MIRROR_VFD
            H5FD__mirror_unregister();
#endif
#ifdef H5_HAVE_PARALLEL
            H5FD__mpio_unregister();
#endif
            H5FD__multi_unregister();
            H5FD__onion_unregister();
#ifdef H5_HAVE_ROS3_VFD
            H5FD__ros3_unregister();
#endif
            H5FD__sec2_unregister();
            H5FD__splitter_unregister();
            H5FD__stdio_unregister();
#ifdef H5_HAVE_SUBFILING_VFD
            H5FD__subfiling_unregister();
#endif

            n++; 
        }        
        else {
            
            n += (H5I_dec_type_ref(H5I_VFL) > 0);

            
            if (0 == n)
                H5_PKG_INIT_VAR = false;
        } 
    }     

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5FD__free_cls(void *_cls, void H5_ATTR_UNUSED **request)
{
    H5FD_class_t *cls = (H5FD_class_t *)_cls;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (cls->terminate) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = cls->terminate();
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTCLOSEOBJ, FAIL, "virtual file driver '%s' did not terminate cleanly",
                        cls->name);
    }

    H5MM_xfree(cls);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5FDregister(const H5FD_class_t *cls)
{
    H5FD_mem_t type;
    hid_t      ret_value = H5I_INVALID_HID;

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!cls)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID, "null class pointer is disallowed");
    if (cls->version != H5FD_CLASS_VERSION)
        HGOTO_ERROR(H5E_ARGS, H5E_VERSION, H5I_INVALID_HID, "wrong file driver version #");
    if (!cls->open || !cls->close)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID,
                    "'open' and/or 'close' methods are not defined");
    if (!cls->get_eoa || !cls->set_eoa)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID,
                    "'get_eoa' and/or 'set_eoa' methods are not defined");
    if (!cls->get_eof)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID, "'get_eof' method is not defined");
    if (!cls->read || !cls->write)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID,
                    "'read' and/or 'write' method is not defined");
    for (type = H5FD_MEM_DEFAULT; type < H5FD_MEM_NTYPES; type++)
        if (cls->fl_map[type] < H5FD_MEM_NOLIST || cls->fl_map[type] >= H5FD_MEM_NTYPES)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid free-list mapping");

    
    if ((ret_value = H5FD_register(cls, sizeof(H5FD_class_t), true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file driver ID");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5FD_register(const void *_cls, size_t size, bool app_ref)
{
    const H5FD_class_t *cls   = (const H5FD_class_t *)_cls;
    H5FD_class_t       *saved = NULL;
    H5FD_mem_t          type;
    hid_t               ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    assert(cls);
    assert(cls->open && cls->close);
    assert(cls->get_eoa && cls->set_eoa);
    assert(cls->get_eof);
    assert(cls->read && cls->write);
    for (type = H5FD_MEM_DEFAULT; type < H5FD_MEM_NTYPES; type++)
        assert(cls->fl_map[type] >= H5FD_MEM_NOLIST && cls->fl_map[type] < H5FD_MEM_NTYPES);

    
    if (NULL == (saved = (H5FD_class_t *)H5MM_malloc(size)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, H5I_INVALID_HID,
                    "memory allocation failed for file driver class struct");
    H5MM_memcpy(saved, cls, size);

    
    if ((ret_value = H5I_register(H5I_VFL, saved, app_ref)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file driver ID");

done:
    if (H5I_INVALID_HID == ret_value)
        if (saved)
            saved = (H5FD_class_t *)H5MM_xfree(saved);

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FDis_driver_registered_by_name(const char *driver_name)
{
    htri_t ret_value = false; 

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5FD_is_driver_registered_by_name(driver_name, NULL)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't check if VFD is registered");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5FDis_driver_registered_by_value(H5FD_class_value_t driver_value)
{
    htri_t ret_value = false;

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5FD_is_driver_registered_by_value(driver_value, NULL)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't check if VFD is registered");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDunregister(hid_t driver_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == H5I_object_verify(driver_id, H5I_VFL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file driver");

    
    if (H5I_dec_app_ref(driver_id) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "unable to unregister file driver");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5FD_class_t *
H5FD_get_class(hid_t id)
{
    H5FD_class_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (H5I_VFL == H5I_get_type(id))
        ret_value = (H5FD_class_t *)H5I_object(id);
    else {
        H5P_genplist_t *plist; 

        
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(id)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, NULL, "can't find object for ID");

        if (true == H5P_isa_class(id, H5P_FILE_ACCESS)) {
            H5FD_driver_prop_t driver_prop; 

            if (H5P_peek(plist, H5F_ACS_FILE_DRV_NAME, &driver_prop) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, NULL, "can't get driver ID & info");
            ret_value = H5FD_get_class(driver_prop.driver_id);
        } 
        else
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a driver id or file access property list");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5FD_sb_size(H5FD_t *file)
{
    hsize_t ret_value = 0;

    FUNC_ENTER_NOAPI(0)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->sb_size) {
        
        H5_BEFORE_USER_CB_NOERR(0)
            {
                ret_value = (file->cls->sb_size)(file);
            }
        H5_AFTER_USER_CB_NOERR(0)
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5FD_sb_encode(H5FD_t *file, char *name , uint8_t *buf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->sb_encode) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->sb_encode)(file, name , buf );
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver sb_encode request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sb_decode(H5FD_t *file, const char *name, const uint8_t *buf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->sb_decode) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->sb_decode)(file, name, buf);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver sb_decode request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_sb_load(H5FD_t *file, const char *name, const uint8_t *buf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (!strncmp(name, "NCSAfami", (size_t)8) && strcmp(file->cls->name, "family") != 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "family driver should be used");
    if (!strncmp(name, "NCSAmult", (size_t)8) && strcmp(file->cls->name, "multi") != 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "multi driver should be used");

    
    if (H5FD__sb_decode(file, name, buf) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDECODE, FAIL, "unable to decode driver information");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FD_fapl_get(H5FD_t *file)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->fapl_get) {
        
        H5_BEFORE_USER_CB_NOERR(NULL)
            {
                ret_value = (file->cls->fapl_get)(file);
            }
        H5_AFTER_USER_CB_NOERR(NULL)
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_free_driver_info(hid_t driver_id, const void *driver_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (driver_id > 0 && driver_info) {
        H5FD_class_t *driver;

        
        if (NULL == (driver = (H5FD_class_t *)H5I_object(driver_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a driver ID");

        
        if (driver->fapl_free) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    
                    
                    ret_value = (driver->fapl_free)((void *)(uintptr_t)driver_info);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "driver free request failed");
        }
        else
            driver_info = H5MM_xfree_const(driver_info);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FD_t *
H5FDopen(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    H5FD_t *ret_value = NULL;

    FUNC_ENTER_API(NULL)

    
    if (H5P_DEFAULT == fapl_id)
        fapl_id = H5P_FILE_ACCESS_DEFAULT;
    else if (true != H5P_isa_class(fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");

    
    if (H5FD_open(false, &ret_value, name, flags, fapl_id, maxaddr) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, NULL, "unable to open file");

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5FD_open(bool try, H5FD_t **_file, const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    H5FD_t                *file = NULL;         
    H5FD_class_t          *driver;              
    H5FD_driver_prop_t     driver_prop;         
    H5P_genplist_t        *plist;               
    unsigned long          driver_flags = 0;    
    H5FD_file_image_info_t file_image_info;     
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    *_file = NULL;

    
    if (0 == maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "zero format address range");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    if (H5P_peek(plist, H5F_ACS_FILE_DRV_NAME, &driver_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get driver ID & info");

    
    if (NULL == (driver = (H5FD_class_t *)H5I_object(driver_prop.driver_id)))
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "invalid driver ID in file access property list");
    if (NULL == driver->open)
        HGOTO_ERROR(H5E_VFL, H5E_UNSUPPORTED, FAIL, "file driver has no `open' method");

    
    if (H5FD_driver_query(driver, &driver_flags) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "can't query VFD flags");

    
    if (H5P_peek(plist, H5F_ACS_FILE_IMAGE_INFO_NAME, &file_image_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get file image info");

    
    assert((file_image_info.buffer && file_image_info.size > 0) ||
           (!file_image_info.buffer && file_image_info.size == 0));
    if (file_image_info.buffer && !(driver_flags & H5FD_FEAT_ALLOW_FILE_IMAGE))
        HGOTO_ERROR(H5E_VFL, H5E_UNSUPPORTED, FAIL, "file image set, but not supported.");

    if (HADDR_UNDEF == maxaddr)
        maxaddr = driver->maxaddr;

    

    
    if (try) {
        H5E_PAUSE_ERRORS
            {
                 H5_BEFORE_USER_CB(FAIL)
                    {
                        file = (driver->open)(name, flags, fapl_id, maxaddr);
                    }
                H5_AFTER_USER_CB(FAIL)
            }
        H5E_RESUME_ERRORS

        
        if (NULL == file)
            HGOTO_DONE(SUCCEED);
    }
    else
    {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                file = (driver->open)(name, flags, fapl_id, maxaddr);
            }
        H5_AFTER_USER_CB(FAIL)
        if (NULL == file)
            HGOTO_ERROR(H5E_VFL, H5E_CANTOPENFILE, FAIL, "can't open file");
    }

    
    file->access_flags = flags;

    
    file->driver_id = driver_prop.driver_id;
    if (H5I_inc_ref(file->driver_id, false) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINC, FAIL, "unable to increment ref count on VFL driver");
    file->cls     = driver;
    file->maxaddr = maxaddr;
    if (H5P_get(plist, H5F_ACS_ALIGN_THRHD_NAME, &file->threshold) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get alignment threshold");
    if (H5P_get(plist, H5F_ACS_ALIGN_NAME, &file->alignment) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get alignment");

    
    if (H5FD__query(file, &file->feature_flags) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "unable to query file driver");

    
    if (++H5FD_file_serial_no_g == 0)
        
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "unable to get file serial number");
    file->fileno = H5FD_file_serial_no_g;

    
    
    file->base_addr = 0;

    
    *_file = file;

done :
    
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDclose(H5FD_t *file)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    
    if (H5FD_close(file) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTCLOSEFILE, FAIL, "unable to close file");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_close(H5FD_t *file)
{
    const H5FD_class_t *driver;
    herr_t              ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    driver = file->cls;
    if (H5I_dec_ref(file->driver_id) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDEC, FAIL, "can't close driver ID");

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            assert(driver->close);
            ret_value = (driver->close)(file);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTCLOSEFILE, FAIL, "close failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5FDcmp(const H5FD_t *f1, const H5FD_t *f2)
{
    int ret_value = -1;

    FUNC_ENTER_API(-1) 

    
    ret_value = H5FD_cmp(f1, f2);

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5FD_cmp(const H5FD_t *f1, const H5FD_t *f2)
{
    int ret_value = -1; 

    FUNC_ENTER_NOAPI(-1) 

    if ((!f1 || !f1->cls) && (!f2 || !f2->cls))
        HGOTO_DONE(0);
    if (!f1 || !f1->cls)
        HGOTO_DONE(-1);
    if (!f2 || !f2->cls)
        HGOTO_DONE(1);
    if (f1->cls < f2->cls)
        HGOTO_DONE(-1);
    if (f1->cls > f2->cls)
        HGOTO_DONE(1);

    
    if (!f1->cls->cmp) {
        if (f1 < f2)
            HGOTO_DONE(-1);
        if (f1 > f2)
            HGOTO_DONE(1);
        HGOTO_DONE(0);
    }

    
    H5_BEFORE_USER_CB_NOCHECK
        {
            
            ret_value = (f1->cls->cmp)(f1, f2);
        }
    H5_AFTER_USER_CB_NOCHECK

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDquery(const H5FD_t *file, unsigned long *flags )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (!flags)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "flags parameter cannot be NULL");

    
    if (H5FD__query(file, flags) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "unable to query feature flags");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5FD__query(const H5FD_t *file, unsigned long *flags )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);
    assert(flags);

    
    if (file->cls->query) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->query)(file, flags);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "unable to query feature flags");
    }
    else
        *flags = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FDalloc(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, hsize_t size)
{
    haddr_t ret_value = HADDR_UNDEF;

    FUNC_ENTER_API(HADDR_UNDEF)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file class pointer cannot be NULL");
    if (type < H5FD_MEM_DEFAULT || type >= H5FD_MEM_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "invalid request type");
    if (size == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "zero-size request");
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, HADDR_UNDEF, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (HADDR_UNDEF == (ret_value = H5FD__alloc_real(file, type, size, NULL, NULL)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "unable to allocate file memory");

    
    ret_value += file->base_addr;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDfree(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, hsize_t size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (type < H5FD_MEM_DEFAULT || type >= H5FD_MEM_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid request type");
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD__free_real(file, type, addr - file->base_addr, size) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "file deallocation request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

haddr_t
H5FDget_eoa(H5FD_t *file, H5FD_mem_t type)
{
    haddr_t ret_value;

    FUNC_ENTER_API(HADDR_UNDEF)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file class pointer cannot be NULL");
    if (type < H5FD_MEM_DEFAULT || type >= H5FD_MEM_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "invalid file type");

    
    if (HADDR_UNDEF == (ret_value = H5FD_get_eoa(file, type)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "file get eoa request failed");

    
    ret_value += file->base_addr;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDset_eoa(H5FD_t *file, H5FD_mem_t type, haddr_t addr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (type < H5FD_MEM_DEFAULT || type >= H5FD_MEM_NTYPES)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid file type");
    if (!H5_addr_defined(addr) || addr > file->maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid end-of-address value");

    
    
    if (H5FD_set_eoa(file, type, addr - file->base_addr) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "file set eoa request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

haddr_t
H5FDget_eof(H5FD_t *file, H5FD_mem_t type)
{
    haddr_t ret_value;

    FUNC_ENTER_API(HADDR_UNDEF)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, HADDR_UNDEF, "file class pointer cannot be NULL");

    
    if (HADDR_UNDEF == (ret_value = H5FD_get_eof(file, type)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "file get eof request failed");

    
    ret_value += file->base_addr;

done:
    FUNC_LEAVE_API(ret_value)
} 

haddr_t
H5FD_get_maxaddr(const H5FD_t *file)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    
    assert(file);

    
    ret_value = file->maxaddr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_get_feature_flags(const H5FD_t *file, unsigned long *feature_flags)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);
    assert(feature_flags);

    
    *feature_flags = file->feature_flags;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FD_set_feature_flags(H5FD_t *file, unsigned long feature_flags)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);

    
    file->feature_flags = feature_flags;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FD_get_fs_type_map(const H5FD_t *file, H5FD_mem_t *type_map)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(type_map);

    
    if (file->cls->get_type_map) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->get_type_map)(file, type_map);
            }
        H5_AFTER_USER_CB(FAIL)
        
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "driver get type map failed");
    } 
    else
        
        H5MM_memcpy(type_map, file->cls->fl_map, sizeof(file->cls->fl_map));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDread(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size, void *buf )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (!buf)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "result buffer parameter can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD_read(file, type, addr - file->base_addr, size, buf) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file read request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDwrite(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size, const void *buf)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (!buf)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "result buffer parameter can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD_write(file, type, addr - file->base_addr, size, buf) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file write request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDread_vector(H5FD_t *file, hid_t dxpl_id, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                size_t sizes[], void *bufs[] )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!types) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "types parameter can't be NULL if count is positive");

    if ((!addrs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addrs parameter can't be NULL if count is positive");

    if ((!sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (types[0] == H5FD_MEM_NOLIST))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "count[0] can't be H5FD_MEM_NOLIST");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD_read_vector(file, count, types, addrs, sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file vector read request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDwrite_vector(H5FD_t *file, hid_t dxpl_id, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                 size_t sizes[], const void *bufs[] )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!types) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "types parameter can't be NULL if count is positive");

    if ((!addrs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addrs parameter can't be NULL if count is positive");

    if ((!sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (types[0] == H5FD_MEM_NOLIST))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "count[0] can't be H5FD_MEM_NOLIST");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD_write_vector(file, count, types, addrs, sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file vector write request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDread_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count, hid_t mem_space_ids[],
                   hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[], void *bufs[] )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    H5CX_set_dxpl(dxpl_id);

    
    
    if (H5FD_read_selection_id(SKIP_NO_CB, file, type, count, mem_space_ids, file_space_ids, offsets,
                               element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file selection read request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDwrite_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count, hid_t mem_space_ids[],
                    hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[], const void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    H5CX_set_dxpl(dxpl_id);

    
    

    if (H5FD_write_selection_id(SKIP_NO_CB, file, type, count, mem_space_ids, file_space_ids, offsets,
                                element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file selection write request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDread_vector_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                               hid_t mem_space_ids[], hid_t file_space_ids[], haddr_t offsets[],
                               size_t element_sizes[], void *bufs[] )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    
    if (H5FD_read_vector_from_selection(file, type, count, mem_space_ids, file_space_ids, offsets,
                                        element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file selection read request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDwrite_vector_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                hid_t mem_space_ids[], hid_t file_space_ids[], haddr_t offsets[],
                                size_t element_sizes[], const void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    
    if (H5FD_write_vector_from_selection(file, type, count, mem_space_ids, file_space_ids, offsets,
                                         element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file selection write request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDread_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count, hid_t mem_space_ids[],
                        hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[], void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    
    if (H5FD_read_from_selection(file, type, count, mem_space_ids, file_space_ids, offsets, element_sizes,
                                 bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file selection read request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDwrite_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count, hid_t mem_space_ids[],
                         hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                         const void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if ((!mem_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "mem_spaces parameter can't be NULL if count is positive");

    if ((!file_space_ids) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file_spaces parameter can't be NULL if count is positive");

    if ((!offsets) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offsets parameter can't be NULL if count is positive");

    if ((!element_sizes) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                    "element_sizes parameter can't be NULL if count is positive");

    if ((!bufs) && (count > 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs parameter can't be NULL if count is positive");

    if ((count > 0) && (element_sizes[0] == 0))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sizes[0] can't be 0");

    if ((count > 0) && (bufs[0] == NULL))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "bufs[0] can't be NULL");

    
    if (H5P_DEFAULT == dxpl_id) {
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    }
    else {
        if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");
    }

    
    
    if (H5FD_write_from_selection(file, type, count, mem_space_ids, file_space_ids, offsets, element_sizes,
                                  bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file selection write request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDflush(H5FD_t *file, hid_t dxpl_id, bool closing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (H5FD_flush(file, closing) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTFLUSH, FAIL, "file flush request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_flush(H5FD_t *file, bool closing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->flush) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->flush)(file, H5CX_get_dxpl(), closing);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver flush request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDtruncate(H5FD_t *file, hid_t dxpl_id, bool closing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data transfer property list");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (H5FD_truncate(file, closing) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTUPDATE, FAIL, "file flush request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_truncate(H5FD_t *file, bool closing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->truncate) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->truncate)(file, H5CX_get_dxpl(), closing);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTUPDATE, FAIL, "driver truncate request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDlock(H5FD_t *file, bool rw)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    
    if (H5FD_lock(file, rw) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTLOCKFILE, FAIL, "file lock request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_lock(H5FD_t *file, bool rw)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->lock) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->lock)(file, rw);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTLOCKFILE, FAIL, "driver lock request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDunlock(H5FD_t *file)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    
    if (H5FD_unlock(file) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTUNLOCKFILE, FAIL, "file unlock request failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_unlock(H5FD_t *file)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->unlock) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->unlock)(file);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTUNLOCKFILE, FAIL, "driver unlock request failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FDctl(H5FD_t *file, uint64_t op_code, uint64_t flags, const void *input, void **output)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");

    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");

    

    
    if (H5FD_ctl(file, op_code, flags, input, output) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_FCNTL, FAIL, "VFD ctl request failed");

done:

    FUNC_LEAVE_API(ret_value)

} 

herr_t
H5FD_ctl(H5FD_t *file, uint64_t op_code, uint64_t flags, const void *input, void **output)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);

    
    if (file->cls->ctl) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->ctl)(file, op_code, flags, input, output);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_FCNTL, FAIL, "VFD ctl request failed");
    }
    else if (flags & H5FD_CTL_FAIL_IF_UNKNOWN_FLAG)
        HGOTO_ERROR(H5E_VFL, H5E_FCNTL, FAIL,
                    "VFD ctl request failed (no ctl callback and fail if unknown flag is set)");

done:

    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_get_fileno(const H5FD_t *file, unsigned long *filenum)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);
    assert(filenum);

    
    *filenum = file->fileno;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FDget_vfd_handle(H5FD_t *file, hid_t fapl_id, void **file_handle )
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (!file)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file pointer cannot be NULL");
    if (!file->cls)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file class pointer cannot be NULL");
    if (false == H5P_isa_class(fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "fapl_id parameter is not a file access property list");
    if (!file_handle)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file handle parameter cannot be NULL");

    
    if (H5FD_get_vfd_handle(file, fapl_id, file_handle) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get file handle for file driver");

done:
    if (FAIL == ret_value) {
        if (file_handle)
            *file_handle = NULL;
    }

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FD_get_vfd_handle(H5FD_t *file, hid_t fapl_id, void **file_handle)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(file_handle);

    
    if (NULL == file->cls->get_handle)
        HGOTO_ERROR(H5E_VFL, H5E_UNSUPPORTED, FAIL, "file driver has no `get_vfd_handle' method");

    
    H5_BEFORE_USER_CB(FAIL)
        {
            ret_value = (file->cls->get_handle)(file, fapl_id, file_handle);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get file handle for file driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_set_base_addr(H5FD_t *file, haddr_t base_addr)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);
    assert(H5_addr_defined(base_addr));

    
    file->base_addr = base_addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

haddr_t
H5FD_get_base_addr(const H5FD_t *file)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);

    
    FUNC_LEAVE_NOAPI(file->base_addr)
} 

herr_t
H5FD_set_paged_aggr(H5FD_t *file, bool paged)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(file);

    
    file->paged_aggr = paged;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FDdriver_query(hid_t driver_id, unsigned long *flags )
{
    H5FD_class_t *driver    = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == flags)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "flags parameter cannot be NULL");

    
    if (NULL == (driver = (H5FD_class_t *)H5I_object_verify(driver_id, H5I_VFL)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "not a VFL ID");
    if (H5FD_driver_query(driver, flags) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "driver flag query failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5FDdelete(const char *filename, hid_t fapl_id)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (!filename || !*filename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no file name specified");

    if (H5P_DEFAULT == fapl_id)
        fapl_id = H5P_FILE_ACCESS_DEFAULT;
    else if (true != H5P_isa_class(fapl_id, H5P_FILE_ACCESS))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    if (H5FD_delete(filename, fapl_id) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "unable to delete file");

done:
    FUNC_LEAVE_API(ret_value)
} 
